/**
* @file Modules/ImageProcessor/ImageProcessorTools/MSH2004ColorCorrector.cpp
* 
* This file contains a class that represents a table used for color correction.
*
* @author <A href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</A>
*/

#include "MSH2004ColorCorrector.h"
#include "Tools/Location.h"
#include "Tools/Streams/InStreams.h"
#include <math.h>


MSH2004ColorCorrector::MSH2004ColorCorrector()
{
  bool loaded_y = false, loaded_u = false, loaded_v = false;
  double dummy;
  InBinaryFile stream_y(getLocation().getFilename("coeff.cy"));
  if(stream_y.exists())
  {
    stream_y >>  radialOrder;
    stream_y >>  colorOrder;
    int i;
    if (radialOrder < maxRadialOrder)
    { //source array bigger than file
      for (i = 0; i < radialOrder; i++)
        stream_y >>  radialP[i+0*maxRadialOrder];
      for (;i<maxRadialOrder; i++)
        radialP[i] = 0.0;
    }
    else
    {
      for (i = 0; i < maxRadialOrder; i++)
        stream_y >>  radialP[i+0*maxRadialOrder];
      for (;i<radialOrder; i++) //discard higher order coeff that doesnt fit in the array size
        stream_y >>  dummy;
    }
    if (colorOrder < maxColorOrder)
    { //source array bigger than file
      for (i = 0; i < colorOrder; i++)
        stream_y >>  colorP[i+0*maxColorOrder];
      for (;i<maxColorOrder; i++)
        colorP[i] = 0.0;
    }
    else
    {
      for (i = 0; i < maxColorOrder; i++)
        stream_y >>  colorP[i+0*maxColorOrder];
      for (;i<colorOrder; i++) //discard higher order coeff that doesnt fit in the array size
        stream_y >>  dummy;
    }
  }
  else {
    int i;
    for (i = 0; i < radialOrder; i++)
      radialP[i+0*maxRadialOrder] = 0.0;
    for (i=0; i < colorOrder; i++)
      colorP[i+0*maxColorOrder] = 0.0;
  }
  InBinaryFile stream_u(getLocation().getFilename("coeff.cu"));
  if(stream_u.exists())
  {
    stream_u >>  radialOrder;
    stream_u >>  colorOrder;
    int i;
    if (radialOrder < maxRadialOrder)
    { //source array bigger than file
      for (i = 0; i < radialOrder; i++)
        stream_u >>  radialP[i+1*maxRadialOrder];
      for (;i<maxRadialOrder; i++)
        radialP[i] = 0.0;
    }
    else
    {
      for (i = 0; i < maxRadialOrder; i++)
        stream_u >>  radialP[i+1*maxRadialOrder];
      for (;i<radialOrder; i++) //discard higher order coeff that doesnt fit in the array size
        stream_u >>  dummy;
    }
    if (colorOrder < maxColorOrder)
    { //source array bigger than file
      for (i = 0; i < colorOrder; i++)
        stream_u >>  colorP[i+1*maxColorOrder];
      for (;i<maxColorOrder; i++)
        colorP[i] = 0.0;
    }
    else
    {
      for (i = 0; i < maxColorOrder; i++)
        stream_u >>  colorP[i+1*maxColorOrder];
      for (;i<colorOrder; i++) //discard higher order coeff that doesnt fit in the array size
        stream_u >>  dummy;
    }
  }
  else {
    int i;
    for (i = 0; i < radialOrder; i++)
      radialP[i+1*maxRadialOrder] = 0.0;
    for (i=0; i < colorOrder; i++)
      colorP[i+1*maxColorOrder] = 0.0;
  }
  InBinaryFile stream_v(getLocation().getFilename("coeff.cv"));
  if(stream_v.exists())
  {
    stream_v >>  radialOrder;
    stream_v >>  colorOrder;
    int i;
    if (radialOrder < maxRadialOrder)
    { //source array bigger than file
      for (i = 0; i < radialOrder; i++)
        stream_v >>  radialP[i+2*maxRadialOrder];
      for (;i<maxRadialOrder; i++)
        radialP[i] = 0.0;
    }
    else
    {
      for (i = 0; i < maxRadialOrder; i++)
        stream_v >>  radialP[i+2*maxRadialOrder];
      for (;i<radialOrder; i++) //discard higher order coeff that doesnt fit in the array size
        stream_v >>  dummy;
    }
    if (colorOrder < maxColorOrder)
    { //source array bigger than file
      for (i = 0; i < colorOrder; i++)
        stream_v >>  colorP[i+2*maxColorOrder];
      for (;i<maxColorOrder; i++)
        colorP[i] = 0.0;
    }
    else
    {
      for (i = 0; i < maxColorOrder; i++)
        stream_v >>  colorP[i+2*maxColorOrder];
      for (;i<colorOrder; i++) //discard higher order coeff that doesnt fit in the array size
        stream_v >>  dummy;
    }
  }
  else {
    int i;
    for (i = 0; i < radialOrder; i++)
      radialP[i+2*maxRadialOrder] = 0.0;
    for (i=0; i < colorOrder; i++)
      colorP[i+2*maxColorOrder] = 0.0;
  }
  active = loaded_y||loaded_u||loaded_v;
  //~ if (active)
  //~ {
  setupRadiusLUT();
  setupCorrectionLUT();
  //~ }
}

void MSH2004ColorCorrector::setupRadiusLUT()
{
  double dx, dy, radius;
  for (int y=0; y<cameraResolutionHeight_ERS7; y++)
    for (int x=0; x<cameraResolutionWidth_ERS7; x++)
    {
      dx = x - centralPointX;
      dy = y - centralPointY;
      radius = sqrt(dx*dx + dy*dy);
      radiusLUT[y][x] = (unsigned char)radius;
    }
}

unsigned char MSH2004ColorCorrector::colorDistortionCorrection(const unsigned char radius_i, 
                      const unsigned char color, const unsigned char channel) const
{
  int radOffsetBase = channel*maxRadialOrder;
  int colOffsetBase = channel*maxColorOrder;
  double radius = radius_i;
  double radial_base;
  double color_base;
  double rad_correction = 0;
  double col_correction = 0;
  int i;
  for (i=0; i<radialOrder; i++){
    radial_base = 1;
    for (int k=0; k<i; k++)
      radial_base *= radius;
    rad_correction += radial_base*radialP[radOffsetBase+i];
  }
  for (i=0; i<colorOrder; i++){
    color_base = 1;
    for (int k=0; k<i; k++)
      color_base *= color;
    col_correction += color_base*colorP[colOffsetBase+i];
  }
  double result = color + rad_correction*col_correction;
  if (result < 0.0)
    result = 0.0;
  if (result > 255.0)
    result = 255.0;
  return (unsigned char) result;
}

void MSH2004ColorCorrector::setupCorrectionLUT()
{
  for (int radius=0; radius<max_radius; radius++)
    for (int color=0; color<256; color++)
      for (int channel=0; channel<3; channel++)
        colorCorrectionLUT[radius][color][channel] = colorDistortionCorrection(radius, color, channel);
}

/*
 * Change log :
 * 
 * $Log: MSH2004ColorCorrector.cpp,v $
 * Revision 1.4  2004/04/22 14:28:46  roefer
 * ColorCorrector now supports MSH and DDD color correction
 *
 * Revision 1.3  2004/04/19 21:36:21  nistico
 * Parameter file format more flexible
 *
 * Revision 1.2  2004/03/19 11:04:58  nistico
 * Some corrections and restructuring
 *
 * Revision 1.1  2004/03/01 12:54:09  nistico
 * -Added MSH2004ColorCorrector
 * -Integrated MSH2004ColorCorrector into ColorTable32KImageProcessor (through a hack! :-) this has to be improved...
 *
 *
 */

