/**
* @file Modules/ImageProcessor/ImageProcessorTools/MSH2004ColorCorrector.cpp
* 
* Implementation of MSH2004EdgeDetection.
*
* @author <A href="mailto:sadprofessor@web.de">Bernd Schmidt</A>
*/

#include "MSH2004EdgeDetection.h"
#include "Tools/Location.h"
#include <math.h>

enum{DEFAULT_EDGE_THRESHOLD = 12};

MSH2004EdgeDetection::MSH2004EdgeDetection(RasterImageProcessor& processor)
	:ip(processor),
	susanDetector(processor.edgeDetector),
	threshold(DEFAULT_EDGE_THRESHOLD)
{

}

MSH2004EdgeDetection::~MSH2004EdgeDetection(){
}


bool MSH2004EdgeDetection::scanEast(int& x, int& y){
	if (!insideImage(x,y)) return false;
	int voteBuffer = 0;
	int vote = fastCrossEdgeVote(x,y);
	int x2 = ip.image.cameraInfo.resolutionWidth-1;

	while(x < x2){
		if (vote > threshold || voteBuffer != 0){
			if (voteBuffer < vote) voteBuffer = vote;
			else{
				x--; 
				return true; //edge found (found local maximum)
			}
		}		
		x++;
		vote = fastCrossEdgeVote(x,y);
	}

	if (vote > threshold){
		x--;
		return true; //edgeFound (vote high enough, so found edge on Border)
	}
	else return false; // noEdgeFound
}

bool MSH2004EdgeDetection::scanWest(int& x, int& y){
	if (!insideImage(x,y)) return false;
	int voteBuffer = 0;
	int vote = fastCrossEdgeVote(x,y);

	while(x > 1){
		if (vote > threshold || voteBuffer != 0){
			if (voteBuffer < vote) voteBuffer = vote;
			else{
				x++; 
				return true; //edge found (found local maximum)
			}
		}		
		x--;
		vote = fastCrossEdgeVote(x,y);
	}

	if (vote > threshold){
		x++;
		return true; //edgeFound (vote high enough, so found edge on Border)
	}
	else return false; // noEdgeFound
}
	
bool MSH2004EdgeDetection::scanNorth(int& x, int& y){
	if(!insideImage(x,y)) return false;
	int voteBuffer = 0;
	int vote = fastCrossEdgeVote(x,y);

	while(y > 1){
		if (vote > threshold || voteBuffer != 0){
			if (voteBuffer < vote) voteBuffer = vote;
			else{
				y++; 
				return true; //edge found (found local maximum)
			}
		}		
		y--;
		vote = fastCrossEdgeVote(x,y);
	}

	if (vote > threshold){
		y++;
		return true; //edgeFound (vote high enough, so found edge on Border)
	}
	else return false; // noEdgeFound
}

bool MSH2004EdgeDetection::scanSouth(int& x, int& y){
	if (!insideImage(x,y)) return false;
	int voteBuffer = 0;
	int vote = fastCrossEdgeVote(x,y);
	int y2 = ip.image.cameraInfo.resolutionHeight -2;

	while(y < y2){
		if (vote > threshold || voteBuffer != 0){
			if (voteBuffer < vote) voteBuffer = vote;
			else{
				y--; 
				return true; //edge found (found local maximum)
			}
		}		
		y++;
		vote = fastCrossEdgeVote(x,y);
	}

	if (vote > threshold){
		y--;
		return true; //edgeFound (vote high enough, so found edge on Border)
	}
	else return false; // noEdgeFound
}

bool MSH2004EdgeDetection::ballScanEast(int& x, int& y){
	int voteBuffer = 0;
	int vote = ballEdgeVote(x,y);

	while(vote > -1){
		if (vote > threshold || voteBuffer != 0){
			if (voteBuffer < vote) voteBuffer = vote;
			else{
				x--; 
				return true; //edge found (found local maximum)
			}
		}		
		x++;
		vote = ballEdgeVote(x,y);
	}

	if (vote > threshold){
		x--;
		return true; //edgeFound (vote high enough, so found edge on Border)
	}
	else return false; // noEdgeFound
}

bool MSH2004EdgeDetection::ballScanWest(int& x, int& y){
	int voteBuffer = 0;
	int vote = ballEdgeVote(x,y);

	while(vote > -1){
		if (vote > threshold || voteBuffer != 0){
			if (voteBuffer < vote) voteBuffer = vote;
			else{
				x++; 
				return true; //edge found (found local maximum)
			}
		}		
		x--;
		vote = ballEdgeVote(x,y);
	}

	if (vote > threshold){
		x++;
		return true; //edgeFound (vote high enough, so found edge on Border)
	}
	else return false; // noEdgeFound
}

bool MSH2004EdgeDetection::scan(int& x,int& y,Vector2<double> direction){
	cx = 1;
	cy = 1;

	if (direction.x < 0) cx = -1;
	if (direction.y < 0) cy = -1;

	tempX = x;
	tempY = y;

	int vote = crossEdgeVote(x,y);
	int buffer = 0;
	if (vote > threshold) buffer = vote;

	//doing variation of Bresenham's algorithm for line-drawing, to scan in direction

	dx = abs((int)(direction.x * 500));
	dy = abs((int)(direction.y * 500));

	if (dx > dy){
		e = dx - 2*dy;
		f = 2*(dx - dy);
		g = -2*dy;
		//setPixel(0,0)
		 
		for (x+=cx;vote != -1;x+=cx) {
			if (e<0) {
				y+=cy;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = crossEdgeVote(x,y);
			
			if (vote > threshold || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					x = tempX;
					y = tempY;
					return true; //edge found (found local maximum)
				}
			}

			tempX = x;
			tempY = y;
		}
	}
	else{
		e = dy - 2*dx;
		f = 2*(dy - dx);
		g = -2*dx;
		//setPixel(0,0)

		 
		for (y+=cy;vote != -1;y+=cy) {
			if (e<0) {
				x+=cx;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = crossEdgeVote(x,y);

			if (vote > threshold || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					x = tempX;
					y = tempY;
					return true; //edge found (found local maximum)
				}
			}
			
			tempX = x;
			tempY = y;
		}
	}

	if (buffer != 0){
				x = tempX;
				y = tempY;
				return true; //edge found (found edge on border)
	}

	return false; //no edge found
}

bool MSH2004EdgeDetection::bufferedScan(int& x,int& y,Vector2<double> direction){	
	
	cx = 1;
	cy = 1;

	if (direction.x < 0) cx = -1;
	if (direction.y < 0) cy = -1;

	tempX = x;
	tempY = y;

	int vote = crossEdgeVote(x,y);
	currentColor = lastColor = getColor(x,y);
	bufferSize = 0;

	int buffer = 0;
	if (vote > threshold) buffer = vote;

	//doing variation of Bresenham's algorithm for line-drawing, to scan in direction

	dx = abs((int)(direction.x * 500));
	dy = abs((int)(direction.y * 500));

	if (dx > dy){
		e = dx - 2*dy;
		f = 2*(dx - dy);
		g = -2*dy;
		//setPixel(0,0)
		 
		for (x+=cx;vote != -1;x+=cx) {
			if (e<0) {
				y+=cy;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = crossEdgeVote(x,y);
			currentColor = getColor(x,y);
			colorRange++;
			
			if (currentColor != lastColor){
				addColor();
				// there are to many colorChanges between two edge-points
				if (bufferSize > 19) return false;
			}	

			if (vote > threshold || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					x = tempX;
					y = tempY;
					if (colorRange!=0){
						lastColor = currentColor;
						addColor();
					}
					return true; //edge found (found local maximum)
				}
			}

			tempX = x;
			tempY = y;
			lastColor = currentColor;
		}
	}
	else{
		e = dy - 2*dx;
		f = 2*(dy - dx);
		g = -2*dx;
		//setPixel(0,0)

		 
		for (y+=cy;vote != -1;y+=cy) {
			if (e<0) {
				x+=cx;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = crossEdgeVote(x,y);
			currentColor = getColor(x,y);
			colorRange++;
			
			if (currentColor != lastColor){
				addColor();
				// there are to many colorChanges between two edge-points
				if (bufferSize > 19) return false;
			}	

			if (vote > threshold || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					if (colorRange!=0){
						lastColor = currentColor;
						addColor();
					}
					x = tempX;
					y = tempY;
					return true; //edge found (found local maximum)
				}
			}
			
			tempX = x;
			tempY = y;
			lastColor = currentColor;
		}
	}

	if (buffer != 0){
				if (colorRange!=0){
					lastColor = currentColor;
					addColor();
				}
				x = tempX;
				y = tempY;
				return true; //edge found (found edge on border)
	}

	return false; //no edge found
}

bool MSH2004EdgeDetection::bufferedScan(int& x,int& y){	
	
	currentColor = lastColor = getColor(x,y);
	colorRange = 0;
	bufferSize = 0;

	//skipping 3 pixels, not to detect 1 edge 2 times
	for (int i = 0;i<3;i++){
		skip(x,y);
		currentColor = getColor(x,y);
		colorRange++;
		if (currentColor != lastColor)addColor();
		lastColor = currentColor;
	}
	
	tempX = x;
	tempY = y;

	int vote = crossEdgeVote(x,y);

	int buffer = 0;
	if (vote > threshold) buffer = vote;

	//doing variation of Bresenham's algorithm for line-drawing, to scan in direction

	if (dx > dy){		 
		for (x+=cx;vote != -1;x+=cx) {
			if (e<0) {
				y+=cy;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = crossEdgeVote(x,y);
			currentColor = getColor(x,y);
			colorRange++;
			
			if (currentColor != lastColor){
				addColor();
				// there are to many colorChanges between two edge-points
				if (bufferSize > 19) return false; 
			}	

			if (vote > threshold || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					x = tempX;
					y = tempY;
					if (colorRange!=0){
						lastColor = currentColor;
						addColor();
					}
					return true; //edge found (found local maximum)
				}
			}

			tempX = x;
			tempY = y;
			lastColor = currentColor;
		}
	}
	else{
		for (y+=cy;vote != -1;y+=cy) {
			if (e<0) {
				x+=cx;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = crossEdgeVote(x,y);
			currentColor = getColor(x,y);
			colorRange++;
			
			if (currentColor != lastColor){
				addColor();
			}	

			if (vote > threshold || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					if (colorRange!=0){
						lastColor = currentColor;
						addColor();
					}
					x = tempX;
					y = tempY;
					return true; //edge found (found local maximum)
				}
			}
			
			tempX = x;
			tempY = y;
			lastColor = currentColor;
		}
	}

	if (buffer != 0){
				if (colorRange!=0){
					addColor();
				}
				x = tempX;
				y = tempY;
				return true; //edge found (found edge on border)
	}

	return false; //no edge found
}

bool MSH2004EdgeDetection::scanField(int& x,int& y,Vector2<double> direction){
	cx = 1;
	cy = 1;

	if (direction.x < 0) cx = -1;
	if (direction.y < 0) cy = -1;

	tempX = x;
	tempY = y;

	int vote = fieldEdgeVote(x,y);
	int buffer = 0;
	if (vote > threshold) buffer = vote;

	//doing variation of Bresenham's algorithm for line-drawing, to scan in direction

	dx = abs((int)(direction.x * 500));
	dy = abs((int)(direction.y * 500));

	if (dx > dy){
		e = dx - 2*dy;
		f = 2*(dx - dy);
		g = -2*dy;
		//setPixel(0,0)
		 
		for (x+=cx;vote != -1;x+=cx) {
			if (e<0) {
				y+=cy;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = fieldEdgeVote(x,y);
			
			if (vote > threshold || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					x = tempX;
					y = tempY;
					return true; //edge found (found local maximum)
				}
			}

			tempX = x;
			tempY = y;
		}
	}
	else{
		e = dy - 2*dx;
		f = 2*(dy - dx);
		g = -2*dx;
		//setPixel(0,0)

		 
		for (y+=cy;vote != -1;y+=cy) {
			if (e<0) {
				x+=cx;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = fieldEdgeVote(x,y);

			if (vote > threshold || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					x = tempX;
					y = tempY;
					return true; //edge found (found local maximum)
				}
			}
			
			tempX = x;
			tempY = y;
		}
	}

	if (buffer != 0){
		x = tempX;
		y = tempY;
		return true; //edge found (found edge on border)
	}

	return false; //no edge found
}

bool MSH2004EdgeDetection::scanField(int& x,int& y){

	tempX = x;
	tempY = y;

	int vote = fieldEdgeVote(x,y);
	int buffer = 0;
	if (vote > threshold) buffer = vote;

	//doing variation of Bresenham's algorithm for line-drawing, to scan in direction
	if (dx > dy){		 
		for (x+=cx;vote != -1;x+=cx) {
			if (e<0) {
				y+=cy;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = fieldEdgeVote(x,y);
			
			if (vote > threshold || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					x = tempX;
					y = tempY;
					return true; //edge found (found local maximum)
				}
			}

			tempX = x;
			tempY = y;
		}
	}
	else{
		for (y+=cy;vote != -1;y+=cy) {
			if (e<0) {
				x+=cx;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = fieldEdgeVote(x,y);

			if (vote > threshold || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					x = tempX;
					y = tempY;
					return true; //edge found (found local maximum)
				}
			}			
			tempX = x;
			tempY = y;
		}
	}

	if (buffer != 0){
				x = tempX;
				y = tempY;
				return true; //edge found (found edge on border)
	}

	return false; //no edge found
}

bool MSH2004EdgeDetection::scan(int& x,int& y,int x1,int y1){
	cx = 1;
	cy = 1;

	dx = x1 - x;
	dy = y1 - y;

	if (dx < 0) cx = -1;
	if (dy < 0) cy = -1;

	tempX = x;
	tempY = y;

	int vote = crossEdgeVote(x,y);
	int buffer = 0;
	if (vote > threshold) buffer = vote;

	//doing variation of Bresenham's algorithm for line-drawing, to scan in direction
	if (dx > dy){
		e = dx - 2*dy;
		f = 2*(dx - dy);
		g = -2*dy;
		//setPixel(0,0)
		 
		for (x+=cx;x<x1;x+=cx) {
			if (e<0) {
				y+=cy;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = fastCrossEdgeVote(x,y);
			
			if (vote > threshold || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					x = tempX;
					y = tempY;
					return true; //edge found (found local maximum)
				}
			}

			tempX = x;
			tempY = y;
		}
	}
	else{
		e = dy - 2*dx;
		f = 2*(dy - dx);
		g = -2*dx;
		//setPixel(0,0)

		 
		for (y+=cy;y<y1;y+=cy) {
			if (e<0) {
				x+=cx;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = fastCrossEdgeVote(x,y);

			if (vote > threshold || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					x = tempX;
					y = tempY;
					return true; //edge found (found local maximum)
				}
			}
			
			tempX = x;
			tempY = y;
		}
	}

	if (buffer != 0){
				x = tempX;
				y = tempY;
				return true; //edge found (found edge on border)
	}

	return false; //no edge found
}

bool MSH2004EdgeDetection::colorScan(int& x,int& y,int x1,int y1,colorClass& lastColor){
	cx = 1;
	cy = 1;

	colorClass start = getColor(x,y);
	colorClass current = start;
	
	dx = x1 - x;
	dy = y1 - y;

	if (dx < 0) cx = -1;
	if (dy < 0) cy = -1;

	//doing variation of Bresenham's algorithm for line-drawing, to scan in direction
	if (dx > dy){
		e = dx - 2*dy;
		f = 2*(dx - dy);
		g = -2*dy;
		//TODO(schmidtb): remove inefficient code
		int count = 0; //not to write more code and hold the order
		//setPixel(0,0)
		 
		for (x+=cx ; current == start && count < dx ; x+=cx,count++) {
			if (e<0) {
				y+=cy;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			current = getColor(x,y);
		}
	}
	else{
		e = dy - 2*dx;
		f = 2*(dy - dx);
		g = -2*dx;
		int count = 0;
		//setPixel(0,0)
		
		 
		for (y+=cy; current==start && count<dy; y+=cy,count++) {
			if (e<0) {
				x+=cx;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			current = getColor(x,y);
		}
	}

	if (current == -1){ 
		lastColor = noColor;
		return false;
	}
	else {
		lastColor = start;
		return true;
	}
}

bool MSH2004EdgeDetection::colorScan(int& x,int& y,Vector2<double> direction,colorClass& lastColor){
	cx = 1;
	cy = 1;

	colorClass start = getColor(x,y);
	colorClass current = start;

	if (direction.x < 0) cx = -1;
	if (direction.y < 0) cy = -1;


	//doing variation of Bresenham's algorithm for line-drawing, to scan in direction

	dx = abs((int)(direction.x * 500));
	dy = abs((int)(direction.y * 500));

	if (dx > dy){
		e = dx - 2*dy;
		f = 2*(dx - dy);
		g = -2*dy;
		//setPixel(0,0)
		 
		for (x+=cx;current == start && current != -1;x+=cx) {
			if (e<0) {
				y+=cy;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			current = getColor(x,y);
		}
	}
	else{
		e = dy - 2*dx;
		f = 2*(dy - dx);
		g = -2*dx;
		//setPixel(0,0)

		 
		for (y+=cy;current == start && current != -1;y+=cy) {
			if (e<0) {
				x+=cx;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			current = getColor(x,y);
		}
	}

	if (current == -1){ 
		lastColor = noColor;
		return false;
	}
	else {
		lastColor = start;
		return true;
	}
}

bool MSH2004EdgeDetection::colorScan(int& x,int& y,colorClass& lastColor){

	colorClass start = getColor(x,y);
	colorClass current = start;

	//doing variation of Bresenham's algorithm for line-drawing, to scan in direction
	if (dx > dy){		 
		for (x+=cx;current == start && current != -1;x+=cx) {
			if (e<0) {
				y+=cy;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			current = getColor(x,y);
		}
	}
	else{		 
		for (y+=cy;current == start && current != -1;y+=cy) {
			if (e<0) {
				x+=cx;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			current = getColor(x,y);
		}
	}

	if (current == -1){ 
		lastColor = noColor;
		return false;
	}
	else {
		lastColor = start;
		return true;
	}
}

bool MSH2004EdgeDetection::susanScan(int& x,int& y,Vector2<double> direction){
	cx = 1;
	cy = 1;

	if (direction.x < 0) cx = -1;
	if (direction.y < 0) cy = -1;

	tempX = x;
	tempY = y;

	int vote = susanVote(x,y);
	int buffer = 0;
	if (vote > 0) buffer = vote;

	//doing variation of Bresenham's algorithm for line-drawing, to scan in direction

	dx = abs((int)(direction.x * 500));
	dy = abs((int)(direction.y * 500));

	if (dx > dy){
		e = dx - 2*dy;
		f = 2*(dx - dy);
		g = -2*dy;
		//setPixel(0,0)
		 
		for (x+=cx;vote != -1;x+=cx) {
			if (e<0) {
				y+=cy;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = susanVote(x,y);
			
			if (vote > 0 || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					x = tempX;
					y = tempY;
					return true; //edge found (found local maximum)
				}
			}

			tempX = x;
			tempY = y;
		}
	}
	else{
		e = dy - 2*dx;
		f = 2*(dy - dx);
		g = -2*dx;
		//setPixel(0,0)

		 
		for (y+=cy;vote != -1;y+=cy) {
			if (e<0) {
				x+=cx;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = susanVote(x,y);

			if (vote > 0 || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					x = tempX;
					y = tempY;
					return true; //edge found (found local maximum)
				}
			}
			
			tempX = x;
			tempY = y;
		}
	}

	if (buffer != 0){
		x = tempX;
		y = tempY;
		return true; //edge found (found edge on border)
	}

	return false; //no edge found
}

bool MSH2004EdgeDetection::scan(int& x,int& y){

	tempX = x;
	tempY = y;

	int vote = crossEdgeVote(x,y);
	int buffer = 0;
	if (vote > threshold) buffer = vote;

	//doing variation of Bresenham's algorithm for line-drawing, to scan in direction
	if (dx > dy){		 
		for (x+=cx;vote != -1;x+=cx) {
			if (e<0) {
				y+=cy;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = crossEdgeVote(x,y);
			
			if (vote > threshold || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					x = tempX;
					y = tempY;
					return true; //edge found (found local maximum)
				}
			}

			tempX = x;
			tempY = y;
		}
	}
	else{
		for (y+=cy;vote != -1;y+=cy) {
			if (e<0) {
				x+=cx;
				e+=f;
			}
			else {
				e+=g;
			}
			//setPixel(x,y)
			//DOT(imageProcessor_general,x,y,
			//	Drawings::white, Drawings::red);
			vote = crossEdgeVote(x,y);

			if (vote > threshold || buffer != 0){
				if (buffer < vote) buffer = vote;
				else{
					x = tempX;
					y = tempY;
					return true; //edge found (found local maximum)
				}
			}
			
			tempX = x;
			tempY = y;
		}
	}

	if (buffer != 0){
				x = tempX;
				y = tempY;
				return true; //edge found (found edge on border)
	}

	return false; //no edge found
}


/*
 * Change log :
 *
 * $Log: MSH2004EdgeDetection.cpp,v $
 * Revision 1.4  2004/05/27 11:18:48  schmidtb
 * new version with further work for open challenge
 *
 * Revision 1.3  2004/04/08 15:33:06  wachter
 * GT04 checkin of Microsoft-Hellounds
 *
 * Revision 1.3  2004/03/25 15:10:03  pg_besc
 * made some changes
 *
 * Revision 1.2  2004/03/17 20:58:47  schmidtb
 * added new scan methods and buffering.
 *
 * Revision 1.1  2004/03/11 20:19:43  schmidtb
 * Established MSH2004EdgeDetection.
 *
 */
