/**
* @file ManualCalibration.cpp
*
* Implementation of class ManualCalibration
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#include "ManualCalibration.h"
#include "Platform/SystemCall.h"

ManualCalibration::ManualCalibration
(
 const Image& image, 
 const CameraMatrix& cameraMatrix,
 ColorTable64& colorTable,
 const CalibrationRequest& calibrationRequest
 )
 :
image(image),
cameraMatrix(cameraMatrix),
colorTable(colorTable),
calibrationRequest(calibrationRequest)
{
  timeOfLastGreenCalibration = 0;
  timeOfLastOrangeCalibration = 0;
}

ManualCalibration::~ManualCalibration()
{
}

void ManualCalibration::execute()
{
  switch(calibrationRequest.mode)
  {
  case CalibrationRequest::none:
    break;
  case CalibrationRequest::calibrateColors:
    if(calibrationRequest.feedback == CalibrationRequest::colorIsGreen)
      calibrateGreen();
    if(calibrationRequest.feedback == CalibrationRequest::colorIsOrange)
      calibrateOrange();
    break;
  default:
    break;
  }
  INFO(sendColorTable64, idColorTable64, bin, colorTable);
}

void ManualCalibration::calibrateGreen()
{
  if(SystemCall::getTimeSince(timeOfLastGreenCalibration) > 2000) 
    colorTable.clearChannel(green);
  timeOfLastGreenCalibration = SystemCall::getCurrentSystemTime();
  for(int x = 0; x < image.cameraInfo.resolutionWidth; x++)
  {
    for(int y = 0; y < image.cameraInfo.resolutionHeight; y++)
    {
      colorTable.addColorClass(
        green, 
        image.image[y][0][x], image.image[y][1][x], image.image[y][2][x], 5);
    }
  }
}

void ManualCalibration::calibrateOrange()
{
  if(SystemCall::getTimeSince(timeOfLastOrangeCalibration) > 2000) 
    colorTable.clearChannel(orange);
  timeOfLastOrangeCalibration = SystemCall::getCurrentSystemTime();
  for(int x = 0; x < image.cameraInfo.resolutionWidth; x++)
  {
    for(int y = 0; y < image.cameraInfo.resolutionHeight; y++)
    {
      colorClass currentColor = colorTable.getColorClass(image.image[y][0][x], image.image[y][1][x], image.image[y][2][x]);
      if(currentColor != green && currentColor != orange)
      {
        colorTable.addColorClass(
          orange, 
          image.image[y][0][x], image.image[y][1][x], image.image[y][2][x], 2);
      }
    }
  }
}

/*
to do
camera parameter kalibrieren
farbklassenbild erzeugen
mit histogrammen arbeiten
*/

/*
* Change log :
* 
* $Log: ManualCalibration.cpp,v $
* Revision 1.7  2004/03/21 12:46:28  juengel
* CalibrationRequest distinguishes between "mode" and "feedback".
*
* Revision 1.6  2004/02/12 14:15:21  juengel
* changed visualization of segmented image 1
*
* Revision 1.5  2003/12/15 11:49:06  juengel
* Introduced CameraInfo
*
* Revision 1.4  2003/11/05 16:42:17  juengel
* comments removed
*
* Revision 1.3  2003/11/03 20:16:09  juengel
* color class images can be sent from robot now
*
* Revision 1.2  2003/10/12 20:17:20  juengel
* only c64 is supported
*
* Revision 1.1  2003/10/12 11:44:58  juengel
* Added CalibrationRequest.
*
*
*/
