
/** 
* @file SUSANEdgeDetectionLite.h
* Declaration of file SUSANEdgeDetectionLite.
*
* @author <A href=mailto:walter.nistico@uni-dortmund.de>Walter Nistico</A>
*/

#ifndef _SUSANEdgeDetectionLite_h_
#define _SUSANEdgeDetectionLite_h_

#include "Representations/Perception/Image.h"
#include <math.h>

/**
* @class SUSANEdgeDetectionLite
*
* This class represents a non-linear image edge detection filter.
* Unlike traditional edge detectors gradient based like Sobel or LoG, this filter doesnt perform a derivative operation on the image, 
* (which is an hi-pass characteristic) thus has a better noise-rejection performance.
* This implementation is dubbed "Lite" as the heaviest passes of the full algorithm (edge direction calculation, non-maxima suppression, binary thinning)
* have been stripped out for performance reasons.
*
* @author <A href=mailto:walter.nistico@uni-dortmund.de>Walter Nistico</A>
*/
class SUSANEdgeDetectionLite 
{
    
public:
  
  enum ColorSpectra {componentA=0, componentB, componentC};
  enum {GEOMETRIC_THRESHOLD=700};

  /** Constructor */
  SUSANEdgeDetectionLite(int edgeThreshold);

  /** Destructor */
  ~SUSANEdgeDetectionLite();

  /** 
  * Checks if a given point is an edge point.
  * @param image The source image
  * @param X The X coordinate of the given point
  * @param Y The Y coordinate of the given point
  * @param channel The chosen color spectrum (Y, U, or V for ex.)
  * @return is the point an edge point?
  */
  bool isEdgePoint(const Image& image, int X, int Y, ColorSpectra channel) const;
 
private:

  /**
  * A LookUpTable containing a correlation function
  */
  unsigned char Susan_LUT[255];

  /**
  * Initializes the LookUpTable
  */
  void setupSusanLUT(int threshold);

  /**
  * The correlation function, precomputed
  */
	inline unsigned char correlation(int delta) const
  {
		return Susan_LUT[(delta>>1)+127];
	}

};

#endif   //  _SUSANEdgeDetectionLite_h_

/*
* Change log :
* 
* $Log: SUSANEdgeDetectionLite.h,v $
* Revision 1.3  2004/03/19 11:04:58  nistico
* Some corrections and restructuring
*
* Revision 1.2  2004/03/11 21:00:39  schmidtb
* removed errors with enums and destructor
*
* Revision 1.1  2004/03/03 13:43:52  nistico
* -Added SUSANEdgeDetectionLite
*
*
*
*/
