/**
* @file BallDetector.cpp
*
* Implementation of class BallDetector
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#include "BallDetector.h"
#include "../ColorTableReferenceColor.h"
#include "../ColorSpaceUsageCounter.h"
#include "LineSegmentation.h"
#include "Tools/FieldDimensions.h"
#include "Tools/Math/Geometry.h"

#include <limits.h>

BallDetector::BallDetector
(
 const Image& image, 
 const CameraMatrix& cameraMatrix,
 const CameraInfo& cameraInfo,
 ColorTable64& colorTable64,
 ColorTableCuboids& colorTableCuboids,
 ColorTableReferenceColor& colorTableReferenceColor,
 BallPercept& ballPercept
 )
 :
image(image),
cameraMatrix(cameraMatrix),
cameraInfo(cameraInfo),
colorTable64(colorTable64), 
colorTableCuboids(colorTableCuboids), 
colorTableReferenceColor(colorTableReferenceColor), 
ballPercept(ballPercept)
{
  borderPointBoundary = 0;
}

BallDetector::~BallDetector()
{
  if(borderPointBoundary != 0)
  {
    delete borderPointBoundary;
  }
}

void BallDetector::init()
{
  circleCalculation.init();
  for(int i = 0; i < maxNumberOfScanLines + 2; i++)
  {
    numberOfOrangeSegments[i] = 0;
    foundSegmentWithNoColorBelowOrange[i] = false;
    foundGreenBelowOrange[i] = false;
  }
  lineIndexOfLastSegment = -1;
  foundBorderPoints = false;
  if(borderPointBoundary != 0)
  {
    delete borderPointBoundary;
  }
  borderPointBoundary = new Boundary<int>(0, INT_MAX);
}

void BallDetector::addSegment(LineSegment& newSegment, int lineIndex)
{
  colorClass colorOfSegment = colorTable64.getColorClass(newSegment.averageIntensity[0], newSegment.averageIntensity[1], newSegment.averageIntensity[2]);

  // initialize if a new line starts
  if(lineIndexOfLastSegment != lineIndex) {}  
  lineIndexOfLastSegment = lineIndex;

  if(
    colorOfSegment == noColor && 
    !foundGreenBelowOrange[lineIndex + 1] 
    && newSegment.length > 4
    )
  {
    foundSegmentWithNoColorBelowOrange[lineIndex + 1] = true;
  }

  if(colorOfSegment == green && numberOfOrangeSegments[lineIndex + 1])
  {
    foundGreenBelowOrange[lineIndex + 1] = true;
  }

  if(colorOfSegment == orange &&
    newSegment.endIsBelowHorizon ) 
  {
    addOrangeC64Segment(newSegment, lineIndex);
    foundSegmentWithNoColorBelowOrange[lineIndex + 1] = false;
  }
}

void BallDetector::addOrangeC64Segment(LineSegment& newSegment, int lineIndex)
{
  //If there is a previous segment on that line ...
  //... and its end is close to the begin of the new one ...
  if(
    numberOfOrangeSegments[lineIndex + 1] != 0 &&
    Geometry::distance(orangeSegments[lineIndex + 1][numberOfOrangeSegments[lineIndex + 1]-1].end, newSegment.begin ) < 12 )
  {
    //... combine them.
    orangeSegments[lineIndex + 1][numberOfOrangeSegments[lineIndex + 1]-1].end = newSegment.end;
    orangeSegments[lineIndex + 1][numberOfOrangeSegments[lineIndex + 1]-1].endIsOnBorder = newSegment.endIsOnBorder;
    orangeSegments[lineIndex + 1][numberOfOrangeSegments[lineIndex + 1]-1].endIsBelowHorizon = newSegment.endIsBelowHorizon;
    
    orangeSegments[lineIndex + 1][numberOfOrangeSegments[lineIndex + 1]-1].averageIntensity[0] += newSegment.averageIntensity[0];
    orangeSegments[lineIndex + 1][numberOfOrangeSegments[lineIndex + 1]-1].averageIntensity[0] /= 2;
    orangeSegments[lineIndex + 1][numberOfOrangeSegments[lineIndex + 1]-1].averageIntensity[1] += newSegment.averageIntensity[1];
    orangeSegments[lineIndex + 1][numberOfOrangeSegments[lineIndex + 1]-1].averageIntensity[1] /= 2;
    orangeSegments[lineIndex + 1][numberOfOrangeSegments[lineIndex + 1]-1].averageIntensity[2] += newSegment.averageIntensity[2];
    orangeSegments[lineIndex + 1][numberOfOrangeSegments[lineIndex + 1]-1].averageIntensity[2] /= 2;
    
    orangeSegments[lineIndex + 1][numberOfOrangeSegments[lineIndex + 1]-1].length = (int)Geometry::distance(
      orangeSegments[lineIndex + 1][numberOfOrangeSegments[lineIndex + 1]-1].begin,
      orangeSegments[lineIndex + 1][numberOfOrangeSegments[lineIndex + 1]-1].end );
  }
  else
  {
    numberOfOrangeSegments[lineIndex + 1] = 0;
    orangeSegments[lineIndex + 1][numberOfOrangeSegments[lineIndex + 1]] = newSegment;
    numberOfOrangeSegments[lineIndex + 1]++;
  }
}

void BallDetector::scanBallHorizontal()
{
  Boundary<int> ballBoundary;
  if(circleCalculation.getBoundary(ballBoundary, 0))
  {
    RECTANGLE(imageProcessor_ball2,
      ballBoundary.x.min, ballBoundary.y.min,
      ballBoundary.x.max, ballBoundary.y.max,
      4, Drawings::ps_solid, Drawings::gray);

    if(foundBorderPoints)
    {
      ballBoundary.add(*borderPointBoundary);
    }
    RECTANGLE(imageProcessor_ball2,
      ballBoundary.x.min, ballBoundary.y.min,
      ballBoundary.x.max, ballBoundary.y.max,
      1, Drawings::ps_solid, Drawings::white);

    int length, spacing;
    length = max(2 * (ballBoundary.x.getSize()), 2 * (ballBoundary.y.getSize()));
    length = max(50, length);
    spacing = max((ballBoundary.x.getSize()) / 3, 5);
    
    Vector2<double> begin[3];
    Vector2<double> end[3];
    Vector2<double> center(
      ballBoundary.x.getCenter(),
      ballBoundary.y.getCenter());

    Vector2<int> beginInt[3];
    Vector2<int> endInt[3];

    Geometry::Line horizonLine = Geometry::calculateHorizon(cameraMatrix, cameraInfo);
    Geometry::Line verticalLine;

    verticalLine.base = horizonLine.base;
    verticalLine.direction.x = - horizonLine.direction.y;
    verticalLine.direction.y = horizonLine.direction.x;

    for(int i = 0; i < 3; i++)
    {
      begin[i] = center + horizonLine.direction * -length / 2 + verticalLine.direction * spacing * (i - 1);
      end[i] = center + horizonLine.direction * length / 2 + verticalLine.direction * spacing * (i - 1);
      
      beginInt[i].x = (int)begin[i].x;
      beginInt[i].y = (int)begin[i].y;
      endInt[i].x = (int)end[i].x;
      endInt[i].y = (int)end[i].y;
      
      LINE(imageProcessor_ball2,
        beginInt[i].x, beginInt[i].y, endInt[i].x, endInt[i].y,
        2, Drawings::ps_solid, Drawings::white);

     ColorSpaceUsageCounter colorSpaceUsageCounter;

//      if(
        Geometry::clipLineWithRectangleCohenSutherland(
        Vector2<int>(0,0), Vector2<int>(image.cameraInfo.resolutionWidth - 1, image.cameraInfo.resolutionHeight - 3),
        beginInt[i], endInt[i]);
//        )
      {
        LineSegmentation lineSegmentation(
          Vector2<int>((int)beginInt[i].x, (int)beginInt[i].y), 
          Vector2<int>((int)endInt[i].x, (int)endInt[i].y), image, 
          25, 12, 18, colorSpaceUsageCounter);
        colorClass colorOfSegment[maxNumberOfSegmentsPerScanLine];
        Vector2<int> endOfLastSegment;
        for(int segmentIndex = 0; segmentIndex < lineSegmentation.getNumberOfSegments(); segmentIndex++)
        {
          LineSegment& currentSegment = lineSegmentation.getSegment(segmentIndex);
          colorOfSegment[segmentIndex] = colorTable64.getColorClass(currentSegment.averageIntensity[0], currentSegment.averageIntensity[1], currentSegment.averageIntensity[2]);
          if(segmentIndex != 0)
          {
            if(colorOfSegment[segmentIndex] == orange && 
               colorOfSegment[segmentIndex - 1] == green)
            {
              circleCalculation.addBallPoint(0, currentSegment.begin.x, currentSegment.begin.y, false);
            }
            if(colorOfSegment[segmentIndex - 1] == orange && 
               colorOfSegment[segmentIndex] == green)
            {
              circleCalculation.addBallPoint(0, endOfLastSegment.x, endOfLastSegment.y, true);
            }
          }
          endOfLastSegment = currentSegment.end;
//          drawSegment(currentSegment);
        }//for(int segmentIndex = 0; segmentIndex < lineSegmentation.getNumberOfSegments(); segmentIndex++)
      }// if(Geometry::clipLineWithRectangleCohenSutherland(
    }// for(int i = 0; i < 3; i++)
  }//  if(circleCalculation.getBoundary(ballBoundingBoxTopLeft, ballBoundingBoxBottomRight ,0))
}


void BallDetector::createBallPercept()
{
  determineBallPoints();

  if(borderPointBoundary->x.max >= borderPointBoundary->x.min)
  {
    RECTANGLE(imageProcessor_ball2,
      borderPointBoundary->x.min, borderPointBoundary->y.min,
      borderPointBoundary->x.max, borderPointBoundary->y.max,
      4, Drawings::ps_solid, Drawings::black);
  }

  Geometry::Circle circle;
  
  bool ballPerceptAdded = false;

  int nextBallPoint = circleCalculation.paintBallPoints(0, 0, 1, Drawings::gray, 3);
  scanBallHorizontal();
  circleCalculation.paintBallPoints(0, nextBallPoint, 2, Drawings::white, 3);
  circleCalculation.paintBallPoints(0, nextBallPoint, 3, Drawings::white, 3);

  int set = 0;
  while(!ballPerceptAdded && set < 2)
  {
    if(circleCalculation.createCircle(set, circle) )
    {
      if(!areForbiddenColorsInsideBall(circle)) 
      {
        addBallPercept((int)circle.center.x, (int)circle.center.y, circle.radius);
        ballPerceptAdded = true;
      }
    }
    set++;
  }

  DEBUG_DRAWING_FINISHED(imageProcessor_ball1);
  DEBUG_DRAWING_FINISHED(imageProcessor_ball2);
  DEBUG_DRAWING_FINISHED(imageProcessor_ball3);
  DEBUG_DRAWING_FINISHED(imageProcessor_ball4);
}

void BallDetector::determineBallPoints()
{
  int accumulatedLengthOfBallSegments = 0;
  int numberOfBallSegments = 0;

  int lineIndex;
  // remove all segments with wrong size
  for(lineIndex = 1; lineIndex < maxNumberOfScanLines + 1; lineIndex++)
  {
    for(int segmentIndex = 0; segmentIndex < numberOfOrangeSegments[lineIndex]; segmentIndex++)
    {
      //calculate the distance to the bottom end of the segment
      Vector2<int> pointOnField;
      if(Geometry::calculatePointOnField(orangeSegments[lineIndex][segmentIndex].end.x, orangeSegments[lineIndex][segmentIndex].end.y, cameraMatrix, cameraInfo, pointOnField) )
      {
        // calculate the size that an ball in that distance has in the image
        //~ int expectedSizeOfBall = Geometry::getSizeByDistance(2 * ballRadius, pointOnField.abs(), cameraInfo.resolutionWidth, cameraInfo.openingAngleWidth);
        int expectedSizeOfBall = Geometry::getSizeByDistance(cameraInfo, 2 * ballRadius, pointOnField.abs());
        
        // filter the segment if its size differs to much
        if(
          //segment is to large
          orangeSegments[lineIndex][segmentIndex].length > expectedSizeOfBall * 1.5 + 4 ||
          //segent is to small and not touching the image border
           (orangeSegments[lineIndex][segmentIndex].length < expectedSizeOfBall / 5 &&
           !orangeSegments[lineIndex][segmentIndex].beginIsOnBorder &&
           !orangeSegments[lineIndex][segmentIndex].endIsOnBorder
           )
           )
        {
          LINE(imageProcessor_ball4, orangeSegments[lineIndex][segmentIndex].begin.x, orangeSegments[lineIndex][segmentIndex].begin.y, orangeSegments[lineIndex][segmentIndex].end.x, orangeSegments[lineIndex][segmentIndex].end.y,
            1, Drawings::ps_solid, Drawings::red );
          //remove the current segment
          if(segmentIndex < numberOfOrangeSegments[lineIndex] - 1)
          {
            orangeSegments[lineIndex][segmentIndex] = orangeSegments[lineIndex][numberOfOrangeSegments[lineIndex] - 1];
            segmentIndex--;
          }
          numberOfOrangeSegments[lineIndex] -= 1;
        }
        else
        {
          accumulatedLengthOfBallSegments += orangeSegments[lineIndex][segmentIndex].length;
          numberOfBallSegments++;
        }
      }// if(Geometry::calculatePointOnField(...
    }// for(int segmentIndex = 0; segmentIndex < numberOfSegments[lineIndex]; segmentIndex++)
  }// for(int lineIndex = 1; lineIndex < maxNumberOfScanLines + 1; lineIndex++)

  int averageLengthOfBallSegments = 0;
  if(numberOfBallSegments > 0) averageLengthOfBallSegments = accumulatedLengthOfBallSegments / numberOfBallSegments;

  for(lineIndex = 1; lineIndex < maxNumberOfScanLines + 1; lineIndex++)
  {
    for(int segmentIndex = 0; segmentIndex < numberOfOrangeSegments[lineIndex]; segmentIndex++)
    {
      // no orange segment of ball size was found next left or right
/*      if(numberOfOrangeSegments[lineIndex - 1] == 0 && 
        numberOfOrangeSegments[lineIndex + 1] == 0)
      {
        LINE(imageProcessor_ball4, 
          orangeSegments[lineIndex][segmentIndex].begin.x, orangeSegments[lineIndex][segmentIndex].begin.y,
          orangeSegments[lineIndex][segmentIndex].end.x, orangeSegments[lineIndex][segmentIndex].end.y,
          1, Drawings::ps_solid, Drawings::gray);
      }
      else*/
      {
        // the segment is not to small compared to the other segments and is not yellow
        if(orangeSegments[lineIndex][segmentIndex].length > averageLengthOfBallSegments / 2 &&
           orangeSegments[lineIndex][segmentIndex].length > 3
           &&
           colorTable64.getColorClass(
           orangeSegments[lineIndex][segmentIndex].averageIntensity[0],
           orangeSegments[lineIndex][segmentIndex].averageIntensity[1],
           orangeSegments[lineIndex][segmentIndex].averageIntensity[2]
           ) != yellow
          )
        {
          LINE(imageProcessor_ball1, 
            orangeSegments[lineIndex][segmentIndex].begin.x, orangeSegments[lineIndex][segmentIndex].begin.y,
            orangeSegments[lineIndex][segmentIndex].end.x, orangeSegments[lineIndex][segmentIndex].end.y,
            1, Drawings::ps_solid, Drawings::white);

          if(orangeSegments[lineIndex][segmentIndex].beginIsOnBorder)
          {
            borderPointBoundary->add(orangeSegments[lineIndex][segmentIndex].begin);
            foundBorderPoints = true;
            CIRCLE(imageProcessor_ball1, 
              orangeSegments[lineIndex][segmentIndex].begin.x, 
              orangeSegments[lineIndex][segmentIndex].begin.y, 
              3,2,
              Drawings::ps_solid, Drawings::black);
          }
          else
          {
            circleCalculation.addBallPoint(0,
              orangeSegments[lineIndex][segmentIndex].begin.x,
              orangeSegments[lineIndex][segmentIndex].begin.y,
              false);
          }
          if(orangeSegments[lineIndex][segmentIndex].endIsOnBorder)
          {
            borderPointBoundary->add(orangeSegments[lineIndex][segmentIndex].end);
            foundBorderPoints = true;
            DOT(imageProcessor_ball1, 
              orangeSegments[lineIndex][segmentIndex].end.x, 
              orangeSegments[lineIndex][segmentIndex].end.y, 
              Drawings::black, Drawings::white);
          }
          else
          {
            if(
              !foundSegmentWithNoColorBelowOrange[lineIndex]
//              || foundGreenBelowOrange[lineIndex]
              )
            {
              circleCalculation.addBallPoint(0,
                orangeSegments[lineIndex][segmentIndex].end.x,
                orangeSegments[lineIndex][segmentIndex].end.y,
                true);
            }
          }
        }//if(orangeSegments[lineIndex][segmentIndex].length > averageLengthOfBallSegments / 2)
      }
    }// for(int segmentIndex = 0; segmentIndex < numberOfOrangeSegments[lineIndex]; segmentIndex++)
  }// for(int lineIndex = 1; lineIndex < maxNumberOfScanLines + 1; lineIndex++)
}

void BallDetector::addBallPoints(Vector2<int> begin, Vector2<int> end)
{
//  return;
  //calculate the distance to the bottom end of the segment
  Vector2<int> pointOnField;
  if(Geometry::calculatePointOnField(end.x, end.y, cameraMatrix, cameraInfo, pointOnField) )
  {
    // calculate the size that an ball in that distance has in the image
    int expectedSizeOfBall = Geometry::getSizeByDistance(cameraInfo, 2 * ballRadius, pointOnField.abs());
    double length = Geometry::distance(begin, end);
    
    // filter the segment if its size differs to much
    if(
      //segment is not to large
      length < expectedSizeOfBall * 1.5 + 4 ||
      //segent is not to small
      length > expectedSizeOfBall / 5 - 4
      )
    {
      circleCalculation.addBallPoint(1, begin.x, begin.y, false);
      circleCalculation.addBallPoint(1, end.x, end.y, true);
    }
  }
}

bool BallDetector::areForbiddenColorsInsideBall(const Geometry::Circle& circle)
{
/*  CIRCLE(imageProcessor_ball4,
    (int)circle.center.x, 
    (int)circle.center.y, 
    (int)circle.radius,
    1, Drawings::ps_solid, Drawings::gray);
*/

  int left = int(circle.center.x - circle.radius),
      right = int(circle.center.x + circle.radius),
      top = int(circle.center.y - circle.radius),
      bottom = int(circle.center.y + circle.radius);
  if(left   <  0)            left  = 0;
  if(right  >= image.cameraInfo.resolutionWidth)  right = image.cameraInfo.resolutionWidth - 1;
  if(top    <  0)            top = 0;
  if(bottom >= image.cameraInfo.resolutionHeight) bottom = image.cameraInfo.resolutionHeight - 1;
  
  if(left >= right || top >= bottom) return false; // no part of circle in image -> cannot be seen
  else
  { // sample ball region
    int xSize = right - left,
        ySize = bottom - top,
        maxCount = min(xSize * ySize, 50),
        count = 0;
    for(int i = 0; i < maxCount; ++i)
    {
      int x = left + int(xSize * random()),
          y = top + int(ySize * random());

      if(Geometry::distance(Vector2<double>(x,y), circle.center) < circle.radius)
      {
        if(
          colorTable64.getColorClass(
          image.image[y][0][x],
          image.image[y][1][x],
          image.image[y][2][x]) == green
           ||
          colorTable64.getColorClass(
          image.image[y][0][x],
          image.image[y][1][x],
          image.image[y][2][x]) == white
          )
        {
          ++count;
          DOT(imageProcessor_ball4, x, y, Drawings::green, Drawings::white);
        }
        else
        {
          DOT(imageProcessor_ball4, x, y, Drawings::black, Drawings::white);
        }
      }
    }
    if(count * 10 > maxCount)
      return true; // to many pixels with forbidden colors -> no ball
  }
  return false;
}

bool BallDetector::areForbiddenColorsInsideBall(const Vector2<double>& offset)
{
  Geometry::Circle circle;
  if(Geometry::calculateBallInImage(offset,cameraMatrix, cameraInfo, circle))
  { // calculate square around ball and clip it with image border
    return areForbiddenColorsInsideBall(circle);
  }
  return false;
}

void BallDetector::addBallPercept
(
 int centerX,
 int centerY,
 double radius
 )
{
//  CIRCLE(imageProcessor_ball4, 5, 5, 5, 1, Drawings::ps_solid, Drawings::orange); 
  Vector2<double>angles;
  Geometry::calculateAnglesForPoint(Vector2<int>(centerX, centerY), cameraMatrix, image.cameraInfo, angles);
  ballPercept.add(
    angles, 
    Geometry::pixelSizeToAngleSize(radius, image.cameraInfo), 
    cameraMatrix.translation, cameraMatrix.isValid);
}

void BallDetector::drawSegment(LineSegment lineSegment)
{
  //lines in color class for debug drawing
  COMPLEX_DRAWING(imageProcessor_coloredSegments3,

    colorClass colorOfLineSegment = colorTable64.getColorClass(lineSegment.averageIntensity[0], lineSegment.averageIntensity[1], lineSegment.averageIntensity[2]);

  Drawings::Color lineColor;
  switch(colorOfLineSegment)
  {
  case orange: lineColor = Drawings::orange; break;
  case yellowOrange: lineColor = Drawings::yellowOrange; break;
  case green: lineColor = Drawings::green; break;
  case white: lineColor = Drawings::white; break;
  case yellow: lineColor = Drawings::yellow; break;
  case skyblue: lineColor = Drawings::skyblue; break;
  case pink: lineColor = Drawings::pink; break;
  default: lineColor = Drawings::gray;
  }
  LINE(imageProcessor_coloredSegments3, 
    lineSegment.begin.x, lineSegment.begin.y,
    lineSegment.end.x, lineSegment.end.y,
    2, Drawings::ps_solid, lineColor);
  );

  //lines in color class for debug drawing
  COMPLEX_DRAWING(imageProcessor_coloredSegments2,

    colorClass colorOfLineSegment = colorTableReferenceColor.getColorClass(lineSegment.averageIntensity[0], lineSegment.averageIntensity[1], lineSegment.averageIntensity[2]);

  Drawings::Color lineColor;
  switch(colorOfLineSegment)
  {
  case orange: lineColor = Drawings::orange; break;
  case yellowOrange: lineColor = Drawings::yellowOrange; break;
  case green: lineColor = Drawings::green; break;
  case white: lineColor = Drawings::white; break;
  case yellow: lineColor = Drawings::yellow; break;
  case skyblue: lineColor = Drawings::skyblue; break;
  case pink: lineColor = Drawings::pink; break;
  default: lineColor = Drawings::gray;
  }
  LINE(imageProcessor_coloredSegments2,
    lineSegment.begin.x, lineSegment.begin.y,
    lineSegment.end.x, lineSegment.end.y,
    2, Drawings::ps_solid, lineColor);
  );

  //lines in color class for debug drawing
  COMPLEX_DRAWING(imageProcessor_coloredSegments1,

    colorClass colorOfLineSegment = colorTableCuboids.getColorClass(lineSegment.averageIntensity[0], lineSegment.averageIntensity[1], lineSegment.averageIntensity[2]);

  Drawings::Color lineColor;
  switch(colorOfLineSegment)
  {
  case green: lineColor = Drawings::green; break;
  case white: lineColor = Drawings::white; break;
  default: lineColor = Drawings::gray;
  }
  LINE(imageProcessor_coloredSegments1,
    lineSegment.begin.x, lineSegment.begin.y,
    lineSegment.end.x, lineSegment.end.y,
    4, Drawings::ps_solid, lineColor);
  );



  // lines in average color for debug image
#if defined(_WIN32) && !defined(SIMROBOT)
  //visualization
  /*  Geometry::PixeledLine linePixels(lineSegment.begin, lineSegment.end);
  for(int i = 0; i < linePixels.getNumberOfPixels(); i++)
  {
  imageProcessorGeneralImage.image[linePixels.getPixelY(i)][0][linePixels.getPixelX(i)] = lineSegment.averageIntensity[0];
  imageProcessorGeneralImage.image[linePixels.getPixelY(i)][1][linePixels.getPixelX(i)] = lineSegment.averageIntensity[1];
  imageProcessorGeneralImage.image[linePixels.getPixelY(i)][2][linePixels.getPixelX(i)] = lineSegment.averageIntensity[2];
  }
  */
#endif
}


/*
* Change log :
* 
* $Log: BallDetector.cpp,v $
* Revision 1.20  2004/05/07 15:16:25  nistico
* All geometry calculations are making use of intrinsic functions.
* I updated most of the images processors to make correct use of this.
* Please test if there are any problems, because i'm going to remove the
* old code soon.
*
* Revision 1.19  2004/02/12 14:31:31  juengel
* Added drawing ball4.
*
* Revision 1.18  2003/12/25 10:46:29  juengel
* removed comment
*
* Revision 1.17  2003/12/15 11:46:12  juengel
* Introduced CameraInfo
*
* Revision 1.16  2003/12/04 17:38:18  juengel
* Moved circle calculation and orange calibration to own classes.
*
* Revision 1.15  2003/12/01 16:24:18  juengel
* Added drawing ball3.
*
* Revision 1.14  2003/11/26 10:31:22  juengel
* no message
*
* Revision 1.13  2003/11/15 17:09:02  juengel
* changed BallPercept
*
* Revision 1.12  2003/11/12 17:38:44  juengel
* bottom ball points with no green below are filtered out
*
* Revision 1.11  2003/11/12 16:19:35  goehring
* frameNumber added to percepts
*
* Revision 1.10  2003/11/11 17:03:16  juengel
* added addBallPoints
*
* Revision 1.9  2003/11/10 11:31:18  juengel
* check green below ball.
*
* Revision 1.8  2003/11/07 11:22:41  juengel
* color class based version added
*
* Revision 1.7  2003/11/05 16:46:46  juengel
* Added drawing imageProcessor_ball2
*
* Revision 1.6  2003/11/05 16:34:28  goehring
* FrameNumber added
*
* Revision 1.5  2003/11/03 20:26:33  juengel
* Added color class yellowOrange
*
* Revision 1.4  2003/10/30 18:28:15  juengel
* Added parameter colorTableCuboids to LineSegment::isFieldBorder
*
* Revision 1.3  2003/10/29 12:57:15  juengel
* added references to colorSpaceUsageCounte and colorTableReferenceColor
*
* Revision 1.2  2003/10/23 07:16:38  juengel
* Renamed ColorTableAuto to ColorTableReferenceColor.
*
* Revision 1.1  2003/10/06 14:10:13  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.7  2003/09/01 15:06:45  juengel
* Removed unused code.
*
* Revision 1.6  2003/09/01 10:16:16  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.5  2003/08/30 10:19:52  juengel
* DebugDrawings clean-up 1
*
* Revision 1.4  2003/08/29 13:11:02  juengel
* Removed detection of yellow.
*
* Revision 1.3  2003/08/25 17:26:09  juengel
* Warnings removed.
*
* Revision 1.2  2003/08/22 17:04:03  roefer
* Warnings removed
*
* Revision 1.1  2003/08/18 11:47:53  juengel
* Added segment based detectors.
*
*
*/
