/**
* @file BallDetector.h
*
* Definition of class BallDetector
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#ifndef __BallDetector_h_
#define __BallDetector_h_

#include "Representations/Perception/Image.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Perception/ColorTable64.h"
#include "Representations/Perception/BallPercept.h"

#include "Tools/Debugging/DebugImages.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Boundary.h"

#include "SegmentBasedDetector.h"
#include "../ColorTableCuboids.h"
#include "../ColorTableReferenceColor.h"
#include "../CircleCalculation.h"

/**
* @class BallDetector
*
* The ball detector detects a ball in the image.
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/ 

class BallDetector
{
public:
  BallDetector(
    const Image& image, 
    const CameraMatrix& cameraMatrix, 
    const CameraInfo& cameraInfo, 
    ColorTable64& colorTable64, 
    ColorTableCuboids& colorTableCuboids, 
    ColorTableReferenceColor& colorTableReferenceColor, 
    BallPercept& ballPercept
    );
  ~BallDetector();

  void init();
  void addSegment(LineSegment& newSegment, int lineIndex);
  void createBallPercept();

  void addBallPoints(Vector2<int> begin, Vector2<int> end);

private:
  enum {maxNumberOfScanLines = 240}; /**< The maximum number scan lines. */
  enum {maxNumberOfSegmentsPerScanLine = 20}; /**< The maximum number of segments per line. */

  LineSegment orangeSegments[maxNumberOfScanLines + 2][maxNumberOfSegmentsPerScanLine];
  int numberOfOrangeSegments[maxNumberOfScanLines + 2];

  bool foundSegmentWithNoColorBelowOrange[maxNumberOfScanLines + 2];
  bool foundGreenBelowOrange[maxNumberOfScanLines + 2];

  CircleCalculation circleCalculation;

  int lineIndexOfLastSegment;

  bool foundBorderPoints;
  Boundary<int>* borderPointBoundary;

  void addOrangeC64Segment(LineSegment& newSegment, int lineIndex);
  void determineBallPoints();
  void addBallPercept(int centerX, int centerY, double radius);  
  
  bool areForbiddenColorsInsideBall(const Geometry::Circle& circle);
  bool areForbiddenColorsInsideBall(const Vector2<double>& offset);

  void drawSegment(LineSegment lineSegment);

  void scanBallHorizontal();

  /** A reference to the image that is scanned for a goal */
  const Image& image;

  /** A reference to the camera matrix that describes position and rotation of the camera when the image was aquired */
  const CameraMatrix& cameraMatrix;

  /** A reference to the camera info that describes the resolution and the opening angles of the camera */
  const CameraInfo& cameraInfo;

  /** A reference to the color table */
  ColorTable64& colorTable64;

  /** A reference to the color table */
  ColorTableReferenceColor& colorTableReferenceColor;

  /** A reference to the color table */
  ColorTableCuboids& colorTableCuboids;

  /** A reference to the ball percept */
  BallPercept& ballPercept;
};

#endif // __BallDetector_h_

/*
* Change log :
* 
* $Log: BallDetector.h,v $
* Revision 1.11  2003/12/15 11:46:13  juengel
* Introduced CameraInfo
*
* Revision 1.10  2003/12/04 17:38:18  juengel
* Moved circle calculation and orange calibration to own classes.
*
* Revision 1.9  2003/11/15 17:09:02  juengel
* changed BallPercept
*
* Revision 1.8  2003/11/12 17:38:45  juengel
* bottom ball points with no green below are filtered out
*
* Revision 1.7  2003/11/11 17:03:16  juengel
* added addBallPoints
*
* Revision 1.6  2003/11/10 11:31:17  juengel
* check green below ball.
*
* Revision 1.5  2003/11/07 11:22:41  juengel
* color class based version added
*
* Revision 1.4  2003/11/05 16:46:46  juengel
* Added drawing imageProcessor_ball2
*
* Revision 1.3  2003/10/29 12:57:15  juengel
* added references to colorSpaceUsageCounte and colorTableReferenceColor
*
* Revision 1.2  2003/10/23 07:16:38  juengel
* Renamed ColorTableAuto to ColorTableReferenceColor.
*
* Revision 1.1  2003/10/06 14:10:13  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.4  2003/09/26 15:27:48  juengel
* Renamed DataTypes to representations.
*
* Revision 1.3  2003/09/01 10:16:16  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.2  2003/08/30 10:19:52  juengel
* DebugDrawings clean-up 1
*
* Revision 1.1  2003/08/18 11:47:53  juengel
* Added segment based detectors.
*
*/
