/**
* @file LineSegmentation.cpp
*
* Implementation of class LineSegmentation
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#include "LineSegmentation.h"

#include "Tools/RingBuffer.h"
#include "Tools/Debugging/DebugImages.h"
#include "Tools/Boundary.h"

LineSegmentation::LineSegmentation
(
 const Vector2<int>& start, 
 const Vector2<int>& end, 
 const Image& image,
 int threshold1,
 int threshold2,
 int threshold3,
 ColorSpaceUsageCounter& colorSpaceUsageCounter
 ) :
start(start),
end(end),
image(image),
threshold1(threshold1),
threshold2(threshold2),
threshold3(threshold3),
colorSpaceUsageCounter(colorSpaceUsageCounter)
{
  numberOfSegments = 0;

  Boundary<int> imageBoundary(0, image.cameraInfo.resolutionWidth);
  imageBoundary.add(Vector2<int>(0,0));
  imageBoundary.add(Vector2<int>(image.cameraInfo.resolutionWidth, image.cameraInfo.resolutionHeight));

  numberOfScannedPixels = 0;
  accumulatedIntensity[0] = 0;
  accumulatedIntensity[1] = 0;
  accumulatedIntensity[2] = 0;

  accumulatedEdgeness[0] = 0;
  accumulatedEdgeness[1] = 0;
  accumulatedEdgeness[2] = 0;

  if(!imageBoundary.isInside(start) || !imageBoundary.isInside(end)) return;
  
  doSegmentation();
}

LineSegmentation::~LineSegmentation()
{
}


int LineSegmentation::getNumberOfSegments()
{
  return numberOfSegments;
}

LineSegment& LineSegmentation::getSegment(int index)
{
  return segments[index];
}

void LineSegmentation::doSegmentation()
{
  int x = 0, y = 0; //the current position in the image
  int lastX, lastY;
  
  // for segmentation  
  RingBuffer<int,3> intensity[3]; //a buffer for the intensities of the three channels
  int changeY, changeU, changeV; //indicates the type of an edge:  1: up,   0: no edge,   -1: down
  
  enum {onEdge, betweenEdges} edgeState = betweenEdges; // states for edge detection
  int numberOfPixelsSinceLastEdge;
  
  enum {up, down, nothing} changeYState = nothing;
  Vector2<int> topOfRoof;
  
  Geometry::PixeledLine line(start, end);
  if(line.getNumberOfPixels() > 3)
  {
    // buffer the first two pixels
    x = line.getPixelX(0); y = line.getPixelY(0);
    intensity[0].add(image.image[y][0][x]);
    intensity[1].add(image.image[y][1][x]);
    intensity[2].add(image.image[y][2][x]);
    x = line.getPixelX(1); y = line.getPixelY(1);
    intensity[0].add(image.image[y][0][x]);
    intensity[1].add(image.image[y][1][x]);
    intensity[2].add(image.image[y][2][x]);
    
    // init local statistics
    numberOfPixelsSinceLastEdge = 1;
    Vector2<int> firstPixelAfterLastEdge(line.getPixelX(2), line.getPixelY(2));
    bool firstPixelAfterLastEdgeWasBelowHorizon = false;
//      Geometry::getDistanceToLine(horizonLine, Vector2<double>(x,y)) < -5;
    
    int accumulatedIntensitySinceLastChange[3] = {0,0,0};
    int minIntensitySinceLastChange[3] = {255, 255, 255};
    int maxIntensitySinceLastChange[3] = {0, 0, 0};
    
    //for all pixels of the current line
    for(int z = 2; z < line.getNumberOfPixels(); z++) 
    {
      numberOfScannedPixels++;
      
      x = line.getPixelX(z); y = line.getPixelY(z);
      lastX = x; lastY = y;
      
      // segmentation of scan lines
      intensity[0].add(image.image[y][0][x]);
      intensity[1].add(image.image[y][1][x]);
      intensity[2].add(image.image[y][2][x]);
      
      // global statistics
      colorSpaceUsageCounter.addColor(image.image[y][0][x],image.image[y][1][x],image.image[y][2][x]);
      accumulatedIntensity[0] += intensity[0][0]; 
      accumulatedIntensity[1] += intensity[1][0]; 
      accumulatedIntensity[2] += intensity[2][0]; 
      
      //detect edges
      int differenceY = intensity[0][0] - intensity[0][2];
      int differenceU = intensity[1][0] - intensity[1][2];
      int differenceV = intensity[2][0] - intensity[2][2];
      
      if(differenceY > threshold1)       { changeY = 1; DOT(imageProcessor_gradients, x-1, y, Drawings::gray, Drawings::gray); }
      else if(differenceY < -threshold1) { changeY = -1;DOT(imageProcessor_gradients, x-1, y, Drawings::black, Drawings::black); }
      else                                              changeY = 0;
      
      if(differenceU > threshold2)       { changeU = 1; DOT(imageProcessor_gradients, x, y, Drawings::pink, Drawings::pink); }
      else if(differenceU < -threshold2) { changeU = -1;DOT(imageProcessor_gradients, x, y, Drawings::red, Drawings::red);}
      else                                              changeU = 0;
      
      if(differenceV > threshold3)       { changeV = 1; DOT(imageProcessor_gradients, x+1, y, Drawings::skyblue, Drawings::skyblue);}
      else if(differenceV < -threshold3) { changeV = -1;DOT(imageProcessor_gradients, x+1, y, Drawings::blue, Drawings::blue); }
      else                                              changeV = 0;
      
      accumulatedEdgeness[0] += abs(differenceY);
      accumulatedEdgeness[1] += abs(differenceU);
      accumulatedEdgeness[2] += abs(differenceV);
      
      // edge state machine
      if(edgeState == betweenEdges)
      {
        if(changeY != 0 || changeU != 0 || changeV != 0)
        {
          //create and add the segment
          LineSegment newSegment(
            firstPixelAfterLastEdge, x, y, numberOfPixelsSinceLastEdge,
            accumulatedIntensitySinceLastChange[0] / numberOfPixelsSinceLastEdge,
            accumulatedIntensitySinceLastChange[1] / numberOfPixelsSinceLastEdge,
            accumulatedIntensitySinceLastChange[2] / numberOfPixelsSinceLastEdge,
            minIntensitySinceLastChange, maxIntensitySinceLastChange,
            firstPixelAfterLastEdgeWasBelowHorizon, 
            false,
//            Geometry::getDistanceToLine(horizonLine, Vector2<double>(x,y)) < 0,
            false,
            image.cameraInfo);

          segments[numberOfSegments++] = newSegment;
          if(numberOfSegments >= maxNumberOfSegmentsPerScanLine )
            numberOfSegments = maxNumberOfSegmentsPerScanLine - 1;
          
          // change state, reset counters
          edgeState = onEdge;
          if(changeY == 1) changeYState = up;
          else changeYState = nothing;
          
          firstPixelAfterLastEdge.x = x; firstPixelAfterLastEdge.y = y;
          accumulatedIntensitySinceLastChange[0] = intensity[0][0]; 
          accumulatedIntensitySinceLastChange[1] = intensity[1][0]; 
          accumulatedIntensitySinceLastChange[2] = intensity[2][0]; 
          
          minIntensitySinceLastChange[0] = intensity[0][0];
          minIntensitySinceLastChange[1] = intensity[1][0];
          minIntensitySinceLastChange[2] = intensity[2][0];
          
          maxIntensitySinceLastChange[0] = intensity[0][0];
          maxIntensitySinceLastChange[1] = intensity[1][0];
          maxIntensitySinceLastChange[2] = intensity[2][0];
          
          firstPixelAfterLastEdgeWasBelowHorizon = false;
            //Geometry::getDistanceToLine(horizonLine, Vector2<double>(x,y)) < -1;
          
          numberOfPixelsSinceLastEdge = 1;
        }//if(changeY != 0 || changeU != 0 || changeV != 0)
        else
        {
          accumulatedIntensitySinceLastChange[0] += intensity[0][0]; 
          accumulatedIntensitySinceLastChange[1] += intensity[1][0]; 
          accumulatedIntensitySinceLastChange[2] += intensity[2][0]; 
          if(intensity[0][0] < minIntensitySinceLastChange[0]) minIntensitySinceLastChange[0] = intensity[0][0];
          if(intensity[1][0] < minIntensitySinceLastChange[1]) minIntensitySinceLastChange[1] = intensity[1][0];
          if(intensity[2][0] < minIntensitySinceLastChange[2]) minIntensitySinceLastChange[2] = intensity[2][0];
          
          if(intensity[0][0] > maxIntensitySinceLastChange[0]) maxIntensitySinceLastChange[0] = intensity[0][0];
          if(intensity[1][0] > maxIntensitySinceLastChange[1]) maxIntensitySinceLastChange[1] = intensity[1][0];
          if(intensity[2][0] > maxIntensitySinceLastChange[2]) maxIntensitySinceLastChange[2] = intensity[2][0];
          numberOfPixelsSinceLastEdge++;
        }
      }//if(edgeState == betweenEdges)
      else if(edgeState == onEdge)
      {
        if(changeY == 0 && changeU == 0 && changeV == 0) 
        {
          if(changeYState == down)
          {
            //create and add a segment
            LineSegment newSegment(
              (firstPixelAfterLastEdge + topOfRoof) / 2, 
              (x + topOfRoof.x) / 2, (y + topOfRoof.y) / 2, 
              numberOfPixelsSinceLastEdge,
              accumulatedIntensitySinceLastChange[0] / numberOfPixelsSinceLastEdge,
              accumulatedIntensitySinceLastChange[1] / numberOfPixelsSinceLastEdge,
              accumulatedIntensitySinceLastChange[2] / numberOfPixelsSinceLastEdge,
              minIntensitySinceLastChange, maxIntensitySinceLastChange,
              firstPixelAfterLastEdgeWasBelowHorizon,
              false,
//              Geometry::getDistanceToLine(horizonLine, Vector2<double>(x,y)) < 0,
              true,
              image.cameraInfo);
            
            LINE(imageProcessor_general, newSegment.begin.x, newSegment.begin.y, newSegment.end.x, newSegment.end.y, 2, Drawings::ps_solid, Drawings::red);
            
            segments[numberOfSegments++] = newSegment;
            if(numberOfSegments >= maxNumberOfSegmentsPerScanLine )
              numberOfSegments = maxNumberOfSegmentsPerScanLine - 1;
          }
          
          edgeState = betweenEdges;
          //reset counters
          firstPixelAfterLastEdge.x = x; firstPixelAfterLastEdge.y = y;
          accumulatedIntensitySinceLastChange[0] = intensity[0][0]; 
          accumulatedIntensitySinceLastChange[1] = intensity[1][0]; 
          accumulatedIntensitySinceLastChange[2] = intensity[2][0]; 
          
          minIntensitySinceLastChange[0] = intensity[0][0];
          minIntensitySinceLastChange[1] = intensity[1][0];
          minIntensitySinceLastChange[2] = intensity[2][0];
          
          maxIntensitySinceLastChange[0] = intensity[0][0];
          maxIntensitySinceLastChange[1] = intensity[1][0];
          maxIntensitySinceLastChange[2] = intensity[2][0];
          
          firstPixelAfterLastEdgeWasBelowHorizon = false;
//            Geometry::getDistanceToLine(horizonLine, Vector2<double>(x,y)) < -1;
          
          numberOfPixelsSinceLastEdge = 1;
        }//if(changeY == 0 && changeU == 0 && changeV == 0)
        if(changeYState == up)
        {
          if(changeY == -1) 
          {
            changeYState = down;
            topOfRoof.x = x;
            topOfRoof.y = y;
          }
        }
        if(changeYState == down)
        {
          if(changeY == 1) changeYState = nothing;
        }
      }
    }//for(int z = 2; z < line.getNumberOfPixels(); z++)
    
    LineSegment newSegment(
      firstPixelAfterLastEdge, x, y, numberOfPixelsSinceLastEdge,
      accumulatedIntensitySinceLastChange[0] / numberOfPixelsSinceLastEdge,
      accumulatedIntensitySinceLastChange[1] / numberOfPixelsSinceLastEdge,
      accumulatedIntensitySinceLastChange[2] / numberOfPixelsSinceLastEdge,
      minIntensitySinceLastChange, maxIntensitySinceLastChange,
      firstPixelAfterLastEdgeWasBelowHorizon,
      false,
//      Geometry::getDistanceToLine(horizonLine, Vector2<double>(x,y)) < 0,
      false,
      image.cameraInfo);

    segments[numberOfSegments++] = newSegment;
    if(numberOfSegments >= maxNumberOfSegmentsPerScanLine )
      numberOfSegments = maxNumberOfSegmentsPerScanLine - 1;
  }//if(line.getNumberOfPixels() > 3)
}

int LineSegmentation::getAccumulatedIntensity(int channel)
{
  return accumulatedIntensity[channel];
}

int LineSegmentation::getAccumulatedEdgeness(int channel)
{
  return accumulatedEdgeness[channel];
}

int LineSegmentation::getNumberOfScannedPixels()
{
  return numberOfScannedPixels;
}

/*
* Change log :
* 
* $Log: LineSegmentation.cpp,v $
* Revision 1.4  2004/01/04 10:21:45  juengel
* bounded numberOfSegments
*
* Revision 1.3  2003/12/15 11:49:06  juengel
* Introduced CameraInfo
*
* Revision 1.2  2003/12/04 17:35:41  juengel
* Added LineSegmentation.
*
* Revision 1.1  2003/12/04 09:43:46  juengel
* Added LineSegmentation
*
*/
