/**
* @file OrangeCalibration.h
*
* Definition of class OrangeCalibration
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#ifndef __OrangeCalibration_h_
#define __OrangeCalibration_h_

#include "SegmentBasedDetector.h"

#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Perception/ColorTable64.h"

#include "../ColorTableCuboids.h"
#include "../ColorTableReferenceColor.h"
#include "../ColorSpaceUsageCounter.h"

#include "Tools/Debugging/DebugImages.h"
#include "Tools/Math/Geometry.h"

/**
* @class OrangeCalibration
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/ 

class OrangeCalibration
{
public:
  OrangeCalibration(
    const CameraMatrix& cameraMatrix, 
    const CameraInfo& cameraInfo, 
    ColorTable64& colorTable64, 
    ColorTableCuboids& colorTableCuboids, 
    ColorTableReferenceColor& colorTableReferenceColor
    );
  ~OrangeCalibration();

  void init();

  void addSegment(LineSegment& newSegment, int lineIndex);

  enum {maxNumberOfScanLines = 240}; /**< The maximum number scan lines. */
  enum {maxNumberOfSegmentsPerScanLine = 20}; /**< The maximum number of segments per line. */
  LineSegment segments[maxNumberOfScanLines + 2][maxNumberOfSegmentsPerScanLine];
  int numberOfSegments[maxNumberOfScanLines + 2];

  LineSegment orangeSegments[maxNumberOfScanLines + 2][maxNumberOfSegmentsPerScanLine];
  int numberOfOrangeSegments[maxNumberOfScanLines + 2];

  bool foundSegmentWithNoColorBelowOrange[maxNumberOfScanLines + 2];

  void calibrateOrange();

private:
  void addOrangeReferenceColorSegment(LineSegment& newSegment, int lineIndex);
  int lineIndexOfLastSegment;
  enum {noOrangeFound, orangeFound, noGreenFound, greenFound} calibrationState;

  /** A reference to the camera matrix that describes position and rotation of the camera when the image was aquired */
  const CameraMatrix& cameraMatrix;

  /** A reference to the camera info that describes resolution and the opening angles of the camera */
  const CameraInfo& cameraInfo;

  /** A reference to the color table */
  ColorTable64& colorTable64;

  /** A reference to the color table */
  ColorTableReferenceColor& colorTableReferenceColor;

  /** A reference to the color table */
  ColorTableCuboids& colorTableCuboids;
};

#endif // __OrangeCalibration_h_

/*
* Change log :
* 
* $Log: OrangeCalibration.h,v $
* Revision 1.2  2003/12/15 11:49:06  juengel
* Introduced CameraInfo
*
* Revision 1.1  2003/12/04 09:44:03  juengel
* Added OrangeCalibration
*
*/
