/**
* @file SegmentBasedDetector.h
*
* Definition of class SegmentBasedDetector
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#ifndef __SegmentBasedDetector_h_
#define __SegmentBasedDetector_h_

#include "Tools/Math/Geometry.h"
#include "Tools/Math/Vector2.h"
#include "Representations/Perception/ColorTable64.h"
#include "Representations/Perception/CameraMatrix.h"
#include "../ColorTableCuboids.h"
#include "../ColorTableReferenceColor.h"

struct LineSegment
{
  Vector2<int> begin;
  Vector2<int> end;
  int length;
  int minIntensity[3];
  int maxIntensity[3];
  int averageIntensity[3];
  bool beginIsOnBorder;
  bool endIsOnBorder;
  bool beginIsBelowHorizon;
  bool endIsBelowHorizon;
  bool isRoofEdge;
  CameraInfo cameraInfo;

  LineSegment(Vector2<int>begin, int endX, int endY, int length,
    int averageIntensity0,
    int averageIntensity1,
    int averageIntensity2,
    int minIntensity[3],
    int maxIntensity[3],
    bool beginIsBelowHorizon,
    bool endIsBelowHorizon,
    bool isRoofEdge,
    CameraInfo cameraInfo
    ) : cameraInfo(cameraInfo)
  {
    this->begin = begin;
    end.x = endX;
    end.y = endY;
    this->length = length;
    averageIntensity[0] = averageIntensity0;
    averageIntensity[1] = averageIntensity1;
    averageIntensity[2] = averageIntensity2;
    this->minIntensity[0] = minIntensity[0];
    this->minIntensity[1] = minIntensity[1];
    this->minIntensity[2] = minIntensity[2];
    this->maxIntensity[0] = maxIntensity[0];
    this->maxIntensity[1] = maxIntensity[1];
    this->maxIntensity[2] = maxIntensity[2];
    
    this->beginIsBelowHorizon = beginIsBelowHorizon;
    this->endIsBelowHorizon = endIsBelowHorizon;
    this->isRoofEdge = isRoofEdge;

    beginIsOnBorder = 
      begin.y < 10 || 
      begin.y > cameraInfo.resolutionHeight - 11 || 
      begin.x < 10 || 
      begin.x > cameraInfo.resolutionWidth - 11;

    endIsOnBorder = 
      endY < 10 || 
      endY > cameraInfo.resolutionHeight - 11 || 
      endX < 10 || 
      endX > cameraInfo.resolutionWidth - 11;

  }

  LineSegment() {};

  bool isFieldBorder_forCalibration(
    const ColorTableCuboids& colorTableCuboids,
    const ColorTableReferenceColor& colorTableReferenceColor,
    const CameraMatrix& cameraMatrix
    )
  {
    return(
      beginIsBelowHorizon && 
      length > 4 &&
      length > Geometry::calculateLineSize(end, cameraMatrix, cameraInfo) * 4 &&
      colorTableCuboids.getColorClass(averageIntensity[0], averageIntensity[1], averageIntensity[2]) == white &&
      colorTableReferenceColor.getColorClass(averageIntensity[0], averageIntensity[1], averageIntensity[2]) != green
      );
  }

  bool isFieldBorder_forDetection(
    const ColorTable64& colorTable64,
    const CameraMatrix& cameraMatrix
    )
  {
    return(
      length > 3 &&
      length > Geometry::calculateLineSize(end, cameraMatrix, cameraInfo) * 3 &&
      colorTable64.getColorClass(averageIntensity[0], averageIntensity[1], averageIntensity[2]) == white
      );
  }


  bool isFieldLine(const ColorTable64& colorTable64, const CameraMatrix& cameraMatrix)
  {
    return(
      beginIsBelowHorizon && 
      !beginIsOnBorder &&
      !endIsOnBorder &&
//      length > 5 &&
      length < Geometry::calculateLineSize(end, cameraMatrix, cameraInfo) * 2.5 &&
      (
      colorTable64.getColorClass(averageIntensity[0], averageIntensity[1], averageIntensity[2]) == white
      ||
      isRoofEdge)
      );
  }

  colorClass isGoal_forCalibration(const ColorTableReferenceColor& colorTableReferenceColor)
  {
    colorClass color = colorTableReferenceColor.getColorClass(averageIntensity[0], averageIntensity[1], averageIntensity[2]);

    if(!beginIsBelowHorizon && 
      endIsBelowHorizon &&
      length > 10 &&
      colorTableReferenceColor.getColorClass(minIntensity[0], minIntensity[1], minIntensity[2]) == color
      )
    {
      if (color == yellow || color == yellowOrange) return yellow;
      if (color == skyblue) return skyblue;
    }
    return noColor;
  }
  colorClass isGoal_forDetection(const ColorTable64& colorTable64)
  {
    colorClass color = colorTable64.getColorClass(averageIntensity[0], averageIntensity[1], averageIntensity[2]);

    if(endIsBelowHorizon && length > 7)
    {
      if (color == yellow) return yellow;
      if (color == skyblue) return skyblue;
    }
    return noColor;
  }
};

class SegmentBasedDetector
{
public:

protected:

};

#endif // __SegmentBasedDetector_h_

/*
* Change log :
* 
* $Log: SegmentBasedDetector.h,v $
* Revision 1.11  2003/12/31 23:50:38  roefer
* Removed inclusion of RobotDimensions.h because it is not used
*
* Revision 1.10  2003/12/15 11:49:06  juengel
* Introduced CameraInfo
*
* Revision 1.9  2003/12/04 17:36:13  juengel
* Missing includes added
*
* Revision 1.8  2003/11/11 16:56:17  juengel
* white is not calibrateted if the segment is green.
*
* Revision 1.7  2003/11/10 11:33:13  juengel
* added calibration and detection versions for isFieldBorder, isGoal and isFieldLine
*
* Revision 1.6  2003/11/05 16:41:49  juengel
* increased size for the border
*
* Revision 1.5  2003/11/03 20:26:15  juengel
* Added color class yellowOrange
*
* Revision 1.4  2003/10/30 18:28:15  juengel
* Added parameter colorTableCuboids to LineSegment::isFieldBorder
*
* Revision 1.3  2003/10/29 13:04:53  juengel
* New constructor for LineSegment.
*
* Revision 1.2  2003/10/23 07:32:24  juengel
* Renamed ColorTableAuto to ColorTableReferenceColor.
*
* Revision 1.1  2003/10/06 14:10:13  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.4  2003/08/30 10:13:53  juengel
* Changed parameter in isBorder
*
* Revision 1.3  2003/08/29 13:10:28  juengel
* Added isFieldLine.
*
* Revision 1.2  2003/08/25 17:26:34  juengel
* changed isGoal
*
* Revision 1.1  2003/08/18 11:47:53  juengel
* Added segment based detectors.
*
*/
