/** 
* @file SegmentationTools.h
* Declaration of file SegmentationTools.
*
* @author <A href=mailto:damdfree.fr>Damien DEOM</A>
*/

/*------------------------------------------------------------------------
you'll find here some interesting data structures an tools for image processing.
I added an slist on my own that can be used in other domains than the little
 world of image processing ...
------------------------------------------------------------------------*/

#ifndef _SEGMENTATION_TOOLS_
#define _SEGMENTATION_TOOLS_



#include "Tools/Math/Geometry.h"
#include "Tools/Math/Vector2.h"
#include "Tools/ColorClasses.h"
//#include "MSH2004EdgeDetection.h"


typedef Vector2<int> point;
typedef Vector2<double> direction;

struct LinePair2;

//---------------------------------------------------------------------------------------
// geometrical stuff



inline point mil(point& v1,point& v2) {
	return point ((v1.x+v2.x) / 2,(v1.y+v2.y) / 2);
};



/** Gives a number between 0 and 360 with the same order as v.angle()*/
inline double theta2(point v1, point v2){
	double t = 0;
	int dx = v1.x - v2.x;
	int dy = v1.y - v2.y;
	int ax = abs(dx);
	int ay = abs(dy);
	if (dx == 0 && dy == 0) t = 0;
	else t = (double)dy /(double)(ax + ay);
	if (dx < 0) t = 2 - t;
	else if (dy < 0) t = 4 + t;

 	return t*90; // multiplication with 90 is not really needed for correct ordering
}


inline double AngleRelativeToHorizontal(const point & pt1, const point& pt2) {
	double d1 =  abs(pt2.x - pt1.x);
	double d2 =  (pt2 - pt1).abs();
  if ((d2 == 0) || (d1/d2 < -1) || (d1/d2 > 1))
    return 0;
  else
	  return acos(d1/d2);
}

/*
inline double AngleBetweenLines(LinePair2* l1, LinePair2* l2)  {

	Vector2<double> pt1();
	Geometry::getIntersectionOfLines(l1->line,l2->line,pt1());
	Vector2<double> v2(l1->line.direction.x,l1->line.direction.y);
	double d =  Geometry::getDistanceToLine(l2->line,v2);

	return asin(d/(l1->line.direction-l1->line.base).abs());
}

*/





//---------------------------------------------------------------------------
// implementation of a special slist : all the objects to be listed must derive from the 
// template class "listed" (use the name of the class as template parameter for normal use) 
// dont be afraid to use it, it's guaranted without bugs :-)


static int cpt =0;

template <class T> class listed {
protected:

	T* next;


public:
	listed(): next(0) {}
	~listed() {}

	T* getNext(){return next;}

	void setNext(T* f) {next = f;}

	void insert(T* i) {
		if (!i) return;
		T* tmp = next;
		next = i;
		i->setNext(tmp);
	}

	void swapNext(T* toSwapWith) {  // swap the element after next with toSwapWith
		
		if (!(next && toSwapWith && toSwapWith->next)) return;
		T* tmp = this->next;
		T* tmp2 = toSwapWith->next->next;

		this->next = toSwapWith->next;
		toSwapWith->next->next = tmp;
		toSwapWith->next = tmp2;
	}

};





template <class T> class slist {
private:
	unsigned int size;
	T* first, *last;

public:

	class iterator {

	private:
		T* elt;
	public:
	
		iterator(T* l = 0) : elt(l) {}
		//~iterator() {if (elt) delete elt;}

		T* operator++() {
			if (!elt) return 0;
			elt= elt->getNext();
			return elt;
		}
		T* operator++(int) {
			if (!elt) return 0;
			T* old = elt;
			elt = elt->getNext();
			return old;
		}
		T*operator+=(const int n) {
			T* tmp = elt;
			int i;
			for (i = 0; i<n;++i) 
				if (tmp) tmp = tmp->getNext();
				else return 0;
			elt = tmp;
			return elt;
		}
		T* operator = (T* e) {
			elt = e;
			return elt;
		}
		bool operator != (const T* e) {
			return (elt != e);
		}

		T* operator ->() {return elt;}
		T* operator *() {return elt;}
		bool end() {return !elt;}
	};

	T* front()  {return first;}
	T* back()  {return last;}
	void setLast(T* l) {last = l;}
	
	void cutFront() { first = last = 0;size = 0;}

	slist() : size(0), first(0), last(0){}
/*	~slist() {
		if (first) delete first;
		if (last) delete last;
	}*/
	
	void push_front(T* lp) {
		if (!lp) return;
		if (!last) last = lp; 

		lp->setNext(first);
		first = lp;
		++size;
	}

	void push_back(T* lp) {
		if (!lp) return;
		if (!first) {first = last = lp;} else last->setNext(lp); 
		last = lp;
		++size;
	}

	void push_back(slist<T>& lst) {
		if (last) {
			last->setNext(lst.front());
			last = lst.back();
		} else {
			first = lst.front();
			last = lst.back();
		}
		size+= lst.getSize();
		lst.cutFront();
	}

	unsigned getSize() const {return size;}
	bool empty() {return size==0;}
	void incSize(unsigned int n) {if (n>0) size+=n;}

	void clear() {
		while (first) {
			T* tmp = first->getNext();
			delete first;
			first = tmp;
		}
		last = 0;
		size = 0;
	}

	T* pop_front() {
		if (!first) return 0;
		T* tmp = first->getNext();
		delete first;
		first = tmp;
		if (size==1) last = first;
		--size;
		return first;
	}

	void insert(T*  pos, T* i) {
		if (!pos) return;
		T* tmp = pos->getNext();

		if (!tmp) {
			pos->setNext(i);
			i->setNext(0);
			last = i;
		} else {
			pos->setNext(i);
			i->setNext(tmp);
		}
		if (pos==last) last = i;
		++size;
	}

	void erase(T* e) {
		if (e->getNext()==last) last = e;
		T* n = e->getNext()->getNext();
		delete e->getNext();
		e->setNext(n);
		--size;
	}

};



//--------------------------------------------------------------------
// definition of an edge and a linePair : very convenient for mixing both 
// in a list.
// If you only want to use linePairs, please use the LinePair2 struct defined below

struct figure : public listed<figure> {


	figure() : listed<figure>() {}
	virtual ~figure() {}

	virtual unsigned short getId() = 0;
	virtual point toConsider() = 0;
	virtual double size() {return 0;}
};




		


class edge : public figure {
private:
	point pt;
	unsigned short id;
public:

	edge(point& p, unsigned short i) : figure(), pt(p) , id(i) {}
	edge(int& x, int& y, unsigned short i) :  figure(), id(i) {
		pt.x = x; pt.y = y;
	}

	point toConsider() {;return pt;}
	unsigned short getId() {return id;}
};



struct LinePair : public figure {
	virtual point pt1() = 0;
	virtual point pt2() = 0;
	virtual direction getDirection() = 0;
};




class horLinePair : public LinePair {
	point p1;
	int p2x;
	//point center;
public:
	horLinePair(point p1, int p2x)  {
		this->p1 = p1;
		this->p2x = p2x;
		//center = point((p1.x+p2x)/2, p1.y);

	}

	~horLinePair() {}

	horLinePair(point& p1, point& p2)  {
		this->p1 = p1;
		this->p2x = p2.x;
		//center = point((p1.x+p2.x)/2, p1.y);
	}

	unsigned short getId() {return (unsigned short)p1.y;}
	point toConsider() {return point((p1.x+p2x)/2, p1.y);}
	point pt1() {return p1;}
	point pt2() {return point(p2x, p1.y);}
	double size() {return (double) (p2x - p1.x);}
	direction getDirection() {return direction(1,0);}


};



struct extLinePair : public LinePair {

	point p1, p2;
	//point center;
	unsigned short id;

	extLinePair(point& p1, point& p2, unsigned short id = 0)  {
		this->p1 = p1;
		this->p2 = p2;
		this->id = id;
		//center = mil(p1, p2);
	}

	extLinePair()  {
		this->p1 = point();
		this->p2 = point();
		this->id = 0;
		//center = point();
	}

	point pt1() {return p1;}
	point pt2() {return p2;}
	double size() {return (p2 - p1).abs();}
	point toConsider() {return mil(p1, p2);}
	unsigned short getId() {return id;}
	direction getDirection() {
		return direction((double)(p2.x - p1.x),(double)(p2.y - p1.y));
	}


};


// ----------------------------------------------

struct LinePair2 : public listed<LinePair2> {


	point p1, p2;

	LinePair2(point& p1, point& p2) : listed<LinePair2>() {
		this->p1 = p1;
		this->p2 = p2;
	}

	
	/** defines the default order for LinePairs.
	order is lexigographical in v1 (v1.y,v1.x).
	*/

	bool operator <(const LinePair2& other){
		return p1.y < other.p1.y || 
			   p1.y == other.p1.y && p1.x < other.p1.y ;
	}

	virtual colorClass getColor() {return noColor;}
	virtual double getLength() {return (p2 - p1).abs();}

};




class LinePair3 : public LinePair2 {
private:
	double length;
	bool updated;

public:

	void setP1(point& p) {
		updated = false;
		p1 = p;
	}
	void setP2(point& p) {
		updated = false;
		p2 = p;
	}


	LinePair3(point& p1, point& p2) : LinePair2(p1, p2), length(0), updated(false) {}

	void updateLength() { 
		length = (p2 - p1).abs();
		updated = true;
	}
	double getLength() {
		if (!updated) updateLength();
		return length;
	}

};


struct coloredLinePair2 : public LinePair2 {
private :
	colorClass color;
public:


	coloredLinePair2(point& v1,point& v2, colorClass c = noColor) :
		LinePair2(v1, v2), color(c) {}

	colorClass getColor() {return color;}
};





//---------------------------------------------------------------




/* adds a new scan line between two line pairs */

/*
inline bool DoubleScanLinesNumber(slist<figure>& lst, MSH2004EdgeDetection& edgeScanner) {

	if (lst.getSize()<2) return false;

	edgeScanner.threshold = 12;

	int stepY;

	slist<figure>::iterator it(lst.front());

	unsigned int c = 0;

	do {

		LinePair *lp = (LinePair *)it.get();
		LinePair *next = (LinePair *)lp->getNext();
		int f1x = lp->pt1().x,
			f2x = lp->pt2().x;
		int n1x = next->pt1().x,
			n2x = next->pt2().x;
		int n1y = next->pt1().y,
			f1y = lp->pt1().y;

		stepY = (int)(n1y - f1y)/2;

		//	OUTPUT(idText,text, "step " << stepY);

		if (n1x<f2x && n2x>f1x 
			//&& n2x-n1x <20 /
			&& n1y!=f1y && stepY>0) {

			int leftCorner = (f1x<n1x) ? f1x : n1x, 
				rightCorner = (f2x>n2x) ? f2x : n2x;

			point center ((int)(leftCorner+rightCorner)/2, f1y+stepY);

			DOT(imageProcessor_ground, center.x,center.y,Drawings::red, Drawings::white);


			point left = center, right = center;

			edgeScanner.scanWest(left.x,left.y);
			edgeScanner.scanEast(right.x,right.y);
		
			extLinePair* lp = new extLinePair(left,right,0);

			LINE(imageProcessor_ground,
					lp->pt1().x,lp->pt1().y, lp->pt2().x,lp->pt2().y, 
					0.5, Drawings::ps_solid, Drawings::red);

			if ((lp->pt2().x - lp->pt1().x)<=3*(f2x-f1x)) {
				it.get()->insert(lp);++c;
			}

		}



	} while (++it && !it.back());

	lst.incSize(c);
	return true;

};

*/



//-----------------------------------------------------------------------------------------

inline figure* getMin (figure * lp) {
	
	figure * min = lp->getNext();

	if (!min) return 0;

	slist<figure>::iterator it = min;

	double d_min = Geometry::distance(lp->toConsider(), it->toConsider());

	unsigned short cur_id = min->getId();

	while(++it && cur_id == it->getId()) {
		
		double d = Geometry::distance(lp->toConsider(),it->toConsider());

		if (d <d_min) {
			d_min = d;
			min = *it;
		}
	} 

	return min;
}


/* join the nearest LinePairs in order to make the interpretation of the shape possible */

inline void createLinearSegment(slist<figure>& lst) {

	if (lst.getSize()<3)return;
	slist<figure>::iterator start = lst.front();

	do {
		figure* min = getMin(*start);
		if (min != *start) start->swapNext(min);
	} while (++start && start->getNext());

	lst.setLast(*start);
}


//---------------------------------------------------------------------------




inline Vector2<double> mil(LinePair2& lp) {
	return Vector2<double> ((lp.p1.x+lp.p2.x)/2, (lp.p1.y+lp.p2.y)/2);
};


inline Geometry::Line makeLine(const LinePair2* lp) {
	Geometry::Line l;

	l.base = Vector2<double> ((double)lp->p1.x, (double)lp->p1.y);
	l.direction = Vector2<double> ((double)lp->p2.x, (double)lp->p2.y) - l.base;

	return l;
};

inline double theta2(LinePair2* l1, LinePair2* l2) {

	double angle1 = theta2(l1->p1,l1->p2);
	double angle = theta2(l2->p1, l2->p2) - angle1;

	while (angle<0) angle+=360;

	return angle;

}


#endif

/*
 * Change log :
 * $Log: SegmentationTools.h,v $
 * Revision 1.14  2004/06/17 14:18:22  kerdels
 * fixed bug in rfieldspecialist, changed threshold for bridge-detection
 *
 * Revision 1.13  2004/06/17 11:55:31  serwe
 * added changelog
 *
 */
