/**
* @file MSH2004ImageProcessor.cpp
*
* Implementation of class MSH2004ImageProcessor
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
* @author <a href="mailto:sadprofessor@web.de">Bernd Schmidt</a>
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
*/

#include "MSH2004ImageProcessor.h"
#include "Tools/Math/Common.h"
#include "Tools/Debugging/GenericDebugData.h"
#include "Tools/FieldDimensions.h"
#include "Tools/RingBuffer.h"
#include "Tools/Location.h"

#define PLOT(p,c) COMPLEX_DRAWING(imageProcessor_general,plot(p,c))

const int Y = 0,
          U = cameraResolutionWidth_ERS7, /**< Relative offset of U component. */
          V = 2 * cameraResolutionWidth_ERS7; /**< Relative offset of V component. */


// This is a hack to speed up color table access. 
#define COLOR_CLASS(y,u,v) ((ColorTable32K&) colorTable).getColorClassFast(y, u, v)
//~ #define COLOR_CLASS(y,u,v) (colorClass)((ColorTable32K&) colorTable).colorClassesUnpacked[((y>>4)<<12)|((v>>2)<<6)|(u>>2)]

MSH2004ImageProcessor::MSH2004ImageProcessor(const ImageProcessorInterfaces& interfaces)
: ImageProcessor(interfaces), advPixFilter(colorTable), null(10000, 10000)
{
  yThreshold = 15;
  vThreshold = 8;
  edgeScanner = new GT2004EdgeDetection(*this,corrector);
  //~ InBinaryFile stream(getLocation().getFilename("ball.thr"));

	//~ if(stream.exists()){ 
		//~ stream >>  ballThreshold;
	//~ }
    //~ else{ 
		//~ ballThreshold = 30;
	//~ };

	//~ OUTPUT(idText,text,"ball-thr: " << ballThreshold);
}

MSH2004ImageProcessor::Box::Box(int minX,int minY,int maxX,int maxY):
	minX(minX),maxX(maxX),minY(minY),maxY(maxY)
{
		
}

void MSH2004ImageProcessor::execute()
{
  INIT_DEBUG_IMAGE(imageProcessorPlayers, image);

  xFactor = yFactor = image.cameraInfo.focalLength;

  advPixFilter.reset();
  

  numberOfScannedPixels = 0;
  // reset everything
  landmarksPercept.reset(image.frameNumber);
  linesPercept.reset(image.frameNumber);
  ballPercept.reset(image.frameNumber);
  playersPercept.reset(image.frameNumber);
  obstaclesPercept.reset(image.frameNumber);
  flagSpecialist.init(image);

  // variations of the camera matrix for different object heights
  cmBall = cameraMatrix;
  cmBall.translation.z -= ballRadius;
  cmTricot = cameraMatrix;
  cmTricot.translation.z -= 100; // upper tricot border



  Vector3<double> vectorInDirectionOfViewCamera(1,0,0);
  Vector3<double> vectorInDirectionOfViewWorld;
  vectorInDirectionOfViewWorld = cameraMatrix.rotation * vectorInDirectionOfViewCamera;

  angleBetweenDirectionOfViewAndGround = 
    toDegrees(
    atan2(
    vectorInDirectionOfViewWorld.z,
    sqrt(sqr(vectorInDirectionOfViewWorld.x) + sqr(vectorInDirectionOfViewWorld.y))
    )
    );

//  OUTPUT(idText,text,"W:" << angleBetweenDirectionOfViewAndGround);

  horizon = Geometry::calculateHorizon(cameraMatrix, image.cameraInfo);

  LINE(imageProcessor_horizon,
    (int)(horizon.base.x - 120 * horizon.direction.x), (int)(horizon.base.y - 120 * horizon.direction.y), 
    (int)(horizon.base.x + 120 * horizon.direction.x), (int)(horizon.base.y + 120 * horizon.direction.y), 
    1, Drawings::ps_solid, Drawings::white );
  DEBUG_DRAWING_FINISHED(imageProcessor_horizon);
  
//  obstaclesNearPointOnField2.x = 0; obstaclesNearPointOnField2.y = 0;
//  obstaclesFarPointOnField2IsOnImageBorder = false;

//  obstaclesFarPointOnField1.x = 0; obstaclesFarPointOnField1.y = 0;
//  obstaclesFarPointOnField2.x = 0; obstaclesFarPointOnField2.y = 0;
  scanColumns();
  
/*  if(Geometry::distance(obstaclesFarPointOnField1, obstaclesFarPointOnField2) < 400)
  {
    obstaclesPercept.add(obstaclesNearPointOnField2, obstaclesFarPointOnField2, obstaclesFarPointOnField2IsOnImageBorder);
  }
*/
/*
  if(obstaclesPercept.numberOfPoints < 3)
  {
    obstaclesPercept.numberOfPoints = 0;
  }
*/
  scanRows();
  filterPercepts();

  landmarksPercept.cameraOffset = cameraMatrix.translation;
  flagSpecialist.getFlagPercept(cameraMatrix, image.cameraInfo, horizon, landmarksPercept);

  MSH2004GoalRecognizer goalRecognizer(image, cameraMatrix, colorTable, advPixFilter, obstaclesPercept, landmarksPercept, BORDER);
  goalRecognizer.getGoalPercept(landmarksPercept);

  INFO(printPixelUsage, idText, text, "used pixels: " << 100 * numberOfScannedPixels / (image.cameraInfo.resolutionWidth * image.cameraInfo.resolutionHeight) << " %");

  GENERATE_DEBUG_IMAGE(segmentedImage1,
  colorTable.generateColorClassImage(image, segmentedImage1ColorClassImage));
  SEND_DEBUG_COLOR_CLASS_IMAGE(segmentedImage1);
  
  SEND_DEBUG_IMAGE(imageProcessorPlayers);
  
  DEBUG_DRAWING_FINISHED(imageProcessor_general);
  DEBUG_DRAWING_FINISHED(imageProcessor_ball1);
}

void MSH2004ImageProcessor::scanColumns()
{
  // Reset horizontal counters
  orangeCount = 0;
  noOrangeCount = noRedCount = noBlueCount = 100;
  numberOfBallPoints = 0;
  closestBottom = closestBottom = 40000;
  goalAtBorder = goalAtBorder = false;
  
  const int scanLineSpacing = 3,
            desiredNumberOfScanLines = 68;

  Geometry::Line scanLine,
                 lineAboveHorizon,
                 lineBelowHorizon[2];
  Vector2<double> intersection;
  Vector2<int> topPoint,
               bottomPoint;
  lastFlag = Vector2<int>(-100,0);
  int i;
  for(i = 0; i < 6; ++i)
    flagCount[i] = 0;

  // scan lines are perpendicular to horizon
  scanLine.direction.x = -horizon.direction.y;
  scanLine.direction.y = horizon.direction.x;

  // upper boundary for scan lines
  lineAboveHorizon.direction = horizon.direction;
  lineAboveHorizon.base = horizon.base - scanLine.direction * 50;
    
  // lower boundaries for scan lines
  for(i = 0; i < 2; ++i)
  {
    lineBelowHorizon[i].direction = horizon.direction;
    lineBelowHorizon[i].base = horizon.base + scanLine.direction * 30 * (i + 1);
  }      

  // calculate and scan lines
  for(i = 0; i < desiredNumberOfScanLines; i++)
  {
    scanLine.base.x = image.cameraInfo.resolutionWidth / 2 + (i - desiredNumberOfScanLines / 2) * horizon.direction.x * scanLineSpacing;
    scanLine.base.y = image.cameraInfo.resolutionHeight / 2 + (i - desiredNumberOfScanLines / 2) * horizon.direction.y * scanLineSpacing;
    
    //Does the scan line intersect with the image?
    if(Geometry::getIntersectionPointsOfLineAndRectangle(
                                        Vector2<int>(BORDER,BORDER),
                                        Vector2<int>(image.cameraInfo.resolutionWidth - 1 - BORDER,image.cameraInfo.resolutionHeight - 1 - BORDER),
                                        scanLine, 
                                        topPoint, bottomPoint))
    {
      Geometry::getIntersectionOfLines(lineAboveHorizon, scanLine, intersection);
      
      //Is the bottomPoint below the horizon?
      if(scanLine.direction.y / (intersection.y - bottomPoint.y) < 0)
      {

        //Is the intersection with the horizon point inside the image?
        if(intersection.x < image.cameraInfo.resolutionWidth-BORDER && intersection.x >= BORDER &&
          intersection.y < image.cameraInfo.resolutionHeight-BORDER && intersection.y >= BORDER) 
        {
          topPoint.x = (int)intersection.x;
          topPoint.y = (int)intersection.y;
        }
        int lineType = i & 3; // different lengths of scan lines
        if(lineType)
        {
          Geometry::getIntersectionOfLines(lineBelowHorizon[lineType == 3 ? 0 : lineType - 1], scanLine, intersection);
          //Is the intersection with the horizon point inside the image?
          if(intersection.x < image.cameraInfo.resolutionWidth-BORDER && intersection.x >= BORDER &&
            intersection.y < image.cameraInfo.resolutionHeight-BORDER && intersection.y >= BORDER) 
          {
            bottomPoint.x = (int)intersection.x;
            bottomPoint.y = (int)intersection.y;
            scan(topPoint, bottomPoint, true, true);
          }
        }
        else
          scan(topPoint, bottomPoint, true, false);
      }
    }
  }

  // finish clustering
  for(i = 0; i < 3; ++i)
  {
    clusterBalls(null,null,null,null,false,false);
    clusterRobots(null,false,false);
  }
  clusterFlags(Flag::numberOfFlagTypes,Vector2<int>(1000,0));
}

void MSH2004ImageProcessor::scanRows()
{
  const int scanLineSpacing = 15,
            desiredNumberOfScanLines = 30;
  Geometry::Line scanLine = horizon;

  // start below horizon
  scanLine.base.x += -horizon.direction.y * 10;
  scanLine.base.y += horizon.direction.x * 10;

  for(int i = 1; i < desiredNumberOfScanLines; i++)
  {
    scanLine.base.x += -horizon.direction.y * scanLineSpacing;
    scanLine.base.y += horizon.direction.x * scanLineSpacing;
    
    Vector2<int> topPoint,
                 bottomPoint;
    //Does the scan line intersect with the image?
    if(Geometry::getIntersectionPointsOfLineAndRectangle(
                                        Vector2<int>(BORDER,BORDER),
                                        Vector2<int>(image.cameraInfo.resolutionWidth - 1 - BORDER,image.cameraInfo.resolutionHeight - 1 - BORDER),
                                        scanLine,
                                        topPoint, bottomPoint))
      if(rand() > RAND_MAX / 2)
        scan(topPoint, bottomPoint, false, false);
      else
        scan(bottomPoint, topPoint, false, false);
  }
}

void MSH2004ImageProcessor::scan(const Vector2<int>& start, const Vector2<int>& end,
                                bool vertical, bool noLines)
{
  LinesPercept::LineType typeOfLinesPercept = LinesPercept::field;
  int pixelsSinceGoal = 0;
  int pixelsBetweenGoalAndObstacle = 0;
  
	Vector2<int> actual = start;
  const Vector2<int> diff = end - start,
               step(sgn(diff.x),sgn(diff.y)),
               size(abs(diff.x),abs(diff.y));
  bool isVertical = abs(diff.y) > abs(diff.x);
  int count,
      sum;

  // init Bresenham
  if(isVertical)
  {
    count = size.y;
    sum = size.y / 2;
  }
  else
  {
    count = size.x;
    sum = size.x / 2;
  }

  if(count > 7)
  {
    int numberOfPoints[LinesPercept::numberOfLineTypes],
        i;
    for(i = 0; i < LinesPercept::numberOfLineTypes; ++i)
      numberOfPoints[i] = linesPercept.numberOfPoints[i];
    RingBuffer<Vector2<int>,7> pixel;
    Vector2<int> tmpPix;
    //the image is scanned vertically
    for(i = 0; i < 6; ++i) // fill buffer
    {
      tmpPix.x = actual.x;
			tmpPix.y = actual.y;
			pixel.add(tmpPix);
		
      // Bresenham
      if(isVertical)
      {
        actual.y += step.y;
        sum += size.x;
        if(sum >= size.y)
        {
          sum -= size.y;
          actual.x += step.x;
        }
      }
      else        
      {
        actual.x += step.x;
        sum += size.y;
        if(sum >= size.x)
        {
          sum -= size.x;
          actual.y += step.y;
        }
      }
    }
    lineColor = Drawings::numberOfColors;

    int redCount = 0,
        blueCount = 0,
        greenCount = 0,
        noGreenCount = 100,
        orangeCountVert = 0,
        noOrangeCountVert = 100,
        pinkCount = 0,
        noPinkCount = 100,
        yellowCount = 0,
        noYellowCount = 100,
        skyblueCount = 0,
        noSkyblueCount = 100;
	Vector2<int> firstRed = null,
            lastRed = null,
            firstBlue = null,
            lastBlue = null,
            firstGreen = null,
            firstOrange = null,
            lastOrange = null,
            firstPink = null,
            lastPink = null,
            pFirst = pixel[2],
            up = pFirst;
    enum{justUP, greenAndUP, down} state = justUP;
    bool borderFound = false, // there can be only one border point per scan line
         greenAboveOrange = false,
         greenBelowOrange = false;
    Flag::FlagType flagType = Flag::numberOfFlagTypes; // none

    // obstacles
    enum{noGroundFound, foundBeginOfGround, foundEndOfGround} groundState = noGroundFound;
    Vector2<int> firstGround(BORDER,BORDER);
    int numberOfPixelsSinceFirstOccurrenceOfGroundColor = 0;
    int numberOfPixelsSinceLastGround = 0;
    int numberOfGroundPixels = 0;
    int numberOfWhiteObstaclePixels = 0;
    int numberOfNotWhiteObstaclePixels = 0;
    bool beginOfGroundIsAtTheTopOfTheImage = false;

    for(; i <= count; ++i)
    {
      pixelsSinceGoal++;
      numberOfScannedPixels++;
      tmpPix.x = actual.x;
			tmpPix.y = actual.y;
			pixel.add(tmpPix);

      // Bresenham
      if(isVertical)
      {
        actual.y += step.y;
        sum += size.x;
        if(sum >= size.y)
        {
          sum -= size.y;
          actual.x += step.x;
        }
      }	
      else        
      {
        actual.x += step.x;
        sum += size.y;
        if(sum >= size.x)
        {
          sum -= size.x;
          actual.y += step.y;
        }
      }

      // line point state machine
      const Vector2<int> p3 = pixel[3],
                      p4 = pixel[4];
      //UP: y-component increases ////////////////////////////////////////
			if (getSpectrum(p3,YY) > getSpectrum(p4,YY) + yThreshold)
      {
        up = p3;
        const Vector2<int> p6 = pixel[6];
				if (advPixFilter.getColorClass(image, p6.x, p6.y) == green)
          state = greenAndUP;
        else 
          state = justUP;
      }
    
      //DOWN: y-component decreases //////////////////////////////////////
      else if (getSpectrum(p3,YY) < getSpectrum(p4,YY) - yThreshold || getSpectrum(p3,VV) < getSpectrum(p4,VV) - vThreshold)
      {
        const Vector2<int> p0 = pixel[0];
        // is the pixel in the field ??
        if(state != down && // was an "up" pixel above?
					advPixFilter.getColorClass(image, p0.x, p0.y) == green) // is there green 3 pixels below ?
        {
          const Vector2<int> p6 = pixel[6];
					colorClass c = advPixFilter.getColorClass(image, p6.x, p6.y);
          if(!noLines || c == skyblue || c == yellow)
          {
            const Vector2<int> coords = p3;
            Vector2<int> pointOnField; //position on field, relative to robot

            if(Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField))
            {
              const Vector2<int> upCoords = up;
              int height = (coords - upCoords).abs();
              int distance = (int) sqrt(sqr(cameraMatrix.translation.z) + sqr(pointOnField.abs())),
                lineHeight = Geometry::getSizeByDistance(image.cameraInfo, 25, distance);
              const Vector2<int> p5 = pixel[5];
              //The bright region is assumed to be on the ground. If it is small enough, it is a field line.
              if(height < 3 * lineHeight && state == greenAndUP &&
                 (advPixFilter.getColorClass(image, p5.x, p5.y) == white ||
                  advPixFilter.getColorClass(image, p4.x, p4.y) == white))
              {
                linesPercept.add(LinesPercept::field,pointOnField);
                if(Geometry::calculatePointOnField(upCoords.x, upCoords.y, cameraMatrix, image.cameraInfo, pointOnField))
                  linesPercept.add(LinesPercept::field,pointOnField);
              }
              else if(height >= 3 && !borderFound)
              {
                switch(c)
                {
                  case skyblue:
                    if(vertical)
                    {
                      borderFound = true;
                      if(getPlayer().getTeamColor() == Player::red && !linesPercept.numberOfPoints[LinesPercept::skyblueGoal])
                        goalAtBorder = pointOnField.abs() < closestBottom;
                      closestBottom = 40000;
                      linesPercept.add(LinesPercept::skyblueGoal,pointOnField);
                      typeOfLinesPercept = LinesPercept::skyblueGoal;
                      pixelsSinceGoal = 0;

                      lastRed = lastBlue = null;
                      redCount = blueCount = 0;
                    }
                    break;
                  case yellow:
                    if(vertical)
                    {
                      borderFound = true;
                      if(getPlayer().getTeamColor() == Player::blue && !linesPercept.numberOfPoints[LinesPercept::yellowGoal])
                        goalAtBorder = pointOnField.abs() < closestBottom;
                      closestBottom = 40000;
                      linesPercept.add(LinesPercept::yellowGoal,pointOnField);
                      typeOfLinesPercept = LinesPercept::yellowGoal;
                      pixelsSinceGoal = 0;

                      lastRed = lastBlue = null;
                      redCount = blueCount = 0; 
                    }
                    break;
                  case white:
                    if(state != greenAndUP && height > lineHeight * 3 && (vertical || height > 30))
                    {
                      borderFound = true;
                      linesPercept.add(LinesPercept::border,pointOnField);
                      typeOfLinesPercept = LinesPercept::border;
                      lastRed = lastBlue = null;
                      redCount = blueCount = 0; 
                    }
                    break;
                }
              }
            }
          }
          state = down;
        }
      }
      if(vertical)
      {
        // in scanColumns, recognize player and ball points
        colorClass c3 = advPixFilter.getColorClass(image, p3.x, p3.y);
        ++noOrangeCountVert;
        ++noGreenCount;
        ++noPinkCount;
        ++noYellowCount;
        ++noSkyblueCount;
        switch(c3)
        {
          case blue: // count blue, remember last
            if(blueCount == 3)
              firstBlue = pixel[6];
            ++blueCount;
            lastBlue = p3;
            firstGreen = null;
            break;
          case gray: // drop green above gray (i.e. robot legs)
            if(firstGreen!=null && noGreenCount < 8)
              firstGreen = null;
            break;
          case green: // find first green below a robot or a ball
            if(firstGreen==null)
            { 
              firstGreen = p3;
              // check if this green is below a ball
              if(orangeCountVert > 2 && noOrangeCountVert < 6)
                greenBelowOrange = true;
            }
            if(noGreenCount > 6)
              greenCount = 1;
            else
              ++greenCount;
            if(greenCount == 2 && noPinkCount < 5 && pinkCount >= 2)
              flagType = Flag::pinkAboveGreen;
            noGreenCount = 0;
            break;
          case red:
            if(orangeCountVert < 6 || noOrangeCountVert > 4 || redCount > orangeCountVert)
            { // count red, remember last
              if(redCount == 3)
                firstRed = pixel[6];
              ++redCount;
              lastRed = p3;
              firstGreen = null;
              break;
            }
            // no break, red below orange is interpreted as orange
          case orange: // find beginning and end of a ball
            if(noOrangeCountVert > 6)
            {
              orangeCountVert = 1;
              firstOrange = p3;
              lastOrange = p3;
              greenAboveOrange = noGreenCount < 6;
            }
            else
            {
              ++orangeCountVert;
              if(orangeCountVert > 2) // ignore robot if ball is below, distance would be wrong
              {
                lastRed = lastBlue = null;
                redCount = blueCount = 0; 
              }
              lastOrange = p3;
            }
            firstGreen = null;
            noOrangeCountVert = 0;
            greenBelowOrange = false;
            break;
          case pink:
            if(noPinkCount > 6)
            {
              pinkCount = 1;
              firstPink = p3;
            }
            else
              ++pinkCount;
            if(pinkCount == 2)
            {
              if(noYellowCount < 5 && yellowCount >= 2)
                flagType = Flag::yellowAbovePink;
              else if(noSkyblueCount < 5 && skyblueCount >= 2)
                flagType = Flag::skyblueAbovePink;
              else if(noGreenCount < 5 && greenCount >= 2)
                flagType = Flag::greenAbovePink;
            }
            lastPink = p3;
            noPinkCount = 0;
            break;
          case yellow:
            if(noYellowCount > 6)
              yellowCount = 1;
            else
              ++yellowCount;
            if(yellowCount == 2 && noPinkCount < 5 && pinkCount >= 2)
              flagType = Flag::pinkAboveYellow;
            noYellowCount = 0;
            break;
          case skyblue:
            if(noSkyblueCount > 6)
              skyblueCount = 1;
            else
              ++skyblueCount;
            if(skyblueCount == 2 && noPinkCount < 5 && pinkCount >= 2)
              flagType = Flag::pinkAboveSkyblue;
            noSkyblueCount = 0;
            break;
          case white:
            switch(flagType)
            {
              case Flag::pinkAboveYellow:
                if(noYellowCount < 5)
                  clusterFlags(flagType, (lastPink + firstPink) / 2);
                flagType = Flag::numberOfFlagTypes; // none
                break;
              case Flag::pinkAboveSkyblue:
                if(noSkyblueCount < 5)
                  clusterFlags(flagType, (lastPink + firstPink) / 2);
                flagType = Flag::numberOfFlagTypes; // none
                break;
              case Flag::yellowAbovePink:
                if(noPinkCount < 5)
                  clusterFlags(flagType, (lastPink + firstPink) / 2);
                flagType = Flag::numberOfFlagTypes; // none
                break;
              case Flag::skyblueAbovePink:
                if(noPinkCount < 5)
                  clusterFlags(flagType, (lastPink + firstPink) / 2);
                flagType = Flag::numberOfFlagTypes; // none
            }
            break;
          
        }//switch(c3)

        // obstacles state machine
        if(count > 90 && cameraMatrix.isValid)
        {
					DEBUG_IMAGE_SET_PIXEL_BLACK(imageProcessorPlayers, pixel[0].x, pixel[0].y);
					colorClass color = advPixFilter.getColorClass(image, pixel[0].x, pixel[0].y);
          if(groundState == noGroundFound)
          {
            DEBUG_IMAGE_SET_PIXEL_YELLOW(imageProcessorPlayers, pixel[0].x, pixel[0].y);
            if(color == green || color == orange)
            {
              if(numberOfPixelsSinceFirstOccurrenceOfGroundColor == 0)
              {
                firstGround = pixel[0];
              }
              numberOfPixelsSinceFirstOccurrenceOfGroundColor++;
              if(numberOfPixelsSinceFirstOccurrenceOfGroundColor > 3 || color == orange)
              {
                Vector2<int> coords = pixel[0];
                int lineHeight = Geometry::calculateLineSize(coords, cameraMatrix, image.cameraInfo);
                if(i < lineHeight * 4) beginOfGroundIsAtTheTopOfTheImage = true;
                groundState = foundBeginOfGround;
                pixelsBetweenGoalAndObstacle = pixelsSinceGoal;
              }
            }
            else
            {
              numberOfPixelsSinceFirstOccurrenceOfGroundColor = 0;
              if(color == white) numberOfWhiteObstaclePixels++;
              else numberOfNotWhiteObstaclePixels++;
            }
          }
          else if(groundState == foundBeginOfGround)
          {
            DEBUG_IMAGE_SET_PIXEL_GREEN(imageProcessorPlayers, pixel[0].x, pixel[0].y);
            if(color != green && color != orange)
            {
              groundState = foundEndOfGround;
              numberOfPixelsSinceLastGround = 0;
              numberOfPixelsSinceFirstOccurrenceOfGroundColor = 0;
            }
            else
            {
              numberOfGroundPixels++;
            }
          }
          else if(groundState == foundEndOfGround)
          {
            numberOfPixelsSinceLastGround++;
            DEBUG_IMAGE_SET_PIXEL_RED(imageProcessorPlayers, pixel[0].x, pixel[0].y);
            if(color == green || color == orange)
            {
              numberOfPixelsSinceFirstOccurrenceOfGroundColor++;
              if(numberOfPixelsSinceFirstOccurrenceOfGroundColor > 3 || color == orange)
              {
                groundState = foundBeginOfGround;
//                pixelsBetweenGoalAndObstacle = pixelsSinceGoal;
                Vector2<int> coords = pixel[0],
                             pointOnField; //position on field, relative to robot
                int lineHeight = Geometry::calculateLineSize(coords, cameraMatrix, image.cameraInfo);
                if(numberOfPixelsSinceLastGround > lineHeight * 4)
                {
                  firstGround = pixel[0];
                  numberOfGroundPixels = 0;
                  beginOfGroundIsAtTheTopOfTheImage = false;
                }
              }
            }
            else
            {
              numberOfPixelsSinceFirstOccurrenceOfGroundColor = 0;  
              if(color == white) numberOfWhiteObstaclePixels++;
              else numberOfNotWhiteObstaclePixels++;
            }
          }
        } // if(count > 100)

        PLOT(p3,ColorClasses::colorClassToDrawingsColor(c3));
      }//if(vertical)
    }

    Vector2<int> pLast = pixel[3];
    PLOT(pLast,Drawings::numberOfColors);

    if(vertical)
    { // line scanning finished, analyze results (only in scanColumns)
      int goal = getPlayer().getTeamColor() == Player::red ? LinesPercept::skyblueGoal
                                                           : LinesPercept::yellowGoal;
      if(linesPercept.numberOfPoints[goal] == numberOfPoints[goal]) // no goal point found in this column
      {
        const Vector2<int> coords = pLast;
        Vector2<int> pointOnField;
        if(Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField))
        {
          int dist = pointOnField.abs();
          if(dist < closestBottom)
            closestBottom = dist;
        }
      }
              
      // ball points found?
      if(orangeCountVert > 2 && (greenAboveOrange || greenBelowOrange || !noLines && noOrangeCountVert <= 5) || 
         orangeCountVert > 20)
      { 
        if(!greenAboveOrange) // drop border and goal points
          for(i = 1; i < LinesPercept::numberOfLineTypes; ++i)
            linesPercept.numberOfPoints[i] = numberOfPoints[i];
        if(noOrangeCountVert <= 5)
          pLast = lastOrange;
        clusterBalls(pFirst,pLast,firstOrange,lastOrange,greenAboveOrange,greenBelowOrange);
      }
      else
        clusterBalls(pFirst,pLast,null,null,greenAboveOrange,greenBelowOrange);

      bool redFound = false,
           blueFound = false;

      // red robot point found?
      if(redCount > blueCount && (firstGreen!=null && redCount > 2 || redCount > 20))
      { // drop border and goal points
        for(i = 1; i < LinesPercept::numberOfLineTypes; ++i)
          linesPercept.numberOfPoints[i] = numberOfPoints[i];
        if(redCount > 20)
        { // if robot is close, use upper boundary of tricot to calculate distance
          Vector2<int> coords = firstRed,
                       pointOnField; //position on field, relative to robot
          if(Geometry::calculatePointOnField(coords.x, coords.y, cmTricot, image.cameraInfo, pointOnField) &&
             pointOnField.abs() < 500)
          {
            DOT(imageProcessor_general, coords.x, coords.y, Drawings::red, Drawings::white);
            linesPercept.add(LinesPercept::redRobot,pointOnField);
            typeOfLinesPercept = LinesPercept::redRobot;
            redFound = true;
          }
        }
        if(!redFound)
        { // otherwise, use foot point
          Vector2<int> coords = firstGreen,
                       pointOnField; //position on field, relative to robot
          if(Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField))
          {
            Vector2<int> redCoords = lastRed;
            if((redCoords - coords).abs() < Geometry::getSizeByDistance(image.cameraInfo, 100, pointOnField.abs()))
            {
              linesPercept.add(LinesPercept::redRobot,pointOnField);
              typeOfLinesPercept = LinesPercept::redRobot;
              redFound = true;
            }
          }
        }
      }
      // blue robot point found?
      else if(blueCount > redCount && (firstGreen!=null && blueCount > 2 || blueCount > 10))
      { // drop border and goal points
        for(i = 1; i < LinesPercept::numberOfLineTypes; ++i)
          linesPercept.numberOfPoints[i] = numberOfPoints[i];
        if(blueCount > 10)
        { // if robot is close, use upper boundary of tricot to calculate distance
          Vector2<int> coords = firstBlue,
                       pointOnField; //position on field, relative to robot
          if(Geometry::calculatePointOnField(coords.x, coords.y, cmTricot, image.cameraInfo, pointOnField) &&
             pointOnField.abs() < 500)
          {
            DOT(imageProcessor_general, coords.x, coords.y, Drawings::pink, Drawings::white);
            linesPercept.add(LinesPercept::blueRobot,pointOnField);
            typeOfLinesPercept = LinesPercept::blueRobot;
            blueFound = true;
          }
        }
        if(!blueFound)
        { // otherwise, use foot point
          Vector2<int> coords = firstGreen,
                       pointOnField; //position on field, relative to robot
          if(Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField))
          {
            Vector2<int> blueCoords = lastBlue;
            if((blueCoords - coords).abs() < Geometry::getSizeByDistance(image.cameraInfo, 100, pointOnField.abs()))
            {
              linesPercept.add(LinesPercept::blueRobot,pointOnField);
              typeOfLinesPercept = LinesPercept::blueRobot;
              blueFound = true;
            }
          }
        }
      }
      // cluster the robot points found
      clusterRobots(pLast, redFound, blueFound);

      // obstacles found?
      bool addObstaclesPoint = false;
      Vector2<int> newObstaclesNearPoint;
      Vector2<int> newObstaclesFarPoint;
      bool newObstaclesFarPointIsOnImageBorder = false;
      if(count > 90 && cameraMatrix.isValid &&
        (
         numberOfGroundPixels > numberOfWhiteObstaclePixels ||
         numberOfGroundPixels > 20 ||
         numberOfNotWhiteObstaclePixels > numberOfWhiteObstaclePixels * 2
         )
        )
      {
        Vector2<int> firstGroundOnField;
        bool firstGroundOnFieldIsOnField = 
          Geometry::calculatePointOnField(firstGround.x, firstGround.y, cameraMatrix, image.cameraInfo, firstGroundOnField);

        Vector2<int> imageBottom = pixel[0];
        Vector2<int> imageBottomOnField;
        bool imageBottomIsOnField = 
          Geometry::calculatePointOnField(imageBottom.x, imageBottom.y, cameraMatrix, image.cameraInfo, imageBottomOnField);

        if(groundState == foundBeginOfGround)
        { 
          if(firstGroundOnFieldIsOnField)
          {
            addObstaclesPoint = true;
            newObstaclesNearPoint = imageBottomOnField;
            newObstaclesFarPoint = firstGroundOnField;
            newObstaclesFarPointIsOnImageBorder = beginOfGroundIsAtTheTopOfTheImage;
          }
        }
        else if(groundState == foundEndOfGround)
        {
          int lineHeight = Geometry::calculateLineSize(imageBottom, cameraMatrix, image.cameraInfo);
          if(numberOfPixelsSinceLastGround < lineHeight * 4)
          {
            if(firstGroundOnFieldIsOnField)
            {
              addObstaclesPoint = true;
              newObstaclesNearPoint = imageBottomOnField;
              newObstaclesFarPoint = firstGroundOnField;
              newObstaclesFarPointIsOnImageBorder = beginOfGroundIsAtTheTopOfTheImage;
            }
          }
          else if(imageBottomIsOnField)
          {
            addObstaclesPoint = true;
            newObstaclesNearPoint = imageBottomOnField;
            newObstaclesFarPoint = imageBottomOnField;
          }
        }
        else if(groundState == noGroundFound)
        {
          if(imageBottomIsOnField &&
            // the carpet often is not green under the robot
            imageBottomOnField.abs() > 150)
          {
            addObstaclesPoint = true;
            newObstaclesNearPoint = imageBottomOnField;
            newObstaclesFarPoint = imageBottomOnField;
          }
        }
      }// if(count > 90 && cameraMatrix.isValid)
      if(addObstaclesPoint)
      {
        if(
/*          (
          Geometry::distance(obstaclesFarPointOnField1, obstaclesFarPointOnField2) < 400 ||
          Geometry::distance(obstaclesFarPointOnField2, newObstaclesFarPoint) < 400
          ) && 
*/
          angleBetweenDirectionOfViewAndGround > -80.0 &&
          !(newObstaclesFarPoint.x == 0 && newObstaclesFarPoint.y == 0)
          )

        {
          ObstaclesPercept::ObstacleType obstacleType = ObstaclesPercept::unknown;
          switch(typeOfLinesPercept)
          {
          case LinesPercept::skyblueGoal: 
            if(pixelsBetweenGoalAndObstacle < 15) 
              obstacleType = ObstaclesPercept::goal; 
            break;
          case LinesPercept::yellowGoal: 
            if(pixelsBetweenGoalAndObstacle < 15) 
              obstacleType = ObstaclesPercept::goal; 
            break;
          case LinesPercept::blueRobot:
            if(getPlayer().getTeamColor() == Player::blue)
              obstacleType = ObstaclesPercept::teammate;
            else 
              obstacleType = ObstaclesPercept::opponent;
            break;
          case LinesPercept::redRobot: 
            if(getPlayer().getTeamColor() == Player::red) 
              obstacleType = ObstaclesPercept::teammate;
            else 
              obstacleType = ObstaclesPercept::opponent;
            break;
          }
          obstaclesPercept.add(
            newObstaclesNearPoint, 
            newObstaclesFarPoint,
            newObstaclesFarPointIsOnImageBorder,
            obstacleType
            );
        }

/*        obstaclesFarPointOnField1 = obstaclesFarPointOnField2;
        obstaclesFarPointOnField2 = newObstaclesFarPoint;
    
        obstaclesFarPointOnField2IsOnImageBorder = newObstaclesFarPointIsOnImageBorder;
        obstaclesNearPointOnField2 = newObstaclesNearPoint;
        */
      }
    }//if(vertical)
  }
}

void MSH2004ImageProcessor::clusterBalls(const Vector2<int>& topPoint,
                                        const Vector2<int>& bottomPoint,
                                        const Vector2<int>& first,
                                        const Vector2<int>& last,
                                        bool greenAboveOrange,
                                        bool greenBelowOrange)
{
  if(first!=null) // ball found
  {
    if(first != topPoint) // not at image border
    {
      (Vector2<int>&) ballPoints[numberOfBallPoints] = first;
      DOT(imageProcessor_general,ballPoints[numberOfBallPoints].x,
                              ballPoints[numberOfBallPoints].y,
                              Drawings::orange,Drawings::white);
      ballPoints[numberOfBallPoints].isBottom = false;
      ballPoints[numberOfBallPoints++].greenIsClose = greenAboveOrange;
    }
    if(last != bottomPoint) // not at image border
    {
      (Vector2<int>&) ballPoints[numberOfBallPoints] = last;
      DOT(imageProcessor_general,ballPoints[numberOfBallPoints].x,
                              ballPoints[numberOfBallPoints].y,
                              Drawings::orange,Drawings::white);
      ballPoints[numberOfBallPoints].isBottom = true;
      ballPoints[numberOfBallPoints++].greenIsClose = greenBelowOrange;
    }
    if(!orangeCount)
      firstBall = (first + last) / 2;
    lastBall = (first + last) / 2;
    ++orangeCount;
    noOrangeCount = 0;
  }
  else // no ball found
  {
    Vector2<int> coords = bottomPoint,
                 pointOnField,
                 ballOnField;
    // count number of columns without ball points, but ignore columns that are
    // not long enough to contain the ball
    if(noOrangeCount > 1 || bottomPoint==null || !numberOfBallPoints ||
       (Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField) &&
        Geometry::calculatePointOnField(ballPoints[numberOfBallPoints - 1].x, 
                                        ballPoints[numberOfBallPoints - 1].y,
                                        cameraMatrix, image.cameraInfo, ballOnField) &&
        ballOnField.abs() > pointOnField.abs()))
    {
      if(++noOrangeCount == 2 && // ignore small balls close to but not below red robots
         (!numberOfBallPoints || numberOfBallPoints > 2 || noRedCount > 3 || 
          !Geometry::calculatePointOnField(ballPoints[numberOfBallPoints - 1].x, 
                                           ballPoints[numberOfBallPoints - 1].y,
                                           cameraMatrix, image.cameraInfo, ballOnField) ||
           closestRed.abs() > ballOnField.abs()))
      {
        createBallPercept();
        orangeCount = 0;
        numberOfBallPoints = 0;
      }
    }
  }
}

void MSH2004ImageProcessor::createBallPercept()
{
  //~ filterBallPoints();
  Box box(firstBall.x,firstBall.y,lastBall.x,lastBall.y);
	LINE(imageProcessor_ball1,
	box.minX,box.minY, 
	box.maxX,box.minY, 
	2, Drawings::ps_solid, Drawings::blue);

	LINE(imageProcessor_ball1,
	box.maxX,box.minY,  
	box.maxX,box.maxY, 
	2, Drawings::ps_solid, Drawings::blue); 

	LINE(imageProcessor_ball1,
	box.maxX,box.maxY,
	box.minX,box.maxY, 
	2, Drawings::ps_solid, Drawings::blue); 

	LINE(imageProcessor_ball1,
	box.minX,box.maxY, 
	box.minX,box.minY, 
	2, Drawings::ps_solid, Drawings::blue); 
	
	edgeScanner->threshold = 32;
    Geometry::Circle circle;
	validity = calculateLargeCircle(box,circle);
	addBallPercept(circle,validity);

}

void MSH2004ImageProcessor::filterBallPoints()
{
  // Filter top points without a corresponding bottom point that are above
  // top points with a corresponding bottom point. This removes outliers that
  // lack a bottom point because the scan line ended too early.
  int i;
  for(i = 0; i < numberOfBallPoints; ++i)
    if(!ballPoints[i].isBottom && // top point without corresponding bottom point
       (i == numberOfBallPoints - 1 || !ballPoints[i + 1].isBottom))
    {
      Vector2<int> pointOnField;
      if(Geometry::calculatePointOnField(ballPoints[i].x, ballPoints[i].y,
                                         cameraMatrix, image.cameraInfo, pointOnField))
      {
        int count = 0,
            dist = pointOnField.abs();
        for(int j = 0; j < numberOfBallPoints; ++j)
          if(!ballPoints[j].isBottom && // top point _with_ corresponding bottom point
             j < numberOfBallPoints - 1 && ballPoints[j + 1].isBottom &&
             Geometry::calculatePointOnField(ballPoints[j].x, ballPoints[j].y,
                                             cameraMatrix, image.cameraInfo, pointOnField) &&
             dist > pointOnField.abs())
            ++count;
        if(count > 1)
        {
          --orangeCount;
          --numberOfBallPoints;
          for(int j = i; j < numberOfBallPoints; ++j)
            ballPoints[j] = ballPoints[j + 1];
          --i;
        }
      }
    }          

  // Filter bottom points that are above top points. They must be outliers.
  for(i = 0; i < numberOfBallPoints; ++i)
    if(ballPoints[i].isBottom)
    {
      Vector2<int> pointOnField;
      if(Geometry::calculatePointOnField(ballPoints[i].x, ballPoints[i].y,
                                         cameraMatrix, image.cameraInfo, pointOnField))
      {
        int count = 0,
            dist = pointOnField.abs();
        for(int j = 0; j < numberOfBallPoints; ++j)
          if(!ballPoints[j].isBottom && 
             Geometry::calculatePointOnField(ballPoints[j].x, ballPoints[j].y,
                                             cameraMatrix, image.cameraInfo, pointOnField) &&
             dist > pointOnField.abs())
            ++count;
        if(count > 1)
        {
          --orangeCount;
          --numberOfBallPoints;
          if(i > 0 && !ballPoints[i - 1].isBottom)
          {
            --numberOfBallPoints;
            --i;
          }
          for(int j = i; j < numberOfBallPoints; ++j)
            ballPoints[j] = ballPoints[j + 1];
          --i;
        }
      }
    }          

  // Filter top points that are below bottom points. They must be outliers.
  for(i = 0; i < numberOfBallPoints; ++i)
    if(!ballPoints[i].isBottom)
    {
      Vector2<int> pointOnField;
      if(Geometry::calculatePointOnField(ballPoints[i].x, ballPoints[i].y,
                                         cameraMatrix, image.cameraInfo, pointOnField))
      {
        int count = 0,
            dist = pointOnField.abs();
        for(int j = 0; j < numberOfBallPoints; ++j)
          if(ballPoints[j].isBottom && 
             Geometry::calculatePointOnField(ballPoints[j].x, ballPoints[j].y,
                                             cameraMatrix, image.cameraInfo, pointOnField) &&
             dist < pointOnField.abs())
            ++count;
        if(count > 1)
        {
          --numberOfBallPoints;
          --orangeCount;
          if(i < numberOfBallPoints - 1 && ballPoints[i + 1].isBottom)
          {
            --numberOfBallPoints;
            for(int j = i; j < numberOfBallPoints; ++j)
              ballPoints[j] = ballPoints[j + 2];
          }
          else
            for(int j = i; j < numberOfBallPoints; ++j)
              ballPoints[j] = ballPoints[j + 1];
          --i;
        }
      }
    }          

  // Remove all points if no point is below the horizon
  int numberOfPointsBelowHorizon = 0;
  for(i = 0; i < numberOfBallPoints; ++i)
  {
    Vector2<int> pointOnField;
    if(Geometry::calculatePointOnField(ballPoints[i].x, ballPoints[i].y, cameraMatrix, image.cameraInfo, pointOnField))
      numberOfPointsBelowHorizon++;
  }
  if(numberOfPointsBelowHorizon == 0)
    numberOfBallPoints = 0;
}

bool MSH2004ImageProcessor::createBallPerceptFrom3Points(bool onlyIfGreenIsClose)
{
  int point1, point2, point3;
  if(select3Points(point1, point2, point3, onlyIfGreenIsClose))
  {
    Vector2<int> center = 
      cutMiddlePerpendiculars(ballPoints[point1], ballPoints[point2], ballPoints[point3]);
    double ballRadiusInImage = (center - ballPoints[point1]).abs() + 1.0; // typically, 1 pixel of border is lost in the color classification process

    Geometry::Circle circle;
    circle.center.x = center.x;
    circle.center.y = center.y;
    circle.radius = ballRadiusInImage;
    if(checkOrangeInsideBall(circle))
      addBallPercept(center.x, center.y, ballRadiusInImage);
    return true;
  }
  return false;
}

bool MSH2004ImageProcessor::select3Points(int& point1, int& point2, int& point3,
                                         bool onlyIfGreenIsClose) const
{
  double maxDist = 0;
  int i;

  // first select the two points with largest distance from each other
  for(i = 0; i < numberOfBallPoints - 1; ++i)
    if(!onlyIfGreenIsClose || ballPoints[i].greenIsClose)
      for(int j = i + 1; j < numberOfBallPoints ; ++j)
        if(!onlyIfGreenIsClose || ballPoints[j].greenIsClose)
        {
          double dist = Vector2<double>(ballPoints[i].x - ballPoints[j].x,
                                        ballPoints[i].y - ballPoints[j].y).abs();
          if(dist > maxDist)
          {
            maxDist = dist;
            point1 = i;
            point2 = j;
          }
        }
  if(maxDist > 0)
  {
    // then select a third point that is farthest away from the other two.
    // point 3 must form a curve with points 1 and 2, so the sum of both distances
    // must be larger then the distance between the first two points.
    maxDist += 0.1;
    point3 = -1;
    for(i = 0; i < numberOfBallPoints; ++i)
      if((!onlyIfGreenIsClose || ballPoints[i].greenIsClose) && i != point1 && i != point2)
      {
        double dist = Vector2<double>(ballPoints[i].x - ballPoints[point1].x,
                                      ballPoints[i].y - ballPoints[point1].y).abs() +
                      Vector2<double>(ballPoints[i].x - ballPoints[point2].x,
                                      ballPoints[i].y - ballPoints[point2].y).abs();
        if(dist > maxDist)
        {
          maxDist = dist;
          point3 = i;
        }
      }
    return point3 != -1;
  }
  else
    return false;
}

Vector2<int> MSH2004ImageProcessor::cutMiddlePerpendiculars(const Vector2<int>& v1,
                                                           const Vector2<int>& v2,
                                                           const Vector2<int>& v3) const
{
  Pose2D p1(atan2((double)(v1.y - v3.y), (double)(v1.x - v3.x)) + pi_2,
            Vector2<double>((v1.x + v3.x) / 2, (v1.y + v3.y) / 2)),
         p2(atan2((double)(v2.y - v3.y), (double)(v2.x - v3.x)) + pi_2,
            Vector2<double>((v2.x + v3.x) / 2, (v2.y + v3.y) / 2)),
         p = p2 - p1;
  double sinAngle = sin(p.getAngle()); 
  if(sinAngle)
  {
    p = p1 + Pose2D(Vector2<double>(p.translation.x - p.translation.y * cos(p.getAngle()) / sinAngle, 0));
    return Vector2<int>(int(p.translation.x), int(p.translation.y));
  }
  else
    return v1;
}

void MSH2004ImageProcessor::addBallPercept
(
 int centerX,
 int centerY,
 double radius
 )
{
  Vector2<double>angles;
  Vector2<int> center(centerX, centerY);
  Geometry::calculateAnglesForPoint(center, cameraMatrix, image.cameraInfo, angles);
  ballPercept.add(
    image.cameraInfo,
    center,
    radius,
    angles, 
    Geometry::pixelSizeToAngleSize(radius, image.cameraInfo), 
    cameraMatrix.translation, 
    cameraMatrix.isValid);
  
  //~ ballPercept.add(
    //~ angles, 
    //~ Geometry::pixelSizeToAngleSize(radius, image.cameraInfo), 
    //~ cameraMatrix.translation, 
    //~ cameraMatrix.isValid);
}

bool MSH2004ImageProcessor::checkOrangeInsideBall(const Geometry::Circle& circle)
{
  CIRCLE(imageProcessor_ball1,
    (int)circle.center.x, 
    (int)circle.center.y, 
    (int)circle.radius,
    1, Drawings::ps_solid, Drawings::gray);

  int left = int(circle.center.x - circle.radius),
      right = int(circle.center.x + circle.radius),
      top = int(circle.center.y - circle.radius),
      bottom = int(circle.center.y + circle.radius);
  if(left < 0)
    left = 0;
  if(right >= image.cameraInfo.resolutionWidth-BORDER)
    right = image.cameraInfo.resolutionWidth - 1 - BORDER;
  if(top < 0)
    top = 0;
  if(bottom >= image.cameraInfo.resolutionHeight-BORDER)
    bottom = image.cameraInfo.resolutionHeight - 1 - BORDER;
  if(left >= right || top >= bottom)
    return false; // no part of circle in image -> cannot be seen
  else
  { // sample ball region
    int xSize = right - left,
        ySize = bottom - top,
        maxCount = min(xSize * ySize, 100),
        count = 0;
    double densityFactor = 1.8;
    if (maxCount <= 25)
      densityFactor = 2.4;
    for(int i = 0; i < maxCount; ++i)
    {
      int x = left + int(xSize * random()),
          y = top + int(ySize * random());
      DOT(imageProcessor_ball1, x, y, Drawings::black, Drawings::white);
			if (advPixFilter.getColorClass(image, x, y) == orange)
        ++count;
    }
    if(int(count * densityFactor) < maxCount)
      return false; // not enough orange pixels -> no ball
  }
  return true;
}

bool MSH2004ImageProcessor::checkOrangeInsideBall(const Vector2<double>& offset)
{
  Geometry::Circle circle;
  if(Geometry::calculateBallInImage(offset,cameraMatrix, image.cameraInfo, circle))
  { // calculate square around ball and clip it with image border
    return checkOrangeInsideBall(circle);
  }
  return false;
}

void MSH2004ImageProcessor::clusterRobots(const Vector2<int>& bottomPoint,
                                         bool redFound, bool blueFound)
{
  Vector2<int> coords = bottomPoint,
               pointOnField;
  int noRedCount2 = noRedCount;

  if(redFound)
  {
    lastRed = 
      linesPercept.points[LinesPercept::redRobot][linesPercept.numberOfPoints[LinesPercept::redRobot] - 1];
    if(noRedCount > 2)
      firstRed = closestRed = lastRed;
    else if(closestRed.abs() > lastRed.abs())
      closestRed = lastRed;
    noRedCount = 0;
  }
  // count number of columns without robot points, but ignore columns that are
  // not long enough to contain the robot. Drop single robot points of the other 
  // robot color.
  else if(noRedCount > 2 || bottomPoint==null ||
          (Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField) &&
          closestRed.abs() > pointOnField.abs()))
  {
    if(++noRedCount == 3 && (firstRed != lastRed || noBlueCount > 4))
    {
      SinglePlayerPercept percept;
      percept.direction = (atan2((double)firstRed.y,(double)firstRed.x) + atan2((double)lastRed.y,(double)lastRed.x)) / 2;
      double distance = closestRed.abs();
      percept.offset.x = cos(percept.direction) * distance;
      percept.offset.y = sin(percept.direction) * distance;
      percept.validity = 1;
      playersPercept.addRedPlayer(percept);
    }
  }

  if(blueFound)
  {
    lastBlue = 
      linesPercept.points[LinesPercept::blueRobot][linesPercept.numberOfPoints[LinesPercept::blueRobot] - 1];
    if(noBlueCount > 2)
      firstBlue = closestBlue = lastBlue;
    else if(closestBlue.abs() > lastBlue.abs())
      closestBlue = lastBlue;
    noBlueCount = 0;
  }
  // count number of columns without robot points, but ignore columns that are
  // not long enough to contain the robot. Drop single robot points of the other 
  // robot color.
  else if(noBlueCount > 2 || bottomPoint==null ||
          (Geometry::calculatePointOnField(coords.x, coords.y, cameraMatrix, image.cameraInfo, pointOnField) &&
          closestBlue.abs() > pointOnField.abs()))
  {
    if(++noBlueCount == 3 && (firstBlue != lastBlue || noRedCount2 > 4))
    {
      SinglePlayerPercept percept;
      percept.direction = (atan2((double)firstBlue.y,(double)firstBlue.x) + atan2((double)lastBlue.y,(double)lastBlue.x)) / 2;
      double distance = closestBlue.abs();
      percept.offset.x = int(cos(percept.direction) * distance);
      percept.offset.y = int(sin(percept.direction) * distance);
      percept.validity = 1;
      playersPercept.addBluePlayer(percept);
    }
  }
}

void MSH2004ImageProcessor::clusterGoal()
{
  // add goal percepts
  if(linesPercept.numberOfPoints[LinesPercept::yellowGoal])
  {
    Vector2<int> pointOnField = linesPercept.points[LinesPercept::yellowGoal][linesPercept.numberOfPoints[LinesPercept::yellowGoal] / 2] * 2,
                 point;
    Geometry::calculatePointInImage(pointOnField, cameraMatrix, image.cameraInfo, point);
  }
  if(linesPercept.numberOfPoints[LinesPercept::skyblueGoal])
  {
    Vector2<int> pointOnField = linesPercept.points[LinesPercept::skyblueGoal][linesPercept.numberOfPoints[LinesPercept::skyblueGoal] / 2] * 2,
                 point;
    Geometry::calculatePointInImage(pointOnField, cameraMatrix, image.cameraInfo, point);
  }

  // calculate free part of goal
  int goal = getPlayer().getTeamColor() == Player::blue 
             ? LinesPercept::yellowGoal : LinesPercept::skyblueGoal;
  double maxWidth = 2 * ballRadius;
  Vector2<double> angles;
  Geometry::calculateAnglesForPoint(Vector2<int>((int)image.cameraInfo.opticalCenter.x, (int)image.cameraInfo.opticalCenter.y),cameraMatrix, image.cameraInfo, angles);

  Pose2D camera(angles.x, Vector2<double>(cameraMatrix.translation.x,
                                          cameraMatrix.translation.y));
  if(linesPercept.numberOfPoints[goal])
  {
    bool goalAtBorder2 = linesPercept.points[goal][linesPercept.numberOfPoints[goal] - 1].abs() < closestBottom,
         isFirst = true;
    Vector2<double> first = (Pose2D(linesPercept.points[goal][0]) - camera).translation,
                    last = first;
    for(int i = 1; i < linesPercept.numberOfPoints[goal]; ++i)
    {
      Vector2<double> point = (Pose2D(linesPercept.points[goal][i]) - camera).translation;
      if(fabs(point.y - last.y) > 150)
      {
        if(isFirst && goalAtBorder)
        {
          Vector2<double> point2 = (Pose2D(linesPercept.points[goal][linesPercept.numberOfPoints[goal] - 1]) - camera).translation;
          first = point2 + (first - point2) * 700 / (first - point2).abs();
        }
        double width = fabs(last.y - first.y);
        if(width > maxWidth)
        {
          maxWidth = width;
          addGoal(camera, first, last);
        }
        first = point;
        isFirst = false;
      }
      last = point;
    }
    if(!isFirst && goalAtBorder2)
    {
      Vector2<double> point2 = (Pose2D(linesPercept.points[goal][0]) - camera).translation;
      last = point2 + (last - point2) * 700 / (last - point2).abs();
    }
    double width = fabs(last.y - first.y);
    if(width > maxWidth)
      addGoal(camera, first, last);
  }
}

void MSH2004ImageProcessor::addGoal(const Pose2D& camera, 
                                   const Vector2<double>& first, const Vector2<double>& last)
{
  bool firstIsClose = false,
       lastIsClose = false;
  for(int i = 0; i < playersPercept.numberOfRedPlayers + playersPercept.numberOfBluePlayers; ++i)
  {
    SinglePlayerPercept& p = i < playersPercept.numberOfRedPlayers 
                      ? playersPercept.redPlayers[i]
                      : playersPercept.bluePlayers[i - playersPercept.numberOfRedPlayers];
    Vector2<double> robot = (Pose2D(p.offset) - camera).translation;
    if(robot.x > 0 && fabs(robot.y - first.y) < 200)
      firstIsClose = true;
    if(robot.x > 0 && fabs(robot.y - last.y) < 200)
      lastIsClose = true;
  }

  Vector2<double> first2 = (camera + Pose2D(first)).translation,
                  last2 = (camera + Pose2D(last)).translation;
  double firstDir = atan2(first2.y, first2.x),
         lastDir = atan2(last2.y, last2.x),
         factor;
  if(firstIsClose == lastIsClose)
    factor = 0.5;
  else if(firstIsClose)
    factor = 0.7;
  else
    factor = 0.3;

  obstaclesPercept.angleToFreePartOfGoalWasDetermined[ObstaclesPercept::opponentGoal] = true;
  obstaclesPercept.angleToFreePartOfGoal[ObstaclesPercept::opponentGoal] = firstDir + factor * (lastDir - firstDir);
  double distance = ((first2 + last2) / 2).abs();
  obstaclesPercept.distanceToFreePartOfGoal[ObstaclesPercept::opponentGoal] = distance;
  obstaclesPercept.widthOfFreePartOfGoal[ObstaclesPercept::opponentGoal] = distance ? 2 * atan((first - last).abs() / 2 / distance) : pi;
}

void MSH2004ImageProcessor::clusterFlags(Flag::FlagType flagType, const Vector2<int>& point)
{
  if((point - lastFlag).abs() > 9)
  {
    if(lastFlag.x >= 0)
    {
      Flag::FlagType bestType = Flag::numberOfFlagTypes;
      int maxCount = 0;
      bool ambiguous = true;
      for(int i = 0; i < 6; ++i)
        if(flagCount[i] > maxCount)
        {
          bestType = Flag::FlagType(i);
          maxCount = flagCount[i];
          ambiguous = false;
        }
        else if(flagCount[i] == maxCount)
          ambiguous = true;
      if(!ambiguous)
      {
        Vector2<int> center = (firstFlag + lastFlag) / 2;
        static const colorClass otherColor[6] = {yellow, green, skyblue, yellow, green, skyblue};
        flagSpecialist.searchFlags(
          image, colorTable, cameraMatrix, 
          advPixFilter, otherColor[bestType], 
          bestType <= Flag::pinkAboveSkyblue,
          horizon, center.x, center.y);
      }
    }
    for(int i = 0; i < 6; ++i)
      flagCount[i] = 0;
    firstFlag = point;
  }
  lastFlag = point;
  if(flagType != Flag::numberOfFlagTypes)
    ++flagCount[flagType];
}

void MSH2004ImageProcessor::filterPercepts()
{
  // Close robots produce far ghost robots. Remove them.
  int i;
  for(i = 0; i < playersPercept.numberOfRedPlayers + playersPercept.numberOfBluePlayers; ++i)
  {
    SinglePlayerPercept& pi = i < playersPercept.numberOfRedPlayers 
                      ? playersPercept.redPlayers[i]
                      : playersPercept.bluePlayers[i - playersPercept.numberOfRedPlayers];
    double iDist = pi.offset.abs(),
           iAngle = atan2(pi.offset.y, pi.offset.x),
           ignoreAngle = atan2(150, iDist);
    for(int j = 0; j < playersPercept.numberOfRedPlayers + playersPercept.numberOfBluePlayers; ++j)
      if(i != j)
      {
        SinglePlayerPercept& pj = j < playersPercept.numberOfRedPlayers 
                            ? playersPercept.redPlayers[j]
                            : playersPercept.bluePlayers[j - playersPercept.numberOfRedPlayers];
        if(iDist < pj.offset.abs() &&
           fabs(atan2(pj.offset.y, pj.offset.x) - iAngle) < ignoreAngle)
        {
          if(j < playersPercept.numberOfRedPlayers)
            pj = playersPercept.redPlayers[--playersPercept.numberOfRedPlayers];
          else
            pj = playersPercept.bluePlayers[--playersPercept.numberOfBluePlayers];
          if(i > j)
          {
            i = j; // check pi again
            break;
          }
          else
            --j; // check pj again
        }
      }
  }

  // If there are too few line points of a certain type, remove them all, they may be misreadings
  for(i = 0; i < LinesPercept::numberOfLineTypes; ++i)
    if(linesPercept.numberOfPoints[i] < 3)
      linesPercept.numberOfPoints[i] = 0;

  // if many more field line points than border points have been found,
  // it is very likely that the border points also result from field lines
  // So remove them.
  if(linesPercept.numberOfPoints[LinesPercept::field] > 
     3 * linesPercept.numberOfPoints[LinesPercept::border])
    linesPercept.numberOfPoints[LinesPercept::border] = 0;
}

Vector2<int> MSH2004ImageProcessor::getCoords(const unsigned char* p) const
{
  int diff = p - &image.image[0][0][0];
  return Vector2<int>(diff % cameraResolutionWidth_ERS7,diff / (cameraResolutionWidth_ERS7 * 3));
}

void MSH2004ImageProcessor::plot(const Vector2<int>& p,Drawings::Color color)
{
  if(lineColor == Drawings::numberOfColors)
  {
    last = p;
    lineColor = color;
  }
  else if(color != lineColor)
  {
    const Vector2<int> p1 = last,
                 p2 = p;
    LINE(imageProcessor_general,p1.x,p1.y,p2.x,p2.y,0,Drawings::ps_solid,lineColor);
    last = p;
    lineColor = color;
  }
}

bool MSH2004ImageProcessor::handleMessage(InMessage& message)
{
  switch(message.getMessageID())
  {
    case idLinesImageProcessorParameters:
      GenericDebugData d;
      message.bin >> d;
      yThreshold = (int)d.data[1];
      vThreshold = (int)d.data[2];
      execute();
      return true;
  }
  return false;
}

double MSH2004ImageProcessor::middleEdgePointDist(Geometry::Circle &circle){
	double validity = 0;
	double dist = 0;
	
	for (int i = 0;i<numberOfEdgePoints;i++){
		Vector2<double> vertex(ballEdgePoints[i].x,ballEdgePoints[i].y);
		dist += fabs((circle.center - vertex).abs() - circle.radius);
	}
	validity = dist/(double)numberOfEdgePoints;

	return validity;
}

double MSH2004ImageProcessor::calculateLargeCircle(
		const Box& input,Geometry::Circle& circle)
{	
	numberOfEdgePoints = 0;
	int cx = (input.minX + input.maxX)/2;
	int cy = (input.minY + input.maxY)/2;

		
	Vector2<int> start,aim,scan;
	double step = pi2/MAX_EDGE_POINTS;
	start.x = cx;
	start.y = cy;
	for (int i = 0;i<MAX_EDGE_POINTS;i++)
	{
		scan = start;
		Vector2<double> dir;
		getCoordinatesByAngle((double)i*step,dir.x,dir.y);
		aim.x = (int)dir.x + start.x;
		aim.y = (int)dir.y + start.y;
		/*LINE(imageProcessor_ball1,
			aim.x,aim.y, 
			start.x,start.y, 
			1, Drawings::ps_solid, Drawings::orange);*/ 
		
		if(edgeScanner->susanScan(scan.x,scan.y,dir))
		{
			ballEdgePoints[numberOfEdgePoints++] = scan;
			LINE(imageProcessor_ball1,
				scan.x,scan.y, 
				start.x,start.y, 
				1, Drawings::ps_solid, Drawings::white); 
		}
		
	}
	if (numberOfEdgePoints<3) return 0;
	return calculateCircleByEdges(circle);

}

double MSH2004ImageProcessor::calculateSmallCircle(Geometry::Circle& circle){
	switch (numberOfEdgePoints){
	case 3:	
		circle = Geometry::getCircle(
			ballEdgePoints[0],ballEdgePoints[1],ballEdgePoints[2]);
		if (circle.radius > 5){
			circle.radius = 5;
			roundness = 0.61f;
			return 0;
		}
		roundness = 0.61f;
		return validateCircle(circle);
	case 2:	
		circle.center.x = (ballEdgePoints[0].x + ballEdgePoints[1].x)/2;
		circle.center.y = (ballEdgePoints[0].x + ballEdgePoints[1].x)/2;
		//only guessed, the radius could be approximated whith a few
		//scanlines through the area of the two points
		circle.radius = (ballEdgePoints[0] - ballEdgePoints[1]).abs()/1.6;
		if (circle.radius > 5){
			circle.radius = 5;
			roundness = 0.61f;
			return 0;
		}
		return validateCircle(circle);
	case 1:
		circle.center.x = ballEdgePoints[0].x;
		circle.center.y = ballEdgePoints[0].y;
		circle.radius = 1.5;
		roundness = 0.61f;
		return validateCircle(circle);
	default:
		roundness = 0;
		return 0;

	}
}

double MSH2004ImageProcessor::calculateCircleByEdges(Geometry::Circle& circle){
	Geometry::Circle current;
	int chances = 20;
	double best = 1000;
	double validity = 0;
	double distance = 0;
	int r1 = 0;
	int r2 = 0;
	int r3 = 0;

	for(int i = 0;i<chances;i++){
		r1 = rand()%numberOfEdgePoints;
		do{ r2 = rand()%numberOfEdgePoints; }while(r2 == r1);
		do{ r3 = rand()%numberOfEdgePoints; }while(r3 == r1 || r3 == r2);
		current = Geometry::getCircle(
			ballEdgePoints[r1],ballEdgePoints[r2],ballEdgePoints[r3]);
		
		distance = middleEdgePointDist(current);
		//OUTPUT(idText,text,"dist-test: " << distance);
		if (distance > best) continue;
		best = distance;
		circle = current;
	}
	//OUTPUT(idText,text,"dist-best: " << best);
	roundness = validateEdgePoints(circle);
	validity = validateCircle(circle);
	return validity;
}

double MSH2004ImageProcessor::validateCircle(Geometry::Circle &circle)
{	

	/*CIRCLE(imageProcessor_ball1,
	(int)circle.center.x, 
	(int)circle.center.y, 
	(int)circle.radius,
	0.3, Drawings::ps_solid, Drawings::white);
	*/
	
	
	/*Validity using size of the radius */
	double size = 1;
	
	double maxRadius =  25  /*((double)176)/(double)7*/;
	size = circle.radius;

	if (size > maxRadius ) size = maxRadius;
	size = size/maxRadius;

	/*Validity using dirty Points*/
	double pValidity = 1;
	int points = 1;
	int validPoints = 1;
	int counts = 7;

	double r2 = circle.radius * circle.radius;
	
	int minX = (int) (circle.center.x - circle.radius);
	int minY = (int) (circle.center.y - circle.radius);
	int maxX = (int) (circle.center.x + circle.radius);
	int maxY = (int) (circle.center.y + circle.radius);

	//merging imageBounds with circle Bounds

	if (minX < 0) minX = 0;
	if (minY < 0) minY = 0;
	if (maxX > image.cameraInfo.resolutionWidth) maxX = image.cameraInfo.resolutionWidth;
	if (maxY > image.cameraInfo.resolutionHeight) maxY = image.cameraInfo.resolutionHeight;
	
	//if (maxX > 176) maxX = 176;
	//if (maxY > 144) maxY = 144;
	int margin = (int) (((maxX - minX) + (maxY - minY))/(2*counts) + (double)0.5);
	//OUTPUT(idText,text,"Margin: " << margin);
	if (margin < 2) margin = 2;

	//counting validPoints
	if (maxX > margin) maxX -= margin;
	if (maxY > margin) maxY -= margin;
	
	for (int x = minX; x < maxX; x += margin)
		for (int y = minY; y < maxY - margin; y+= margin){
		points++;
		
		colorClass color = getColor(x,y);
		
		if (Geometry::insideCircle(circle,r2,x,y)){
			if (color == orange)
				validPoints++;
			else{
				DOT(imageProcessor_ball1, x, y, Drawings::white, Drawings::red);
			}
		}
		else if	(color == green || color == white || color == gray || 
				 color == yellow || color == skyblue){
			validPoints++;
		}
		else{
			DOT(imageProcessor_ball1, x, y, Drawings::white, Drawings::red);
		}
		
	}

	pValidity = (double)validPoints/(double)points;
	//OUTPUT(idText,text,"validPoints: " << validPoints);
	//OUTPUT(idText,text,"allPoints: " << points);
	//OUTPUT(idText,text,"Size-Validity: " << size);
	//OUTPUT(idText,text,"DirtyPoints-Validity: " << pValidity);
	//OUTPUT(idText,text,"Ball-Validity: " << (size + pValidity*4)/5);
	//return (size + pValidity*2)/3 ;
	return pValidity;
}

double MSH2004ImageProcessor::validateEdgePoints(Geometry::Circle &circle){
	double validity = 0;
	int validPoints = 0;
	double dist = 0;
	
	for (int i = 0;i<numberOfEdgePoints;i++){
		Vector2<double> vertex(ballEdgePoints[i].x,ballEdgePoints[i].y);
		dist = fabs((circle.center - vertex).abs() - circle.radius);
		if (dist < MAX_CIRCLE_DIST) validPoints++;
	}
	validity = ((double)validPoints)/(double)numberOfEdgePoints;

	return validity;
}

void MSH2004ImageProcessor::addBallPercept(Geometry::Circle &circle,double validity)
{	
//	OUTPUT(idText,text,"Dirty-test: " << validity);
//	OUTPUT(idText,text,"Roundness-test: " << roundness);

	if (roundness < 0.6 || validity < 0.5) return;
	
	//putting both together and test again
	validity = (roundness + validity)/2.0f;
	if(validity < 0.5) return;

	
	//if ( circle.radius/(pointsInSegment*2) > 5 ) return;

	if (Geometry::getDistanceToLine(horizon,circle.center) > 0) return;

	
	Vector2<int> bottomPoint((int)circle.center.x,(int)(circle.center.y + circle.radius));
	Vector2<int> pointOnField;
	if(!Geometry::calculatePointOnField(bottomPoint.x,bottomPoint.y,
		cameraMatrix,image.cameraInfo,pointOnField)) return;

	

	CIRCLE(imageProcessor_ball1,
		(int)circle.center.x, 
		(int)circle.center.y, 
		(int)circle.radius,
		1, Drawings::ps_solid, Drawings::blue);
    
//	OUTPUT(idText,text,"Ball-Validity: " << validity);
    Vector2<int> center((int)circle.center.x,(int)circle.center.y);

	//Ball zum Percept hinzufgen
	Vector2<double> angles;
    Geometry::calculateAnglesForPoint(
		center,
		cameraMatrix, image.cameraInfo, angles);
  
	ballPercept.add(
	image.cameraInfo,
    center,
    circle.radius,
    angles, 
    Geometry::pixelSizeToAngleSize(circle.radius,image.cameraInfo),
		cameraMatrix.translation, 
		cameraMatrix.isValid);

	//OUTPUT(idText,text,"Ball-Distance: " << ballPercept.getDistanceSizeBased());
}



/*
* Change log :
* 
* $Log: MSH2004ImageProcessor.cpp,v $
* Revision 1.7  2004/05/17 11:57:04  nistico
* Imported ball detection from RasterImageProcessor (integrated in MSH by Bernd in New Orleans)
*
* Revision 1.6  2004/05/12 11:22:41  nistico
* Minor changes
*
* Revision 1.5  2004/05/09 16:35:10  nistico
* Removed references to -Intrinsic named functions
*
* Revision 1.4  2004/04/18 18:14:53  nistico
* USE_INTRINSIC layout removed.
* All functions properly replicated in intrinsic version.
* However, image processor (MSH2004) making use of them get distorted visualization
* of percepts, because drawing functions use the old parameters.
* It has to be decided wheter to fully move to intrinsic, or discard it.
*
* Revision 1.3  2004/04/18 11:57:46  nistico
* Removed MSH2004ImageProcessor2 (integrated all changes into MSH2004ImageProcessor)
*
* Revision 1.1  2004/04/08 16:21:03  wachter
* GT04 checkin of Microsoft-Hellounds
*
*/
