/**
* @file MSH2004ImageProcessor.h
*
* Definition of class MSH2004ImageProcessor
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
* @author <a href="mailto:sadprofessor@web.de">Bernd Schmidt</a>
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
*/

#ifndef __MSH2004ImageProcessor_h_
#define __MSH2004ImageProcessor_h_

#include "Modules/ImageProcessor/ImageProcessor.h"
#include "MSH2004FlagSpecialist.h"
#include "MSH2004GoalRecognizer.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Debugging/DebugImages.h"
#include "Representations/Perception/ColorTable32K.h"
#include "MSH2004PixelFilter.h"
#include "Modules/ImageProcessor/ImageProcessorTools/GT2004EdgeDetection.h"


#define BORDER 1

/**
* @class MSH2004ImageProcessor
*
* The lines image processor recognizes characteristic lines in the image.
* Four types of lines are distinguished:
* edges between the skyblue goal and the field, edges between the yellow goal 
* and the field, edges between the border and the field, and edges between the
* field lines and the field.
*
* The module scans vertical and horizontal lines in the image from top to bottom
* and from left to right. As the green of the field is very dark, all edges are
* characterized by a big difference of the y-channel of adjacent pixels. An
* increase in the y-channel followed by a decrease is an indication for an edge.
*
* The projection of the pixels on the field plane is used to determine their
* relative position to the robot.
*
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/ 

class MSH2004ImageProcessor : public ImageProcessor
{
public:
  /** 
  * Constructor.
  * @param interfaces The paramters of the MSH2004ImageProcessor module.
  */
  MSH2004ImageProcessor(const ImageProcessorInterfaces& interfaces);

  DECLARE_DEBUG_COLOR_CLASS_IMAGE(segmentedImage1);

  /** Executes the module */
  virtual void execute();
  virtual bool handleMessage(InMessage& message);

  enum {YY=0, UU, VV};
  inline unsigned char getSpectrum(const Vector2<int>& pix, int spectrum)
  {
    return advPixFilter.getSpectrum(image, pix.x, pix.y, spectrum);
  }

  const Vector2<int> null;

private:

  Image* scratchImgPointer;
  MSH2004PixelFilter advPixFilter;
  Image undistortedImg;

  /**
  * The struct represents a ball point.
  */
  struct BallPoint : Vector2<int>
  {
    bool isBottom, /**< Is the point at the lower side of the ball? */
         greenIsClose;
  };

	struct Box;

	struct Box{
	Box(int minX,int minY,int maxX,int maxY);

	int minX;
	int minY;
	int maxX;
	int maxY;
		
	inline void merge(Box& box){
		if (box.minX < minX) minX = box.minX;
		if (box.maxX > maxX) maxX = box.maxX;
		if (box.minY < minY) minY = box.minY;
		if (box.maxY > maxY) maxY = box.maxY;
	}

	inline void add(int x,int y){
		if (x<minX) minX = x;
		else if (x>maxX) maxX = x;
		if (y<minY) minY = y;
		else if(y>maxY) maxY = y;
	}

	inline void reset(int x, int y){
		minX = x;
		minY = y;
		maxX = x;
		maxY = y;
	}
	};
	int numberOfEdgePoints;
	double roundness;
	double validity;
  double xFactor, /**< Factor to convert the pixel coordinate space to the anglular coordinate space. */
         yFactor; /**< Factor to convert the pixel coordinate space to the anglular coordinate space. */
  int yThreshold; /**< Brightness increase threshold. */
  int vThreshold; /**< Brightness decrease threshold. */
  Geometry::Line horizon; /**< The horizon. */
  int orangeCount,  /**< Number of columns with ball points. */
      noOrangeCount, /**< Number of columns without a ball point. */
      noRedCount, /**< Number of columns without a red robot point. */
      noBlueCount, /**< Number of columns without a blue robot point. */
      noGoalCount, /**< Number of columns without a opponent goal seen. */
      closestBottom; /**< Closest bottom point on the grid. */
  Vector2<int> firstRed, /**< First red robot point in a cluster. */
               closestRed, /**< Closest red robot point in a cluster. */
               lastRed, /**< Last red robot point in a cluster. */
               firstBlue, /**< First blue robot point in a cluster. */
               closestBlue, /**< Closest blue robot point in a cluster. */
               lastBlue, /**< Last blue robot point in a cluster. */
               firstBall, /**< First center between top and bottom of a ball. */
               lastBall, /**< Last center between top and bottom of a ball. */
               firstFlag, /**< First flag point in a cluster. */
               lastFlag; /**< Last flag point in a cluster. */
  bool goalAtBorder; /**< Is the first goal point at the image border? */
  enum {maxNumberOfPoints = 400,MAX_EDGE_POINTS = 25,MAX_CIRCLE_DIST = 2}; /**< The maximum number of ball points. */
	Vector2<int> ballEdgePoints[MAX_EDGE_POINTS];
  BallPoint ballPoints[maxNumberOfPoints]; /**< The ball points. */
  int numberOfBallPoints; /**< The actual number of ball points. */
  int flagCount[6]; /**< Histogram for flag points. */

  CameraMatrix cmBall, /**< Camera matrix without ball radius. */
               cmTricot; /**< Camera matrix without tricot height. */

  MSH2004FlagSpecialist flagSpecialist; /**< The flag specialist. */
  GT2004EdgeDetection* edgeScanner;
	MSH2004ColorCorrector corrector;

	//~ int ballThreshold;
  
  /**
  * The function scans columns for line points.
  */
  void scanColumns();

  /** 
  * The function scans rows for line points. 
  */
  void scanRows();

  /** 
  * The function scans a line for line points. 
  * @param start The start point of the line.
  * @param end The end point of the line.
  * @param vertical Vertical lines are scanned for more information.
  * @param noLines Should the line not be scanned for points on field lines or borders?
  */
  void scan(const Vector2<int>& start, const Vector2<int>& end,
            bool vertical, bool noLines);

  /** 
  * The function clusters points of balls.
  * @param topPoint The top point of the current scan column.
  * @param bottomPoint The bottom point of the current scan column.
  * @param first The upper ball point. If 0, no ball has been found.
  * @param last The lower ball point. If 0, no ball has been found.
  * @param greenAboveOrange Was there green above the top point?
  * @param greenBelowOrange Was there green below the bottom point?
  */
  void clusterBalls(const Vector2<int>& topPoint,
                    const Vector2<int>& bottomPoint,
                    const Vector2<int>& first,
                    const Vector2<int>& last,
                    bool greenAboveOrange,
                    bool greenBelowOrange);

  /**
  * The function creates a ball percept from the ball points stored in "ballPoints".
  * It replaces the previous percept if the new ball is larger than the larger one 
  * found so far.
  */
  void createBallPercept();

  /**
  * The function filters outlier points.
  */
  void filterBallPoints();

  /**
  * The functions tries to calculate the ball percept by calculating the center of the ball.
  * @param onlyIfGreenIsClose Use only points that are close to green.
  * @return Has a percept been generated?
  */
  bool createBallPerceptFrom3Points(bool onlyIfGreenIsClose);

  /**
  * The function selects three points from the array "ballPoints".
  * The points will be used to determine the center of the ball.
  * @param point1 The index of the first point is returned to this variable.
  * @param point2 The index of the second point is returned to this variable.
  * @param point3 The index of the third point is returned to this variable.
  * @param onlyIfGreenIsClose Use only points that are close to green.
  * @return Were there enough adequate points?
  */
  bool select3Points(int& point1, int& point2, int& point3, bool onlyIfGreenIsClose) const;

  /**
  * The function determines the center of the ball by cutting the middle perpendiculars.
  * @param v1 The first point on the border of the ball.
  * @param v2 The second point on the border of the ball.
  * @param v3 The third point on the border of the ball.
  * @return The center of the ball.
  */
  Vector2<int> cutMiddlePerpendiculars(const Vector2<int>& v1,
                                       const Vector2<int>& v2,
                                       const Vector2<int>& v3) const;

  /**
  * The function checks whether a ball percept is plausible and will add it if so.
  * @param centerX The x-coordinate of the center of the ball.
  * @param centerY The y-coordinate of the center of the ball.
  * @param radius The radius of the ball.
  */
  void addBallPercept(
    int centerX,
    int centerY,
    double radius
    );  


  /**
  * The function checks whether a ball percept is plausible.
  * @param offset The offset of the ball.
  * @return Is enough orange inside the ball? 
  */
  bool checkOrangeInsideBall(const Vector2<double>& offset);

  /**
  * The function checks whether a ball percept is plausible.
  * @param circle A circle in the image.
  * @return Is enough orange inside the ball? 
  */
  bool checkOrangeInsideBall(const Geometry::Circle& circle);

  
  /** 
  * The function clusters points of red and blue robots.
  * @param bottomPoint The bottom point of the current scan column.
  * @param redFound Has a red robot point been found? In that case, the last
  *                 entry in the lines percept is that point.
  * @param blueFound Has a blue robot point been found? In that case, the last
  *                  entry in the lines percept is that point.
  */
  void clusterRobots(const Vector2<int>& bottomPoint, bool redFound, bool blueFound);

  /** 
  * The function clusters points of the opponent's goal.
  */
  void clusterGoal();

  /** 
  * The function adds or replaces the direction to the free part of the opponent's goal.
  */
  void addGoal(const Pose2D& camera, 
               const Vector2<double>& first, const Vector2<double>& last);

  /** 
  * The function clusters flag points.
  * @param flagType The type of the flag.
  * @param point The center of the pink area.
  */
  void clusterFlags(Flag::FlagType flagType, const Vector2<int>& point);

  /**
  * The function filters the percepts, i.e. it removes potential misreadings.
  */
  void filterPercepts();

  /** 
  * The function converts an address to pixel coordinates.
  * @param p An address in image.image.
  * @return The x- and y-coordinates of the corresponding pixel.
  */
  Vector2<int> getCoords(const unsigned char* p) const;

  //!@name Helpers for grid drawing
  //!@{
  Vector2<int> last;
  Drawings::Color lineColor;
  void plot(const Vector2<int>& p,Drawings::Color color);
  //!@}

  //obstacle buffers
/*  Vector2<int> obstaclesNearPointOnField2;
  bool obstaclesFarPointOnField2IsOnImageBorder;
  
  Vector2<int> obstaclesFarPointOnField1;
  Vector2<int> obstaclesFarPointOnField2;
*/
  double angleBetweenDirectionOfViewAndGround;

  int numberOfScannedPixels;

	  double calculateLargeCircle(
		const Box& input,Geometry::Circle& circle);

	void addBallPercept(Geometry::Circle &circle,double validity);

  /** Calculates middled difference of the circle and an edge-point. */ 
  double validateEdgePoints(Geometry::Circle& circle);

	double calculateCircleByEdges(Geometry::Circle& circle);

	double middleEdgePointDist(Geometry::Circle& circle);

	double calculateSmallCircle(Geometry::Circle& circle);

	double validateCircle(Geometry::Circle &circle);

		/* implemented with ColorTable interface*/
	inline colorClass getColor(int x,int y){

		//TODO remove hack for color correction
		unsigned char cy = image.image[y][0][x];
		unsigned char cu = image.image[y][1][x];
		unsigned char cv = image.image[y][2][x];
		corrector.getCorrectedPixel(x,y,cy,cu,cv);
		return colorTable.getColorClass(cy,cu,cv);	
	};

	inline void getCoordinatesByAngle(double angle,double& x,double& y){
		x = 20*cos(angle);
		y = 20*sin(angle);
	}
 
  DECLARE_DEBUG_IMAGE(imageProcessorPlayers);
};
#endif// __MSH2004ImageProcessor_h_

/*
* Change log :
* 
* $Log: MSH2004ImageProcessor.h,v $
* Revision 1.5  2004/05/17 11:57:04  nistico
* Imported ball detection from RasterImageProcessor (integrated in MSH by Bernd in New Orleans)
*
* Revision 1.4  2004/05/12 11:22:41  nistico
* Minor changes
*
* Revision 1.3  2004/04/18 11:57:46  nistico
* Removed MSH2004ImageProcessor2 (integrated all changes into MSH2004ImageProcessor)
*
* Revision 1.1  2004/04/08 16:21:04  wachter
* GT04 checkin of Microsoft-Hellounds
*
*/
