// RasterSpecialist.h: Schnittstelle fr die Klasse RasterSpecialist.
//
//////////////////////////////////////////////////////////////////////

#ifndef RASTERSPECIALIST_H
#define RASTERSPECIALIST_H


#include "RasterImageProcessor.h"
#include "Tools/Math/Geometry.h"
#include <list>
#include <vector>


/** Die Richtungen, die fr die scan-methoden relevant sind. */

enum RasterDirections{
	north = 0,
	northEast,
	east,
	southEast,
	south,
	southWest,
	west,
	northWest,
	numberOfRasterDirections
};



/** Base for RasterSpecialists */

class RasterSpecialist  
{
public:

	//Two Points on a line 
	struct LinePair;
	//Stores enclosed pairs in a list
	struct ExtLinePair;

	struct ColoredLP;

	struct GridLP;

	struct LinePair{
		LinePair();
		LinePair(Vector2<int> vec1,Vector2<int> vec2);
		LinePair(Vector2<int> vec1,Vector2<int> vec2,colorClass c);
		Vector2<int> v1;
		Vector2<int> v2;
		int segment;
		colorClass color;

		/** defines the default order for LinePairs.
			order is lexigographical in v1 (v1.y,v1.x).
			*/
		bool operator <(const LinePair& other){
			return v1.y < other.v1.y || 
				   v1.y == other.v1.y && v1.x < other.v1.y ;
		}
	};

	struct ExtLinePair:LinePair{		
		ExtLinePair *enclosed;		
	};
	
	/** Four colors for the vector2 begin and end representing in/out colors
	 *	0 and 1 should be v1-in and v1-out, same for  
	 *	2 and 3  v2-in and v2-out
	 */
	struct ColoredLP:LinePair{
		colorClass inOut[4];
	};
	
	/** Holds also information about the position in grid */
	struct GridLP:LinePair{
		int column;
		int d1;
		int d2;
	};


	typedef struct{
		int operator()(const LinePair& pair1,const LinePair& pair2){
			return abs(pair1.v1.x - pair1.v2.x) > abs(pair2.v1.x - pair2.v2.x);
		}
	}  LineCompare;

	struct ListSizeOrder{
		bool operator()(const std::list<LinePair >& l1, const std::list<LinePair >& l2){
			return l1.size() < l2.size();
		}
	};

	struct VectorSizeOrder{
		bool operator()(const std::vector<LinePair >& l1, const std::vector<LinePair >& l2){
			return l1.size() < l2.size();
		}
	};

	struct SegmentNumberOrder{
		bool operator()(const LinePair& pair1, const LinePair& pair2){
			return pair1.segment < pair2.segment;
		}
	};

	class Box;

	class Box{
	public:
		Box(int minX,int minY,int maxX,int maxY);
		
		int minX;
		int minY;
		int maxX;
		int maxY;
			
		inline void merge(Box& box){
			if (box.minX < minX) minX = box.minX;
			if (box.maxX > maxX) maxX = box.maxX;
			if (box.minY < minY) minY = box.minY;
			if (box.maxY > maxY) maxY = box.maxY;
		}

		inline void add(int x,int y){
			if (x<minX) minX = x;
			else if (x>maxX) maxX = x;
			if (y<minY) minY = y;
			else if(y>maxY) maxY = y;
		}

		inline void reset(int x, int y){
			minX = x;
			minY = y;
			maxX = x;
			maxY = y;
		}
	};

	
	typedef const unsigned char* I_Pin ;
	
	RasterSpecialist(RasterImageProcessor &imagePro);
	
	virtual ~RasterSpecialist();
	
	virtual int getType() = 0;
	
	virtual void init() = 0;

	virtual void executePostProcessing(){/*do nothing*/}

	virtual void invokeOnPreScan(int x, int y){/*do nothing*/}

	virtual void invokeOnPostScan(int x, int y){/*do nothing*/}

	bool preScanNeeded;
	
	bool postScanNeeded;

	int const imageHeight;

	int const imageWidth;


protected:	
	
	RasterImageProcessor *rip;

	Vector2<int> errorDummy;
	
	int scanHorizontalEdge(int const& row,int const& start, int const& end);
	/** precondition: Lines must be sorted in increasing x - order */
	void createSegmentsFromLines(std::list<LinePair>& lines,
		std::vector<std::list<LinePair> >& segments,int spaceX,int spaceY);

	void createSegmentsFromLines(std::vector<LinePair>& lines,int spaceX,int spaceY,std::vector<int>& segments);
	
	/** precondition: Lines must be sorted in increasing x - order */
	bool createSegmentsFromLines2(std::list<LinePair>& lines,
		std::vector<std::list<LinePair> >& segments,int xSpace, int ySpace);

	/** precondition: Lines must be sorted in increasing column-order */
	bool createGridSegments(std::list<GridLP>& lines,
		std::vector<std::list<GridLP> >& segments,int xSpace, int ySpace);

	/** precondition: Columns must be sorted in increasing y - order */
	void createSegmentsFromColumns(std::list<LinePair>& columns,
		std::vector<std::list<LinePair> >& segments);

	/** precondition: Columns must be sorted in increasing y - order */
	bool doColorSegmentation(std::vector<LinePair>& lines,int& numberOfSegments,
		int xSpace,int ySpace);
	
	/** precondition: Columns must be sorted in increasing y - order */
	bool doColorSegmentation(std::vector<LinePair>& lines,
		std::vector<std::vector<LinePair> >& segments,int xSpace,int ySpace);

	/** precondition: Lines must be sorted in increasing x - order */
	void createConvexPoly(std::list<LinePair>& input,std::vector<Vector2<int> > output);
	
	/* implemented with ColorTable interface*/
	inline colorClass getColorFromRaster(int x,int y){
		return getColor(getVecFromRaster(x,y));	
	};

	/* implemented with ColorTable interface*/
	inline colorClass getColor(Vector2<int> v){
		return getColor(v.x,v.y);	
	};

	/* implemented with ColorTable interface*/
	inline colorClass getColor(int x,int y){
		//return rip->colorTable.getColorClass(
		//	rip->image.image[y][0][x], rip->image.image[y][1][x], rip->image.image[y][2][x]);
		//TODO remove hack for color correction
		unsigned char cy = rip->image.image[y][0][x];
		unsigned char cu = rip->image.image[y][1][x];
		unsigned char cv = rip->image.image[y][2][x];
		rip->corrector.getCorrectedPixel(x,y,cy,cu,cv);
		return rip->colorTable.getColorClass(cy,cu,cv);	
	};

	/* implemented with ColorTable interface*/
	inline bool checkColor(int x,int y,colorClass color){
		return	color ==  
			rip->colorTable.getColorClass(
				rip->image.image[y][0][x], 
				rip->image.image[y][1][x], 
				rip->image.image[y][2][x]);
	};

	/* implemented with ColorTable interface*/
	inline bool checkColor(Vector2<int> v,colorClass color){
		return	color ==  
			rip->colorTable.getColorClass(
				rip->image.image[v.y][0][v.x], 
				rip->image.image[v.y][1][v.x], 
				rip->image.image[v.y][2][v.x]);
	};
	
	//inline I_Pin getPinFromRaster(int x,int y){
	//	return &(rip->image.image
	//				[rip->raster[x][y].x][0][rip->raster[x][y].y]);	
	//};
	//inline I_Pin getPin(Vector2<int> v){
	//	return &(rip->image.image[v.x][0][v.y]);
	//};

	//inline I_Pin getPin(int x, int y){
	//	return &(rip->image.image[x][0][y]);
	//}
	
	/** Gibt einen Vector2 aus dem Raster zurck, 
		der aber die Bildkoordinaten enthlt! */
	inline Vector2<int> getVecFromRaster(int x,int y){
		return Vector2<int>(rip->marginX*x,rip->marginY*y);
	};

	
	
	
	
	/** Scant nach dem gesuchten Farbbergang ()*/
	Vector2<int> scanNorth(int x,int y)
	{	
		Vector2<int> v(x,y);
		if (y<1) return v;
		search = getColor(v.x,v.y /*- rip->marginY*/);

		while (v.y > 0 && checkColor(v,search)) --v.y;
		++v.y;

		return v;
	};

	/** Scant nach dem gesuchten Farbbergang ()*/
	Vector2<int> scanSouth(int x,int y){
		Vector2<int> v(x,y);

		search = getColor(v.x,v.y /*- rip->marginY*/);

		while (v.y < (imageHeight - 1) && checkColor(v,search)) ++v.y;
		--v.y;
		return v;
	};

	/** Scant nach dem gesuchten Farbbergang ()*/
	Vector2<int> scanWest(int x,int y){
		Vector2<int> v(x,y);
		
		search = getColor(v.x /*- rip->marginX*/,v.y);

		while (v.x > 1 && checkColor(v,search))--v.x;
		++v.x;
		return v;
	};

	/** Scant nach dem gesuchten Farbbergang */
	Vector2<int> scanEast(int x,int y){
		Vector2<int> v(x,y);
		search = getColor(v.x /*+ rip->marginX*/,v.y);
		while (v.x < (imageWidth - 2) && checkColor(v,search))++v.x;
		--v.x;
		
		return v;
	};

	/** Scant nach dem gesuchten Farbbergang ()*/
	inline Vector2<int> scanNW(int x,int y){
		Vector2<int> v(x,y);
		
		search = getColor(v.x,v.y);

		while (checkColor(v,search)){	
			--v.x; 
			--v.y;
		}
		
		return v;
	};

	/** Scant nach dem gesuchten Farbbergang ()*/
	inline Vector2<int> scanSW(int x,int y){
		Vector2<int> v(x,y);
		search = getColor(v.x,v.y);


		while (checkColor(v,search)){	
			--v.x; 
			++v.y;
		}
		
		return v;
	};

	/** Scant nach dem gesuchten Farbbergang ()*/
	inline Vector2<int> scanSE(int x,int y){
		Vector2<int> v(x,y);
		search = getColor(v.x,v.y);

		while (checkColor(v,search)){	
			++v.x; 
			++v.y;
		}
		
		return v;	
	};

	/** Scant nach dem gesuchten Farbbergang ()*/
	inline Vector2<int> scanNE(int x,int y){
		Vector2<int> v(x,y);
		search = getColor(v.x,v.y);

		while (checkColor(v,search)){	
			++v.x; 
			--v.y;
		}
		
		return v;
	};

	/** Scant nach dem gesuchten Farbbergang.
		Gescannt wird im Bildkoordinatensystem.
		Hier kann die Richtung als Parameter angegeben werden.
		(Ntzlich wenn man alle acht Typen indiziert verwaltet).
		Der bergebene Vector2 zeigt auf das erste Pixel
		mit der Gesuchten farbe*/
	inline Vector2<int> scan(Vector2<int> v,short direction){


		switch (direction){
		
		case 0:
			currentX = 0;
			currentY = -1;
		case 1:
			currentX = 0;
			currentY = +1;
			break;
		case 2:
			break;
			currentX = -1;
			currentY = 0;
		case 3:
			break;
			currentX = +1;
			currentY = 0;
		case 4:
			break;
			currentX = -1;
			currentY = -1;
		case 5:
			break;
			currentX = -1;
			currentY = +1;
		case 6:
			break;
			currentX = +1;
			currentY = +1;
		case 7:
			break;
			currentX = +1;
			currentY = -1;
		
		default:
			return errorDummy;	
		
		}
		search = getColor(v.x,v.y);

		while (checkColor(v,search)){	 
			v.x =+ currentX; 
			v.y =+ currentY;
		}
		
		return v;
	}

		/** Scant nach dem gesuchten Farbbergang.
		Gescannt wird im Bildkoordinatensystem.
		Hier kann die Richtung als Parameter angegeben werden.
		(Ntzlich wenn man alle acht Typen indiziert verwaltet).
		Der bergebene Vector2 zeigt auf das erste Pixel
		mit der gesuchten Farbe.
		Wohl die allg. Form dieser Methode.*/
	inline Vector2<int> scan(Vector2<int> v,short direction,colorClass search){


		switch (direction){
		
		case 0:
			currentX = 0;
			currentY = -1;
		case 1:
			currentX = 0;
			currentY = +1;
			break;
		case 2:
			break;
			currentX = -1;
			currentY = 0;
		case 3:
			break;
			currentX = +1;
			currentY = 0;
		case 4:
			break;
			currentX = -1;
			currentY = -1;
		case 5:
			break;
			currentX = -1;
			currentY = +1;
		case 6:
			break;
			currentX = +1;
			currentY = +1;
		case 7:
			break;
			currentX = +1;
			currentY = -1;
		
		default:
			return errorDummy;	
		
		}
		while (!checkColor(v,search)){	 
			v.x =+ currentX; 
			v.y =+ currentY;
		}
		
		return v;
	}


	/** Scant nach dem gesuchten Farbbergang (Typ nach Wahl)*/
	inline Vector2<int> scan(int x,int y,short direction){
		Vector2<int> v(x,y);
		return scan(v,direction);
	}
	
	/** Gives a number between 0 and 360 with the same order as v.angle()*/
	inline double theta(Vector2<int>& v1, Vector2<int>& v2){
		double t = 0;
		int dx = v1.x - v2.x;
		int dy = v1.y - v2.y;
		int ax = abs(dx);
		int ay = abs(dy);
		if (dx == 0 && dy == 0) t = 0;
		else t = (double)dy /(double)(ax + ay);
		if (dx < 0) t = 2 - t;
		else if (dy < 0) t = 4 + t;
		return t*90; // multiplication with 90 is not needed for correct ordering
	}

	inline int sign(int value){
		return value < 0 ? -1 : 1;
	}

	/** Scans for edges in the image.
	 *  @param threshold higher values are giving less edges
	 *  @param direction the scan-direction
	 *
	*/
inline void scanEdge(Vector2<int>& v,short direction,int threshold){

	int cx = v.x;
	int cy = v.y;
	int diff = 0;
	int buffer = 0;

	switch (direction){
	
	case north:
		currentX = 0;
		currentY = -1;
	case south:
		currentX = 0;
		currentY = +1;
		break;
	case west:
		break;
		currentX = -1;
		currentY = 0;
	case east:
		break;
		currentX = +1;
		currentY = 0;
	case northWest:
		break;
		currentX = -1;
		currentY = -1;
	case southWest:
		break;
		currentX = -1;
		currentY = +1;
	case southEast:
		break;
		currentX = +1;
		currentY = +1;
	case northEast:
		break;
		currentX = +1;
		currentY = -1;
	
	default:
		v = errorDummy;
		return;
	
	}
	while (true){
		cx =+ currentX;
		cy =+ currentY;

		if (cx < 0 || cy < 0 || cx >= imageWidth || cy >= imageHeight) break; //edge touches Border

		diff =(rip->image.image[v.y][0][v.x] + rip->image.image[v.y][1][v.x]) 
			- (rip->image.image[cy][0][cx] + rip->image.image[cy][1][cx]);

		if(diff > threshold || buffer != 0){
			if (diff > buffer) buffer = diff;
			else break; //edge Found			
		}
		v.x = cx; 
		v.y = cy;
	}
}
	
private:
	
	I_Pin pin;

	int offset;

	int currentX;

	int currentY;

	colorClass search;

	int const Y_VALUE;
	
	int const U_VALUE;

	int const V_VALUE;

	std::list<std::list<LinePair> > segList;

	inline bool linePairsFit(LinePair& inSegment ,LinePair& newPair,int allowedSpace){
        return (!((inSegment.v2.x+allowedSpace<newPair.v1.x)
			||(inSegment.v1.x-allowedSpace>newPair.v2.x)));
	}

	inline bool gridFit(GridLP& inSegment ,GridLP& newPair,int allowedSpace){
        return (!((inSegment.d2+allowedSpace<newPair.d1)
			||(inSegment.d1-allowedSpace>newPair.d2)));
	}

	inline bool columnPairsFit(LinePair& inSegment,LinePair& newPair){
		return 	!(newPair.v2.y < inSegment.v1.y) &&
				!(newPair.v1.y > inSegment.v2.y);
	}

	inline bool linesFit(LinePair& inSegment ,LinePair& newPair,int allowedSpace){
		return (!((inSegment.v2.x+allowedSpace<newPair.v1.x)
			||(inSegment.v1.x-allowedSpace>newPair.v2.x))
			&& newPair.color == inSegment.color);
	}

};

#endif


/*
* Change log :
* 
* $Log: RasterSpecialist.h,v $
* Revision 1.20  2004/05/27 11:18:48  schmidtb
* new version with further work for open challenge
*
* Revision 1.19  2004/05/25 13:27:34  schmidtb
* modified version of rip for open-challenge
*
* Revision 1.20  2004/04/22 16:57:26  pg_besc
* new version of RBallSpecialist2, now omidirectional scans for detection
*
* Revision 1.19  2004/04/20 07:50:27  pg_besc
* new version of pre scan
*
* Revision 1.18  2004/03/17 17:06:09  koh
* added enemyOnlySpecialist;
* added enemySpecialist2;
* added some int and bool variables
*
* Revision 1.17  2004/03/11 20:32:58  schmidtb
* new version of rip
*
* Revision 1.16  2004/03/04 09:54:51  schmidtb
* color correction integrated
*
* Revision 1.15  2004/03/03 12:53:21  schmidtb
* color correction integrated
*
* Revision 1.14  2004/02/18 14:56:19  neubach
* new Segmentation established, code not cleared at all
*
* Revision 1.13  2004/02/04 13:00:49  schmidtb
* new version of BoxSpecialist
*
* Revision 1.12  2004/02/02 13:42:12  schmidtb
* merged sources of RIP. added som functions.
*
* Revision 1.11  2004/01/31 11:45:02  hyung
* modified enemyValidity-calculation;
* established basical enviroment for TrikotErkennung, based on Arrays and Lists, changes will take affect only #ifdef TrikotErkennung!
*
* Revision 1.10  2004/01/26 22:21:09  schmidtb
* bugfixed BoxSpecialist
*
* Revision 1.9  2004/01/26 20:37:56  schmidtb
* removed errors and warnings, merged versions of RGoalSpecialist
*
* Revision 1.8  2004/01/19 16:18:45  schmidtb
* GoalSpecialist recognizes Landmarks
*
* Revision 1.7  2004/01/15 13:45:02  schmidtb
* segmentation established
*
* Revision 1.6  2003/12/15 13:55:32  schmidtb
* Merged and patched new version of RasterImageProcessor.
*
* Revision 1.5  2003/12/08 15:02:55  schmidtb
* new version of RIP
*
* Revision 1.4  2003/12/04 09:51:23  schmidtb
* better BallSpecialist
*
* Revision 1.3  2003/12/02 21:59:02  schmidtb
* New version of RasterImageProcessor
*
* Revision 1.2  2003/11/20 10:26:56  schmidtb
* Ball Detection added
*
* Revision 1.1  2003/11/12 13:13:20  schmidtb
* new RasterImageProcessor added
*
*
*/
