/**
* @file DefaultLEDControl.cpp
* 
* This file contains a class for LED Control.
* @author Max Risler
*/

#include "Representations/Motion/JointDataBuffer.h"
#include "DefaultLEDControl.h"
#include "Tools/Player.h"
#include "Tools/RobotConfiguration.h"

DefaultLEDControl::DefaultLEDControl(const LEDControlInterfaces& interfaces) 
: LEDControl(interfaces),
executeCallCount(0)
{
}

void DefaultLEDControl::executeERS210()
{
/* led coding used here :
redLB     1
greenL    2
redLT     4
redRB     8
greenR    16
redRT     32
greenT    64
tailBlue  128
tailRed   256
  */
  
  const int numOfTailModes = 3;
  const int numOfTeamSequences = 3;
  const int lengthOfTeamSequences = 8;
  const int teamSequences [numOfTailModes][numOfTeamSequences][lengthOfTeamSequences] = 
  {{{256,256,256,256,256,256,256,256}, //red        on
  {128,128,128,128,128,128,128,128},   //blue
  {256,128,256,128,256,128,256,128}},  //undefined
  {{256,256,256,256,0,0,0,0},          //red        slowBlink
  {128,128,128,128,0,0,0,0},           //blue
  {256,128,256,128,256,128,256,128}},  //undefined  
  {{256,0,256,0,256,0,256,0},          //red        fastBlink
  {128,0,128,0,128,0,128,0},           //blue
  {256,128,256,128,256,128,256,128}}}; //undefined
  
  //  
  /*  
  const int numOfRoleSequences = 5;
  const int lengthOfRoleSequences = 4;
  const int roleSequences [numOfRoleSequences][lengthOfRoleSequences] =
  {{18, 18, 18, 18},  //goalie
  {18, 18, 0,  0},    //defender
  {2 , 2,  2,  2},      //striker1
  {16, 16, 16, 16},   //striker2
  {2 , 16, 2,  16}};   //undefined
  //  
  const int numOfGreenSequences = 4;
  const int lengthOfGreenSequences = 8;
  const int greenSequences[numOfGreenSequences][lengthOfGreenSequences] =
  {{64,64,64,64,64,64,64,64}, //on
  {64,0,64,0,64,0,64,0},      //fastBlink
  {64,64,64,64,0,0,0,0},      //slowBlink
  {0,0,0,0,0,0,0,0}};         //off
  //  
  const int numOfRed4Sequences = 24;
  const int lengthOfRed4Sequences = 4;
  const int red4Sequences [numOfRed4Sequences][lengthOfRed4Sequences] =
  {{45,45,45,45}, //all on
  {0,0,0,0},      //off
  {1,1,1,1},      //left bottom
  {4,4,4,4},      //left top
  {8,8,8,8},      //right bottom
  {32,32,32,32},  //right top
  {5,5,5,5},      //left
  {40,40,40,40},  //right
  {9,9,9,9},      //bottom
  {36,36,36,36},  //top
  {33,33,33,33},  //slash
  {12,12,12,12},  //backslash
  {44,44,44,44},  //left bottom off
  {41,41,41,41},  //left top off
  {37,37,37,37},  //right bottom off
  {13,13,13,13},  //right top off
  {1,4,32,8},     //circle left
  {8,32,4,1},     //circle right
  {5,36,40,9},    //2 led circle left
  {9,40,36,5},    //2 led circle right
  {33,12,33,12},  //flash diagonal
  {5,40,5,40},    //flash vertical
  {9,36,9,36},    //flash horizontal
  {45,0,45,0}};   //flash all
  */  
  const int numOfRed2Sequences = 12;
  const int lengthOfRed2Sequences = 4;
  const int red2Sequences [numOfRed2Sequences][lengthOfRed2Sequences] =
  {{9,9,9,9}, //bothOn
  {9,0,9,0},  //bothFastBlink
  {9,9,0,0},  //bothSlowBlink
  {0,0,0,0},  //bothOff
  {1,1,1,1},  //leftOnly
  {8,8,8,8},  //rightOnly
  {1,0,1,0},  //leftFastBlink
  {8,0,8,0},  //rightFastBlink
  {1,1,0,0},  //leftSlowBlink
  {8,8,0,0},  //rightSlowBlink
  {1,8,1,8},  //alternateFast
  {1,1,8,8}};  //alternateSlow

/*  
  const int initialLedCodes[] =
  {0,     // normalLEDOperation -> no change
  192,   // blueOwnKickoff     -> green top on, blue tail on
  128,   // blueOtherKickoff   -> green top off, blue tail on 
  320,   // redOwnKickoff      -> green top on, red tail on
  256    // redOtherKickoff    -> green top off, red tail on
  };     
*/
  
  int ledCode,i,j,k,c;
  k = (ledRequest.tailMode < numOfTailModes) ? ledRequest.tailMode : numOfTailModes - 1;
  i = (getPlayer().getTeamColor() < numOfTeamSequences) ? getPlayer().getTeamColor() : numOfTeamSequences - 1;
  
  for (j=0; j < jointDataBufferNumOfFrames; j++)
  {
    ledCode=0;
    c = (executeCallCount++ * jointDataBufferNumOfFrames+j) / 16;
    
//    if (ledRequest.initialSetupMode == LEDRequest::normalLedOperation)
    {
      ledCode |= teamSequences[k][i][c % lengthOfTeamSequences];
      int bits[] = {2,16,64};
      if (wLanStatus & 1) ledCode |= bits[0];
      if (wLanStatus & 2) ledCode |= bits[1];
      if (wLanStatus & 4) ledCode |= bits[2];
    }
//    else
//    {
//      ledCode |= initialLedCodes[(int)ledRequest.backMode];
//    }
    
    i = (ledRequest.redBottomLEDs < numOfRed2Sequences) ? 
      ledRequest.redBottomLEDs : numOfRed2Sequences - 1;
    ledCode |= red2Sequences[i][c % lengthOfRed2Sequences];
    
    i = (ledRequest.redTopLEDs < numOfRed2Sequences) ? 
      ledRequest.redTopLEDs : numOfRed2Sequences - 1;
    ledCode |= (red2Sequences[i][c % lengthOfRed2Sequences] << 2);
    
    ledValue.data[j]=ledCode;
  }
}

void DefaultLEDControl::executeERS7()
{
  
/** 
* For the ERS7 the same LEDRequests and TailRequest are used as for 
* the ERS210, but they are mapped to other LEDs.
*
* The WLANSatus is shown by the ear-leds. 
* off: no teammate connected, red: one teammate connected, 
* yellow: two teammates connected, green: all three teammates connected.
*
* The red and blue tail-leds are mapped to the back-leds
* 
* The 4 red ERS210 face leds are mapped to 4 ERS7 face leds on the corners of
* the face (top leds are white, bottom leds are green)
  */
  
  /* led coding used here for ERS7
  headOrange   1     
  headWhite    2  << 1
  modeRed      4  << 2
  modeGreen    8  
  modeBlue    16
  wireless    32
  face1       64  << 7
  face2      128
  face3      256
  face4      512
  face5     1024
  face6     2048
  face7     4096
  face8     8192  << 14
  face9    16384
  face10   32768
  face11   65536     Shit - i need a calculator now ;)
  face12  131072
  face13  262144
  face14  524288
  backFrontBlue  1048576        << 20
  backFrontWhite 2097152
  backMiddleOrange 4194304      
  backMiddleWhite 8388608
  backRearRed  16777216
  backRearWhite 33554432
  */
  
  const int numOfTailModes = 3;
  const int numOfTeamSequences = 3;
  const int lengthOfTeamSequences = 8;
  
  const int tsRed = 1 << 24;
  const int tsBlue = 1 << 20;
  const int tsRB = tsRed+tsBlue;
  
  const int teamSequences [numOfTailModes][numOfTeamSequences][lengthOfTeamSequences] = 
  {{{tsRed,tsRed,tsRed,tsRed,tsRed,tsRed,tsRed,tsRed},         //red        on
  {tsBlue,tsBlue,tsBlue,tsBlue,tsBlue,tsBlue,tsBlue,tsBlue},   //blue
  {tsRB,tsRB,tsRB,tsRB,tsRB,tsRB,tsRB,tsRB}},                  //undefined
  {{tsRed,tsRed,tsRed,tsRed,0,0,0,0},                          //red        slowBlink
  {tsBlue,tsBlue,tsBlue,tsBlue,0,0,0,0},                       //blue
  {tsRB,tsRB,tsRB,tsRB,0,0,0,0}},                              //undefined  
  {{tsRed,0,tsRed,0,tsRed,0,tsRed,0},                          //red        fastBlink
  {tsBlue,0,tsBlue,0,tsBlue,0,tsBlue,0},                       //blue
  {tsRB,0,tsRB,0,tsRB,0,tsRB,0}}};                             //undefined
  
  const int numOfRed2Sequences = 12;
  const int lengthOfRed2Sequences = 4;
  
  const int red2Sequences [numOfRed2Sequences][lengthOfRed2Sequences] =
  {
    {3,3,3,3},  //bothOn
    {3,0,3,0},  //bothFastBlink
    {3,3,0,0},  //bothSlowBlink
    {0,0,0,0},  //bothOff
    {2,2,2,2},  //leftOnly
    {1,1,1,1},  //rightOnly
    {2,0,2,0},  //leftFastBlink
    {1,0,1,0},  //rightFastBlink
    {2,2,0,0},  //leftSlowBlink
    {1,1,0,0},  //rightSlowBlink
    {2,1,2,1},  //alternateFast
    {2,2,1,1}   //alternateSlow
  };  

  /*
  const int initialLedCodes[] =
  {0,         // normalLedOperation  -> no change
  5<<20,     // blueOwnKickoff      -> blue back led on, orange back led on
  1<<20,     // blueOtherKickoff    -> blue back led on, orange back led off
  20<<20,    // redOwnKickoff       -> red back led on, orange back led on
  16<<20     // redOtherKickoff     -> red back led on, orange back led off
  };*/
  
  int ledCode,i,j,k,c;
  int battPower = SystemCall::getRemainingPower();
  int wLanCount = 0;
  if (wLanStatus & 1) wLanCount++;
  if (wLanStatus & 2) wLanCount++;
  if (wLanStatus & 4) wLanCount++;
  int bits[] = {0,4,12,8};
  k = (ledRequest.tailMode < numOfTailModes) ? ledRequest.tailMode : numOfTailModes - 1;
  i = (getPlayer().getTeamColor() < numOfTeamSequences) ? getPlayer().getTeamColor() : numOfTeamSequences - 1;
  
  
  for (j=0; j < jointDataBufferNumOfFrames; j++)
  {
    ledCode=0;
    c = (executeCallCount++ * jointDataBufferNumOfFrames+j)/ 16;
    
    
//    if (ledRequest.initialSetupMode == LEDRequest::normalLedOperation)
//    {
      ledCode |= teamSequences[k][i][c % lengthOfTeamSequences];
      if (battPower < 25) 
      {
        ledCode |= 1;   // middle orange back-button led
      }
      
      if (battPower < 15)
      {
        ledCode |= 2;   // middle white back-button led
      }
//    }
//    else
    {
//      ledCode |= initialLedCodes[(int)ledRequest.backMode];
    }
    
    ledCode |= bits[wLanCount];
    
    i = (ledRequest.redBottomLEDs < numOfRed2Sequences) ? 
      ledRequest.redBottomLEDs : numOfRed2Sequences - 1;
    ledCode |= (red2Sequences[i][c % lengthOfRed2Sequences] << 6);
    
    i = (ledRequest.redTopLEDs < numOfRed2Sequences) ? 
      ledRequest.redTopLEDs : numOfRed2Sequences - 1;
    ledCode |= (red2Sequences[i][c % lengthOfRed2Sequences] << 14);
    
    ledValue.data[j]=ledCode;
  }
}

void DefaultLEDControl::execute()
{
  if (getRobotConfiguration().getRobotDesign()==RobotDesign::ERS7)
  {
    executeERS7();
  }
  else
  {
    executeERS210();
  }
}




/*
* Change log :
* 
* $Log: DefaultLEDControl.cpp,v $
* Revision 1.10  2004/05/14 14:12:08  wachter
* - Added communication support for 5 robots
* - rewrote parts of team-communication to be faster and more stable
*
* Revision 1.9  2004/04/09 09:25:15  wachter
* Warning fixed
*
* Revision 1.8  2004/03/25 19:49:21  juengel
* initialLedCodes are not used any more
*
* Revision 1.7  2004/03/25 17:39:17  loetzsch
* renamed initial-setup-mode to back-mode
*
* Revision 1.6  2004/03/19 14:25:58  juengel
* code beautified
*
* Revision 1.5  2004/03/08 01:39:02  roefer
* Interfaces should be const
*
* Revision 1.4  2004/03/03 18:22:21  wachter
* Fixed bug.
*
* Revision 1.3  2004/03/01 12:37:21  wachter
* Added LED-Request for button-interface
*
* Revision 1.2  2004/02/28 00:00:38  wachter
* LED-Requests are now useful on the ERS7.
*
* Revision 1.1  2003/10/06 14:10:14  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 15:27:49  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.6  2003/05/08 14:27:33  risler
* added tailMode to LEDRequest
*
* Revision 1.5  2003/03/06 11:38:41  dueffert
* unused variable and re-order warning removed
*
* Revision 1.4  2002/11/18 17:18:31  loetzsch
* removed "old version" of LED request
*
* Revision 1.3  2002/11/12 23:00:47  dueffert
* started restore greenhills compatibility
*
* Revision 1.2  2002/09/11 20:06:19  loetzsch
* continued experiments with modules/solutions
*
* Revision 1.1  2002/09/10 15:36:15  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.5  2002/08/30 13:37:17  dueffert
* removed unused includes
*
* Revision 1.4  2002/07/08 14:45:29  dueffert
* gcc warning removed
*
* Revision 1.3  2002/05/25 22:52:19  roefer
* WLan, first working approach
*
* Revision 1.2  2002/05/16 22:36:11  roefer
* Team communication and GTMath bugs fixed
*
* Revision 1.11  2002/05/07 05:49:56  dueffert
* new LED handling implemented
*
* Revision 1.10  2002/05/04 17:47:44  loetzsch
* LEDControl now has access to an instance of WLanStatus
* (which shall be displayed then with the green leds. follows next)
*
* Revision 1.9  2002/04/17 17:04:41  risler
* Darmstadt GO
*
* Revision 1.8  2002/03/28 15:18:31  risler
* corrected numOfSequences
*
* Revision 1.7  2002/02/28 14:35:53  risler
* changed names in LEDRequest
*
* Revision 1.6  2002/02/07 19:57:06  risler
* DefaultLEDControl uses getPlayer now, removed team and role from LEDRequest
*
* Revision 1.5  2001/12/28 09:03:34  roefer
* led speed now takes jointDataBufferNumOfFrames into account
*
* Revision 1.4  2001/12/10 17:47:06  risler
* change log added
*
*/
