/**
* @file WakeUpEngine.cpp
* 
* Implementation of class WakeUpEngine.
*
* @author Andreas Heinze
*/

#include "WakeUpEngine.h"
#include "Tools/Debugging/Debugging.h"

WakeUpEngine::WakeUpEngine()
{
  frameNumberOfKeyFrame[0] =   0;
  frameNumberWhenDefaultPidDataIsReached = 50;
  frameNumberOfKeyFrame[1] = 250;
  frameNumberOfKeyFrame[2] = 500;
  frameNumberOfKeyFrame[3] = 750;


  position[JointData::legFL1][2] =  -111000;
  position[JointData::legFL2][2] =  1230000; 
  position[JointData::legFL3][2] =   777000;

  position[JointData::legFR1][2] =   -50000;
  position[JointData::legFR2][2] =  1213000; 
  position[JointData::legFR3][2] =   737000;
  
  position[JointData::legHL1][2] =    93000;
  position[JointData::legHL2][2] =  1207000; 
  position[JointData::legHL3][2] =  1028000;

  position[JointData::legHR1][2] =   142000;
  position[JointData::legHR2][2] =  1144000; 
  position[JointData::legHR3][2] =  1142000;
  

  position[JointData::legFL1][3] = -300000;
  position[JointData::legFL2][3] =  250000;
  position[JointData::legFL3][3] = 1800000;
  
  position[JointData::legFR1][3] = -300000;
  position[JointData::legFR2][3] =  250000;
  position[JointData::legFR3][3] = 1800000;

  position[JointData::legHL1][3] = -600000;
  position[JointData::legHL2][3] =  150000;
  position[JointData::legHL3][3] = 1600000;
  
  position[JointData::legHR1][3] = -600000;
  position[JointData::legHR2][3] =  150000;
  position[JointData::legHR3][3] = 1600000;
}

WakeUpEngine::~WakeUpEngine()
{
}

void WakeUpEngine::start(const SensorData& sensorData, PIDData& pidData)
{
  counter = 0;
  currentKeyFrame = 0;
  
  beginPid = pidData;
  PIDData defaultPid;
  
  int i;
  for(i = 0; i < JointData::numOfJoint; i++)
  {
    diffP[i] = (double)(defaultPid.p[i] - pidData.p[i]) / (double)frameNumberWhenDefaultPidDataIsReached;
    diffI[i] = (double)(defaultPid.i[i] - pidData.i[i]) / (double)frameNumberWhenDefaultPidDataIsReached;
    diffD[i] = (double)(defaultPid.d[i] - pidData.d[i]) / (double)frameNumberWhenDefaultPidDataIsReached;
  }
  
  position[JointData::legFL1][0] = sensorData.data[SensorData::legFL1];
  position[JointData::legFL2][0] = sensorData.data[SensorData::legFL2];
  position[JointData::legFL3][0] = sensorData.data[SensorData::legFL3];

  position[JointData::legFR1][0] = sensorData.data[SensorData::legFR1];
  position[JointData::legFR2][0] = sensorData.data[SensorData::legFR2];
  position[JointData::legFR3][0] = sensorData.data[SensorData::legFR3];

  position[JointData::legHL1][0] = sensorData.data[SensorData::legHL1];
  position[JointData::legHL2][0] = sensorData.data[SensorData::legHL2];
  position[JointData::legHL3][0] = sensorData.data[SensorData::legHL3];

  position[JointData::legHR1][0] = sensorData.data[SensorData::legHR1];
  position[JointData::legHR2][0] = sensorData.data[SensorData::legHR2];
  position[JointData::legHR3][0] = sensorData.data[SensorData::legHR3];

  position[JointData::legFL1][1] = sensorData.data[SensorData::legFL1];
  position[JointData::legFL2][1] = 1200000;
  position[JointData::legFL3][1] = sensorData.data[SensorData::legFL3];

  position[JointData::legFR1][1] = sensorData.data[SensorData::legFR1];
  position[JointData::legFR2][1] = 1200000;
  position[JointData::legFR3][1] = sensorData.data[SensorData::legFR3];

  position[JointData::legHL1][1] = sensorData.data[SensorData::legHL1];
  position[JointData::legHL2][1] = 1200000;
  position[JointData::legHL3][1] = sensorData.data[SensorData::legHL3];

  position[JointData::legHR1][1] = sensorData.data[SensorData::legHR1];
  position[JointData::legHR2][1] = 1200000;
  position[JointData::legHR3][1] = sensorData.data[SensorData::legHR3];

  int keyFrame;
  for(keyFrame = 0; keyFrame < numberOfKeyFrames - 1; keyFrame++)
  {
    for(i = JointData::legFR1; i <= JointData::legHL3; i++)
    {
        stepSize[i][keyFrame] = (position[i][keyFrame+1] - position[i][keyFrame]) / (frameNumberOfKeyFrame[keyFrame + 1] - frameNumberOfKeyFrame[keyFrame]);
    }
  }
}


bool WakeUpEngine::step(JointData& jointData, PIDData& pidData)
{
  counter++;

  int i;
  if(counter < frameNumberWhenDefaultPidDataIsReached)
  {
    for(i = 0; i < JointData::numOfJoint; i++)
    {
      pidData.p[i] = (int)(beginPid.p[i] + counter * diffP[i]);
      pidData.i[i] = (int)(beginPid.i[i] + counter * diffI[i]); 
      pidData.d[i] = (int)(beginPid.d[i] + counter * diffD[i]); 
    }
  }
  if(counter == frameNumberWhenDefaultPidDataIsReached)
  {
    pidData.setToDefaults();
  }

  if(counter > frameNumberOfKeyFrame[currentKeyFrame+1]) currentKeyFrame++;
  if(currentKeyFrame == numberOfKeyFrames) return true;

  int positionInCurrentPhase = counter - frameNumberOfKeyFrame[currentKeyFrame];

  for(i = JointData::legFR1; i <= JointData::legHL3; i++)
  {
    jointData.data[i] = position[i][currentKeyFrame] + positionInCurrentPhase * stepSize[i][currentKeyFrame];
  }

  return false;
}




/*
* Change log :
* 
* $Log: WakeUpEngine.cpp,v $
* Revision 1.4  2004/05/22 14:28:13  juengel
* First working version.
*
* Revision 1.3  2004/05/19 13:32:39  heinze
* worked at the engine
*
* Revision 1.2  2004/05/03 15:29:48  roefer
* Warnings removed
*
* Revision 1.1  2004/05/03 09:45:01  heinze
* Added WakeUpEngine.
*
*/
