/**
 *  @file PointsWithValidityAndAge.h
 *
 *  Definition of PointsWithValidityAndAge 
 *
 *  @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Juengel</A>
 *  @author <A href=mailto:kspiess@informatik.uni-bremen.de>Kai Spiess</A>
 */ 

#ifndef _PointsWithValidityAndAge_h_
#define _PointsWithValidityAndAge_h_

#include "Tools/FieldDimensions.h"

#include "Tools/Streams/InOut.h"


/** The threshold for voting, minimum of votes */
static const int VOTE_THRESHOLD = 5;


/** 
 *  Represents a set of points on the field. 
 *  It is used to determine the location of other players on the field.
 *  If a new point is added and there are as many points in the set as it can held,
 *  the oldest point is overwritten.
 */
class PointsWithValidityAndAge
{
public:

  /** The maximum number of points in a set */
  enum{MAX_NUMBER_OF_POINTS_WITH_AGE = 600};
  /** The space between grid points*/
  enum{GRID_SPACING = 50};
  /** The number of grid points in x-direction */
  enum{NUMBER_OF_GRID_POINTS_X = (xPosFrontFlags - xPosBackFlags) / GRID_SPACING};
  /** The number of grid points in y-direction */
  enum{NUMBER_OF_GRID_POINTS_Y = (yPosLeftFlags - yPosRightFlags) / GRID_SPACING};

  /** Default-Construktor */
//  PointsWithValidityAndAge();
  /** Constructor */
  PointsWithValidityAndAge(short int sizeOfSet);
  /** Destructor */
  ~PointsWithValidityAndAge();
  
  /** 
   *  Sets the maximum of points in the set
   *  @param size The size of the set
   */
  void setSizeOfSet(int size);

  /** 
   *  Adds a point to the set, if necessary an old point is overwritten.
   *  The distributions (in x and y direction) are also updated.
   *  @param x The x coordinate of the point
   *  @param y The y coordinate of the point
   */
  void add(int x, int y);
  
  /** 
   *  Returns the index of the next position in the set, where a point 
   *  can be added/set.
   *  @return The index of the next position
   */
  short int getIndexOfNextPoint();
  
  /**
   *  Gets the position of an obstacle and stores the coordinates
   *  in x_position and y_position.
   *  The parameter index can range from 0 up to 3.
   *  If no obstacle with that index exists, the return value
   *  is false.
   *  @param index The index of the obstacle
   *  @param x_position A pointer to the varible storing the x coordinat
   *  @param y_position A pointer to the varible storing the y coordinat
   *  @return True, if there is an obstacle at the given index, otherwise false
   */
  bool getPositionOfObstacle(int index, int* x_position, int* y_position);
  
  /**
   *  Calculates the distribution of points for x and y direction
   */
  void calculateXYDistribution();
  
  /**
   *  Finds maxima in the distribution of points for x and y direction
   */
  void findMaximaInXYDistribution();
  
  /**
   *  Votes for hills in the distribution of points.
   *  The votes are sorted in a ranking, best first
   */
  void voteForHills();
  
  /**
   *  Calculates the distance of two given points.
   *  @param x1 The x coordinate of the first point
   *  @param y1 The y coordinate of the first point
   *  @param x2 The x coordinate of the second point
   *  @param y2 The y coordinate of the second point
   */
  int calculateDistance(int x1, int y1, int x2, int y2);
  
  
  // streaming data
  /** The x coordinates of points in the set*/
  short int x[MAX_NUMBER_OF_POINTS_WITH_AGE];
  /** The y coordinates of points in the set*/
  short int y[MAX_NUMBER_OF_POINTS_WITH_AGE];
  /** The number of points in the set*/
  short int numberOfPoints;
  /** The index of the position in the set where the next point can be stores */
  short int indexOfNextPoint;
  /** The size of the set*/
  short int sizeOfSet;
  //end streaming data
  
  
  /** The distribution of points in x direction */
  short int xDistribution[NUMBER_OF_GRID_POINTS_X];
  /** The distribution of points in y direction */
  short int yDistribution[NUMBER_OF_GRID_POINTS_Y];
  /** The number of detected hills in the x distribution */
  short int numberOf_xHills;
  /** The number of detected hills in the y distribution */
  short int numberOf_yHills;
  /** The hills (gridposition of hills) in the x distribution */
  short int xHills[8];
  /** The hills (gridposition of hills) in the y distribution */
  short int yHills[8];
  /** All possible hills in the grid, due to the distributions */
  short int voteHills[8][8];
  /** The x coordinates of the four best votes (generated by voteForHills) */
  short int x_vote[4];
  /** The y coordinates of the four best votes (generated by voteForHills) */
  short int y_vote[4];
  
  /** The number of determined obstacles */
  short int numberOfObstacles;

};

/**
 * Streaming operator that reads a set of PointsWithValidityAndAge from a stream.
 * @param stream The stream from which is read.
 * @param pointsWithValidityAndAge The set of PointsWithValidityAndAge.
 * @return The stream.
 */ 
In& operator>>(In& stream,PointsWithValidityAndAge& pointsWithValidityAndAge);

/**
 * Streaming operator that writes a set of PointsWithValidityAndAge to a stream.
 * @param stream The stream to write on.
 * @param pointsWithValidityAndAge The Set of PointsWithValidityAndAge.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const PointsWithValidityAndAge& pointsWithValidityAndAge);


#endif   //  _PointsWithValidityAndAge_h_


/*
 * Change log :
 * 
 * $Log: PointsWithValidityAndAge.h,v $
 * Revision 1.1  2003/10/06 14:10:15  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1  2003/09/26 11:38:52  juengel
 * - sorted tools
 * - clean-up in DataTypes
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.2  2003/04/15 15:52:12  risler
 * DDD GO 2003 code integrated
 *
 * Revision 1.3  2003/04/13 02:41:57  dthomas
 * modified: raised queue size
 *
 * Revision 1.2  2003/03/31 15:27:00  mkunz
 * more points with validity and age
 * (without validity and age)
 *
 * Revision 1.1  2002/09/10 15:26:39  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.4  2002/07/23 13:32:57  loetzsch
 * new streaming classes
 *
 * removed many #include statements
 *
 * Revision 1.3  2002/06/07 10:15:46  kspiess
 * constructor changed
 *
 * Revision 1.2  2002/06/06 18:37:17  loetzsch
 * increased number of samples for player modelling
 *
 * Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.3  2002/02/08 14:20:33  kspiess
 * Anpassung an die Namenskonventionen
 *
 * Revision 1.2  2002/01/22 11:03:15  kspiess
 * Fehler im Default-Konstruktor behoben und
 * SetSizeOfSet-Methode eingebaut
 *
 * Revision 1.1  2002/01/21 23:18:30  kspiess
 * PointsWithValidityAndAge portiert;
 * wird vom BremenBerlin2001PlayersLocator verwendet
 *
 *
 */
