//VAPoints.h
/**
 *  @file VAPoints.h
 *
 *  Definition of VAPoints 
 *
 *  @author <A href=mailto:mkunz@sim.tu-darmstadt.de.de>Michael Kunz</A>
 */ 

#ifndef _VAPoints_h_
#define _VAPoints_h_

#include "Tools/FieldDimensions.h"
#include "Platform/SystemCall.h"

#include "Tools/Streams/InOut.h"

/**
 * Represents a maximum in the grid.
 */

struct Maximum
{
  int x, y;
  double height;
};
  

/**
 * Represents a point on the field.
 */

class VAPoint
{
  public:
  
  VAPoint() : x(0), y(0), validity(0), relevance(0), timestamp(0) {}
  
  enum{FADE_OUT_TIME = 20000};
  
  int x, y;
  double validity, relevance;
  unsigned long timestamp;
  
  void updateRelevance()
  {
    double age = SystemCall::getTimeSince(timestamp);
    if (age < validity * FADE_OUT_TIME)
    {
//      relevance = (1 - (age / FADE_OUT_TIME)) * validity;
      relevance = validity - (age / FADE_OUT_TIME);
    }
    else
    {
      relevance = 0;
    }
  }
  
  double getRelevance()
  {
    this->updateRelevance();
    return relevance;
  }
};

/** 
 *  Represents a set of points on the field. 
 */
class VAPoints
{
  public:

  /** minimum height of maximum to get maximal validity */
  enum{MIN_HEIGHT_FOR_MAX_VALIDITY = 4};
  /** The space between grid points*/
  enum{GRID_SPACING = 100};
  /** The number of grid points in x-direction */
  enum{NUMBER_OF_GRID_POINTS_X = (xPosFrontFlags - xPosBackFlags) / GRID_SPACING};
  /** The number of grid points in y-direction */
  enum{NUMBER_OF_GRID_POINTS_Y = (yPosLeftFlags - yPosRightFlags) / GRID_SPACING};

  enum{FADE_OUT_TIME2 = 6000};

  /** Constructor */
  VAPoints(int numberOfVAPoints=100, int numberOfMaxima=4);
  /** Destructor */
  ~VAPoints();

  /**
  *  Adds a point to the set, if necessary an old point is overwritten.
  *  @param x The x coordinate of the point
  *  @param y The y coordinate of the point
  *  @param validity The validity of the point
  *  @param timestamp The timestamp of the point
  */
  void addPoint(int x, int y, double validity, unsigned long timestamp);

  /**
  *  Returns the index of the next position in the set, where a point
  *  can be added/set.
  *  @return The index of the next position
  */
  int findInsertPos();

  /**
  *  Searches maxima of points on the field
  */
  void searchMaxima();

  /**
  *  Gets the position of a maximum and stores the coordinates
  *  in x_position and y_position.
  *  The parameter index is limited.
  *  If no maximum with that index exists, the return value
  *  is false.
  *  @param index The index of the maximum
  *  @param x_position A reference to the varible storing the x coordinat
  *  @param y_position A reference to the varible storing the y coordinat
  *  @param validity A reference to the varible storing the validity
  *  @return True, if there is an obstacle at the given index, otherwise false
  */
  bool getMaximum(int index, int& x_position, int& y_position, double& validity);

  /**
  *  Adds the points to the field
  */
  void pointsToField();

  /**
  *  Updates the grid with one point
  *  @param xPos The x coordinat on the field
  *  @param yPos The y coordinat on the field
  *  @param relevance The relevance of the point to add
  */
  void updateGridByPoint(int xPos, int yPos, double relevance);

  /**
  *  Increments a value in the grid
  *  @param xIndex The x index of the grid
  *  @param yIndex The y index of the grid
  *  @param increment The increment for this position in the grid
  */
  void incrementGridPoint(int xIndex, int yIndex, double increment);

  /**
  *  Get indices of the grid for positions on the field.
  *  @param xPos The x coordinat on the field
  *  @param yPos The y coordinat on the field
  *  @param xIndex A reference to the variable storing the x index of the grid
  *  @param yIndex A reference to the variable storing the y index of the grid
  */
  void getGridIndices(int xPos, int yPos, int& xIndex, int& yIndex);

  /**
  *  Get positions on the field for indices of the grid.
  *  @param xIndex The x index of the grid
  *  @param yIndex The y index of the grid
  *  @param xPos A reference to the variable storing the x coordinat on the field
  *  @param yPos A reference to the variable storing the y coordinat on the field
  */
  void getFieldPoints(int xIndex, int yIndex, int& xPos, int& yPos);

  /**
  *  Tests if the specified grid point is a lokal maxima.
  *  @param xIndex The x index of the grid
  *  @param yIndex The y index of the grid
  */
  bool isLokMax(int xIndex, int yIndex);

  /** The Set of Points*/
  VAPoint* points;
  /** The number of points in the set*/
  int numberOfVAPoints;

  /** The grid representing the field */
  double field[NUMBER_OF_GRID_POINTS_X] [NUMBER_OF_GRID_POINTS_Y];
  /** The maxima on field */
  Maximum* maxima;
  /** Number of maxima on field */
  int numberOfMaxima;

  unsigned long timeOfLastMaximaSearch;

};

/**
* Streaming operator that reads a set of VAPoints from a stream.
* @param stream The stream from which is read.
* @param VAPoints The set of VAPoints.
* @return The stream.
*/
In& operator>>(In& stream,VAPoints& VAPoints);

/**
* Streaming operator that writes a set of VAPoints to a stream.
* @param stream The stream to write on.
* @param VAPoints The Set of VAPoints.
* @return The stream.
*/
Out& operator<<(Out& stream, const VAPoints& VAPoints);


#endif   //  _VAPoints_h_


/*
* $Log: VAPoints.h,v $
* Revision 1.1  2003/10/06 14:10:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1  2003/09/26 11:38:52  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.7  2003/05/27 12:48:49  mkunz
* parameter tuning
*
* Revision 1.6  2003/05/27 09:05:50  mkunz
* more flexible number of maxima
* restricted height in field
* some performance hacks
*
* Revision 1.5  2003/05/16 14:49:26  mkunz
* enlarged gridsize
*
* Revision 1.4  2003/05/14 13:21:09  mkunz
* flexible array size
*
* Revision 1.3  2003/05/09 09:54:27  mkunz
* some finetuning
*
* Revision 1.2  2003/05/08 18:05:35  mkunz
* methods completed
*
* Revision 1.1  2003/05/08 16:49:39  mkunz
* class VAPoints added.
* similar to PointsWithValidityAndAge but with validity and age
*
*/
