/**
* @file DefaultRobotStateDetector.cpp
*
* Implementation of class DefaultRobotStateDetector
*
* @author Max Risler
*/

#include "DefaultRobotStateDetector.h"
#include "Platform/SystemCall.h"

DefaultRobotStateDetector::DefaultRobotStateDetector(const RobotStateDetectorInterfaces& interfaces)
: RobotStateDetector(interfaces)
{
  int i;
  
  for (i = 0; i < BodyPercept::numOfSwitches;switchDown[i++]=false);
  
  for (i = 0; i < 6; i++)
  {
    timeSinceLastCollision[i] = 0;
    consecutiveCollisionTime[i] = 0;
  }
  
  remindCollision = 750;

  /****/
  for (i = 0; i < BodyPercept::numOfSwitches;i++)
  {
    buttonPressed[i]=false;
    buttonTime[i]=0;
    buttonPressedTime[i]=0;
    buttonDuration[i]=0;
  }
  anyBackButtonPressed=false;
  anyBackButtonTime=0;
  anyBackButtonPressedTime=0;
  anyBackButtonDuration=0;
  /****/
}

void DefaultRobotStateDetector::execute()
{
  robotState.setFrameNumber(bodyPercept.frameNumber);
  
  calculateCollisionState();
  
  robotState.setState(
    (RobotState::States)bodyPercept.getState());
  
  robotState.setTailPosition(
    (RobotState::TailPositions)bodyPercept.getTailPosition());
  
  robotState.setMouthState(
    (RobotState::MouthStates)bodyPercept.getMouthState());

  robotState.setForelegOpeningAngle(bodyPercept.getForelegOpeningAngle());
  
  robotState.acceleration=bodyPercept.acceleration;

//  robotState.setForelegOpeningAngle(bodyPercept.

  int i;
  for (i = 0; i < BodyPercept::numOfSwitches; i++) 
  {
    if ((bodyPercept.getSwitches() & (1<<i)) > 0)
    {
      if (!switchDown[i]) 
      {
        // switch pressed
        switchDown[i] = true;
        switchDownTime[i] = SystemCall::getCurrentSystemTime();

      /***/
      buttonPressed[i]=true;  //Button pressed
      buttonTime[i]=SystemCall::getCurrentSystemTime(); //Aktuelle Zeit
      buttonPressedTime[i]=0; //Later: CurrentTime-buttonTime, buttonPressedTime started
      buttonDuration[i]=0;
      if (getRobotConfiguration().getRobotDesign() == RobotDesign::ERS7)
      {
        switch(i)
        {
        case BodyPercept::back:
        case BodyPercept::backBack:
        case BodyPercept::backFront:
         {
           if (anyBackButtonPressed=false)
           {
             anyBackButtonPressed=true;
             anyBackButtonTime=SystemCall::getCurrentSystemTime();
             anyBackButtonPressedTime=0;
             anyBackButtonDuration=0;
           }
           break;
         }
        }
      }
      /***/

        switch (i)
        {
        case BodyPercept::back:
          setMessage(RobotState::backDown,0);
          break;
        case BodyPercept::backFront:
          setMessage(RobotState::backFrontDown,0);
          break;
        case BodyPercept::backBack:
          setMessage(RobotState::backBackDown,0);
          break;
        case BodyPercept::head:
          setMessage(RobotState::headDown,0);
          break;
        case BodyPercept::headFront:
          setMessage(RobotState::headFrontDown,0);
          break;
        case BodyPercept::headBack:
          setMessage(RobotState::headBackDown,0);
          break;
        case BodyPercept::chin:
          setMessage(RobotState::chin,0);
          break;
        case BodyPercept::tailUp:
          setMessage(RobotState::tailUp,0);
          break;
        case BodyPercept::tailDown:
          setMessage(RobotState::tailDown,0);
          break;
        case BodyPercept::tailLeft:
          setMessage(RobotState::tailLeft,0);
          break;
        case BodyPercept::tailRight:
          setMessage(RobotState::tailRight,0);
          break;
        case BodyPercept::mouth:
          setMessage(RobotState::mouth,0);
          break;
        }
      }
    } else { 
      if (switchDown[i]) 
      {
        // switch released
        switchDown[i] = false;

        /***/
        buttonPressed[i]=false;  //Button released
        buttonPressedTime[i]=SystemCall::getCurrentSystemTime()-buttonTime[i]; //how long the button was pressed
        buttonTime[i]=SystemCall::getCurrentSystemTime(); //when the button was pressed
        buttonDuration[i]=0; // And it starts again
        if ((anyBackButtonPressed=true) && (getRobotConfiguration().getRobotDesign() == RobotDesign::ERS7))
        {
          if ((buttonPressed[BodyPercept::back]=false)&&(buttonPressed[BodyPercept::backFront]=false)&&(buttonPressed[BodyPercept::backBack]=false))
          {
            anyBackButtonPressed=false;
            anyBackButtonPressedTime=SystemCall::getCurrentSystemTime()-anyBackButtonTime;
            anyBackButtonTime=SystemCall::getCurrentSystemTime();
            anyBackButtonDuration=0;
          }
        }
        /***/

        // ignore button, too short
        if (SystemCall::getTimeSince(switchDownTime[i])>50)
        {
          if(getRobotConfiguration().getRobotDesign() == RobotDesign::ERS210)
          {
            switch (i) 
            {
            case BodyPercept::back:
              if (switchDown[BodyPercept::headFront])
                setMessage(RobotState::headFrontAndBackPressed,SystemCall::getTimeSince(switchDownTime[i]));
              else if (switchDown[BodyPercept::headBack])
                setMessage(RobotState::headBackAndBackPressed,SystemCall::getTimeSince(switchDownTime[i]));
              else 
                setMessage(RobotState::backPressed,SystemCall::getTimeSince(switchDownTime[i]));
              break;
            case BodyPercept::headFront:
              if (switchDown[BodyPercept::back])
                setMessage(RobotState::headFrontAndBackPressed,SystemCall::getTimeSince(switchDownTime[i]));
              else
                setMessage(RobotState::headFrontPressed,SystemCall::getTimeSince(switchDownTime[i]));
              break;
            case BodyPercept::headBack:
              if (switchDown[BodyPercept::back])
                setMessage(RobotState::headBackAndBackPressed,SystemCall::getTimeSince(switchDownTime[i]));
              else
                setMessage(RobotState::headBackPressed,SystemCall::getTimeSince(switchDownTime[i]));
              break;
            case BodyPercept::chin:
              setMessage(RobotState::chinPressed,SystemCall::getTimeSince(switchDownTime[i]));
              break;
            case BodyPercept::tailUp:
              if (switchDown[BodyPercept::back])
                setMessage(RobotState::tailUpAndBackPressed,SystemCall::getTimeSince(switchDownTime[i]));
              break;
            case BodyPercept::tailDown:
              if (switchDown[BodyPercept::back])
                setMessage(RobotState::tailDownAndBackPressed,SystemCall::getTimeSince(switchDownTime[i]));
              break;
            }
          }// if(getRobotConfiguration().getRobotDesign() == RobotDesign::ERS210)
          else
          {
            switch (i) 
            {
            case BodyPercept::back:
              if (switchDown[BodyPercept::headFront])
                setMessage(RobotState::headFrontAndBackPressed,SystemCall::getTimeSince(switchDownTime[i]));
              else if (switchDown[BodyPercept::headBack])
                setMessage(RobotState::headBackAndBackPressed,SystemCall::getTimeSince(switchDownTime[i]));
              else 
                setMessage(RobotState::backPressed,SystemCall::getTimeSince(switchDownTime[i]));
              break;
            case BodyPercept::head:
                setMessage(RobotState::headPressed,SystemCall::getTimeSince(switchDownTime[i]));
              break;
            case BodyPercept::backFront:
                setMessage(RobotState::backFrontPressed,SystemCall::getTimeSince(switchDownTime[i]));
              break;
            case BodyPercept::backBack:
                setMessage(RobotState::backBackPressed,SystemCall::getTimeSince(switchDownTime[i]));
              break;
            case BodyPercept::chin:
              setMessage(RobotState::chinPressed,SystemCall::getTimeSince(switchDownTime[i]));
              break;
            case BodyPercept::tailUp:
              if (switchDown[BodyPercept::back])
                setMessage(RobotState::tailUpAndBackPressed,SystemCall::getTimeSince(switchDownTime[i]));
              break;
            case BodyPercept::tailDown:
              if (switchDown[BodyPercept::back])
                setMessage(RobotState::tailDownAndBackPressed,SystemCall::getTimeSince(switchDownTime[i]));
              break;
            }
          }//else (ERS7)
        }
      }
    }
  }
  /****/
  for (i = 0; i < BodyPercept::numOfSwitches; i++)  // update my duration time
  {
    buttonDuration[i]=SystemCall::getTimeSince(buttonTime[i]);
/*  
    OUTPUT(idText, text, "ButtonDuration: " << buttonDuration[i]);
    OUTPUT(idText, text, "ButtonTime: " << buttonTime[i]);
    OUTPUT(idText, text, "ButtonPressed: " << buttonPressed[i]);
    OUTPUT(idText, text, "ButtonPressedTime: " << buttonPressedTime[i]);
*/  
  }
    anyBackButtonDuration=SystemCall::getTimeSince(anyBackButtonTime);
/*
    OUTPUT(idText, text, "anyBackButtonDuration: " << anyBackButtonDuration);
    OUTPUT(idText, text, "--------------- ");
*/
  /****/

  // update duration if message is button down
  if (
    (robotState.getSwitches() == RobotState::backDown && switchDown[BodyPercept::back]) ||
    (robotState.getSwitches() == RobotState::backFrontDown && switchDown[BodyPercept::backFront]) ||
    (robotState.getSwitches() == RobotState::backBackDown && switchDown[BodyPercept::backBack]) ||
    (robotState.getSwitches() == RobotState::headDown && switchDown[BodyPercept::head]) ||
    (robotState.getSwitches() == RobotState::headFrontDown && switchDown[BodyPercept::headFront]) ||
    (robotState.getSwitches() == RobotState::headBackDown && switchDown[BodyPercept::headBack]) ||
    (robotState.getSwitches() == RobotState::chin && switchDown[BodyPercept::chin]) ||
    (robotState.getSwitches() == RobotState::tailUp && switchDown[BodyPercept::tailUp]) ||
    (robotState.getSwitches() == RobotState::tailDown && switchDown[BodyPercept::tailDown]) ||
    (robotState.getSwitches() == RobotState::tailLeft && switchDown[BodyPercept::tailLeft]) ||
    (robotState.getSwitches() == RobotState::tailRight && switchDown[BodyPercept::tailRight]) ||
    (robotState.getSwitches() == RobotState::mouth && switchDown[BodyPercept::mouth]))
    robotState.setSwitchDuration(SystemCall::getTimeSince(robotState.getSwitchTimestamp()));
}

void DefaultRobotStateDetector::setMessage (RobotState::Switches s,long duration)
{
  robotState.setSwitches(s);
  robotState.setSwitchTimestamp(SystemCall::getCurrentSystemTime());
  robotState.setSwitchDuration(duration);
  //OUTPUT(idText, "DefaultRobotStateDetector : switch message : "<<RobotState::getSwitchName(s) << " for "<<duration<<" ms");
}

void DefaultRobotStateDetector::calculateCollisionState()
{
  // collisionTime  1. Component 0 fl, 1 fr, 2 hl, 3 hr, 4 head, 5 anyone of them
  //                2. Component two previous collision times
  

  long time = SystemCall::getCurrentSystemTime();
  
  //setze alle states auf 0!!!
  
  robotState.setCollisionFrontLeft(false);
  robotState.setCollisionFrontRight(false);
  robotState.setCollisionHindLeft(false);
  robotState.setCollisionHindRight(false);
  robotState.setCollisionHead(false);
  robotState.setCollisionAggregate(false);
  
  
  if ((time - timeSinceLastCollision[0]) >  remindCollision)
  {
    consecutiveCollisionTime[0] = time;
    robotState.setConsecutiveCollisionTimeFrontLeft(0);
  }

  if ((time - timeSinceLastCollision[1]) >  remindCollision)
  {
    consecutiveCollisionTime[1] = time;
    robotState.setConsecutiveCollisionTimeFrontRight(0);
  }

  if ((time - timeSinceLastCollision[2]) >  remindCollision)
  {
    consecutiveCollisionTime[2] = time;
    robotState.setConsecutiveCollisionTimeHindLeft(0);
  }

  if ((time - timeSinceLastCollision[3]) >  remindCollision)
  {
    consecutiveCollisionTime[3] = time;
    robotState.setConsecutiveCollisionTimeHindRight(0);
  }

  if ((time - timeSinceLastCollision[4]) >  remindCollision)
  {
    consecutiveCollisionTime[4] = time;
    robotState.setConsecutiveCollisionTimeHead(0);
  }

  if ((time - timeSinceLastCollision[5]) >  remindCollision)
  {
    consecutiveCollisionTime[5] = time;
    robotState.setConsecutiveCollisionTimeAggregate(0);
  }
  
      if (collisionPercept.getCollisionFrontLeft())
      {
        robotState.setCollisionFrontLeft(true);
        if ((time-timeSinceLastCollision[0]) > remindCollision)
        {
          consecutiveCollisionTime[0] = time; // first collision
        }
        robotState.setConsecutiveCollisionTimeFrontLeft(time-consecutiveCollisionTime[0]);
        timeSinceLastCollision[0] = time; //last collision
      }
      
      if (collisionPercept.getCollisionFrontRight()) 
      {
        robotState.setCollisionFrontRight(true);
        if ((time-timeSinceLastCollision[1]) > remindCollision)
        {
          consecutiveCollisionTime[1] = time;
        }
        robotState.setConsecutiveCollisionTimeFrontRight(time-consecutiveCollisionTime[1]);
        timeSinceLastCollision[1] = time;
      }
      
      if (collisionPercept.getCollisionHindLeft()) 
      {
        robotState.setCollisionHindLeft(true);
        if ((time-timeSinceLastCollision[2]) > remindCollision)
        {
          consecutiveCollisionTime[2] = time;
        }
        robotState.setConsecutiveCollisionTimeHindLeft(time-consecutiveCollisionTime[2]);
        timeSinceLastCollision[2] = time;
      }
      
      if (collisionPercept.getCollisionHindRight()) 
      {
        robotState.setCollisionHindRight(true);
        if ((time-timeSinceLastCollision[3]) > remindCollision)
        {
          consecutiveCollisionTime[3] = time;
        }
        robotState.setConsecutiveCollisionTimeHindRight(time-consecutiveCollisionTime[3]);
        timeSinceLastCollision[3] = time;
      }
      
      if (collisionPercept.getCollisionHead()) 
      {
        robotState.setCollisionHead(true);
        if ((time-timeSinceLastCollision[4]) > remindCollision)
        {
          consecutiveCollisionTime[4] = time;
        }
        robotState.setConsecutiveCollisionTimeHead(time-consecutiveCollisionTime[4]);
        timeSinceLastCollision[4] = time;
      }
      
      if (collisionPercept.getCollisionAggregate()) 
      {
        robotState.setCollisionAggregate(true);
        if ((time-timeSinceLastCollision[5]) > remindCollision)
        {
          consecutiveCollisionTime[5] = time;
        }
        robotState.setConsecutiveCollisionTimeAggregate(time-consecutiveCollisionTime[5]);
        timeSinceLastCollision[5] = time;
      }
}

/*
* Change log :
* 
* $Log: DefaultRobotStateDetector.cpp,v $
* Revision 1.8  2004/05/11 08:00:19  juengel
* Fixed VC6 compiler errors.
*
* Revision 1.7  2004/05/11 06:59:05  brueckne
* added new way to know how long a button is/was pressed/released (not ready yet)
*
* Revision 1.6  2004/03/08 02:11:44  roefer
* Interfaces should be const
*
* Revision 1.5  2004/02/10 11:12:31  dueffert
* acceleration added
*
* Revision 1.4  2004/01/24 11:32:32  juengel
* Added ERS7 switches (head, backFront, backBack).
*
* Revision 1.3  2003/11/14 19:02:26  goehring
* frameNumber added
*
* Revision 1.2  2003/10/16 12:58:40  goehring
* remindCollision reduced for CollisionDetection
*
* Revision 1.1  2003/10/06 14:10:14  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/26 21:23:20  loetzsch
* renamed class JointState to CollisionPercept
*
* Revision 1.2  2003/07/06 12:05:31  schumann
* added foreleg opening angle for ball challenge
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.10  2003/06/21 13:07:36  goehring
* CollisionStateSymbol modelling changed
*
* Revision 1.9  2003/06/20 16:57:46  goehring
* CollisionDetectorValues improved
*
* Revision 1.8  2003/06/20 14:50:07  risler
* numOfSwitches added
*
* Revision 1.7  2003/06/20 13:27:19  risler
* added tailLeft tailRight messages
*
* Revision 1.6  2003/06/20 10:30:37  goehring
* calculateCollisionState implemented
*
* Revision 1.5  2003/04/01 22:40:44  cesarz
* added mouth states
*
* Revision 1.4  2003/03/24 18:39:46  loetzsch
* message backPressedAndTailUp and backPressedAndTailDown removed
*
* Revision 1.3  2003/01/30 11:29:24  juengel
* Added tailPosition to bodyPercept
*
* Revision 1.2  2002/09/12 09:45:58  juengel
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:15  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.4  2002/08/30 14:33:38  dueffert
* removed unused includes
*
* Revision 1.3  2002/05/15 10:52:42  risler
* no message
*
* Revision 1.2  2002/05/15 08:09:10  risler
* added button down messages
*
* Revision 1.1.1.1  2002/05/10 12:40:15  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.6  2002/03/28 16:55:58  risler
* RobotStateDetector receives BodyPercept instead of PerceptCollection
* added switch duration in RobotStateDetector
*
* Revision 1.5  2002/03/28 15:21:40  risler
* removed output
*
* Revision 1.4  2002/03/28 15:19:20  risler
* implemented switch messages in RobotStateDetector
*
* Revision 1.3  2002/02/23 16:33:07  risler
* finished GetupEngine
*
* Revision 1.2  2001/12/10 17:47:06  risler
* change log added
*
*/

