/**
 * @file Modules/SelfLocator/LinesSelfLocator.cpp
 * 
 * This file contains a class for self-localization based on the Monte Carlo approach using field lines.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */

#include "Platform/GTAssert.h"
#include "LinesSelfLocator.h"
#include "Platform/SystemCall.h"
#include "Tools/Player.h"
#include "Tools/Debugging/Debugging.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Streams/InStreams.h"
#include "Tools/Debugging/GenericDebugData.h"

double LinesSelfLocator::paramUp = 0.01, // step size for increasing qualities
       LinesSelfLocator::paramDown = 0.005, // step size for decreasing qualities
       LinesSelfLocator::paramDelay = 3, // factor
       LinesSelfLocator::paramHeight = 150, // hypothetic height of head
       LinesSelfLocator::paramZ = 10, // sharpness of Gauss for rotational errors 
       LinesSelfLocator::paramY = 2, // sharpness of Gauss for translational errors
       LinesSelfLocator::paramTrans = 100, // maximum translational shift if quality is bad
       LinesSelfLocator::paramRot = 0.5; // maximum rotational shift if quality is bad

LinesSelfLocator::Sample::Sample() 
{
  for(int i = 0; i < LinesPercept::numberOfLineTypes - 1; ++i)
    quality[i] = 0.5;
  probability = 2;
}

LinesSelfLocator::Sample::Sample(const Pose2D& pose,const double* quality)
: PoseSample(pose)
{
  for(int i = 0; i < LinesPercept::numberOfLineTypes - 1; ++i)
    this->quality[i] = 2;
  probability = 2;
}

void LinesSelfLocator::Sample::updateQuality(const double* average)
{
  probability = 1;
  for(int i = 0; i < LinesPercept::numberOfLineTypes - 1; ++i)
    probability *= quality[i] == 2 ? average[i] - paramDelay * paramUp : quality[i];
}

void LinesSelfLocator::Sample::setProbability(LinesPercept::LineType type,double value)
{
  double& q = quality[type == LinesPercept::skyblueGoal ? LinesPercept::yellowGoal : type];
  if(q == 2)
    q = value - paramDelay * paramUp;
  else if(value < q)
    if(value < q - paramDown)
      q -= paramDown;
    else
      q = value;
  else 
    if(value > q + paramUp)
      q += paramUp;
    else
      q = value;
}

LinesSelfLocator::LinesSelfLocator(const SelfLocatorInterfaces& interfaces)
: SelfLocator(interfaces)
{
  int i;
  for(i = 0; i < LinesPercept::numberOfLineTypes - 1; ++i)
    average[i] = 0.5;
  for(i = 0; i < SAMPLES_MAX; ++i)
    (Pose2D&) sampleSet[i] = field.randomPose();
  numberOfTypes = 0;
  sensorUpdated = false;
  timeStamp = 0;
  speed = 0;
}

void LinesSelfLocator::execute()
{
  robotPose.setFrameNumber(linesPercept.frameNumber);
  Pose2D odometry = odometryData - lastOdometry;
  lastOdometry = odometryData;
  if(SystemCall::getTimeSince(timeStamp) > 500)
  {
    speed = (odometryData - lastOdometry2).translation.abs() / SystemCall::getTimeSince(timeStamp) * 1000.0;
    lastOdometry2 = odometryData;
    timeStamp = SystemCall::getCurrentSystemTime();
  }
  paramZ = 10 - 9 * speed / 200;
  if(speed > 150)
    paramTrans = 0;
  else
    paramTrans = 50 - 40 * speed / 150;
  // First step: action update
  updateByOdometry(odometry,sensorUpdated);

  // Second step: observation update
  numberOfTypes = 0;
  sensorUpdated = false;
  if(cameraMatrix.isValid)
  {
    int i;
    for(i = 0; i < LinesPercept::numberOfLineTypes; ++i)
      if(linesPercept.numberOfPoints[i])
      {
        if(i < 2) // ignore goals here
          updateByPoint(linesPercept.points[i][rand() % linesPercept.numberOfPoints[i]],
                        LinesPercept::LineType(i));
        else
          types[numberOfTypes++] = LinesPercept::LineType(i);
        sensorUpdated = true;
      }

    if(linesPercept.numberOfPoints[LinesPercept::yellowGoal] ||
       linesPercept.numberOfPoints[LinesPercept::skyblueGoal])
    { // This works if only one goal or both were seen
      LinesPercept::LineType select = (random() + 1e-6) * linesPercept.numberOfPoints[LinesPercept::yellowGoal] >
                                      linesPercept.numberOfPoints[LinesPercept::skyblueGoal]
                                      ? LinesPercept::yellowGoal
                                      : LinesPercept::skyblueGoal;
      updateByPoint(linesPercept.points[select][rand() % linesPercept.numberOfPoints[select]],
                    select);
    }
    
    if(sensorUpdated)
      resample();
  }

  Pose2D pose;
  double validity;
  calcPose(pose,validity);
  robotPose.setPose(pose);
  robotPose.setValidity(validity);
  sampleSet.link(selfLocatorSamples);

  DEBUG_DRAWING_FINISHED(selfLocatorField); 
  DEBUG_DRAWING_FINISHED(selfLocator); 
}

static double fmax(double a,double b)
{
  return a> b ? a : b;
}

void LinesSelfLocator::updateByOdometry(const Pose2D& odometry,
                                        bool noise)
{     
  double rotNoise = noise ? paramRot : 0,
         xError = fmax(fabs(odometry.translation.x) * 0.1,fabs(odometry.translation.y) * 0.02),
         yError = fmax(fabs(odometry.translation.y) * 0.1,fabs(odometry.translation.x) * 0.02),
         rotError = fmax(odometry.translation.abs() * 0.002,fabs(odometry.getAngle()) * 0.2);
  for(int i = 0; i < SAMPLES_MAX; ++i)
  {
    Sample& s = sampleSet[i];
    s += odometry;
    if(s.isValid())
      s += Pose2D(((random() * 2 - 1) * fmax(rotError,rotNoise * (1 - s.getQuality()))),
                  Vector2<double>((random() * 2 - 1) * xError,
                                  (random() * 2 - 1) * yError));
  } 
}

void LinesSelfLocator::updateByPoint(const Vector2<int>& point,LinesPercept::LineType type)
{
  COMPLEX_DRAWING(selfLocator,draw(point,type));  
  
  int index = type;
  if(getPlayer().getTeamColor() == Player::blue && type >= LinesPercept::yellowGoal) 
    index = LinesPercept::yellowGoal + LinesPercept::skyblueGoal - type;
  double d = point.abs();
  double zObs = atan2(paramHeight /*cameraMatrix.translation.z*/,d),
         yObs = atan2((double)point.y,(double)point.x);
  Pose2D point2(point);
  for(int i = 0; i < SAMPLES_MAX; ++i)
  {
    Sample& s = sampleSet[i];
    Vector2<double> v = (s + point2).translation;
    Vector2<double> vMin = observationTable[index].getClosestPoint(v);
    if(vMin.x != 1000000)
    {
      Vector2<double> diff = vMin - v;
      v = (Pose2D(vMin) - s).translation;
      double zModel = atan2(paramHeight /*cameraMatrix.translation.z*/,v.abs()),
             yModel = atan2(v.y,v.x);
      if(d < 1500 && diff.abs() < 1500)
      {
        double p = pow(paramTrans / (paramTrans + d / 10),2);
        s.translation += diff * fmax(0.2,1 - s.getQuality()) * p;
      }
      s.setProbability(type,sigmoid((zModel - zObs) * paramZ) * sigmoid((yModel - yObs) * paramY));
    }
    else
      s.setProbability(type,0.000001);
  }
}

void LinesSelfLocator::resample()
{
  int i,j,
      count[LinesPercept::numberOfLineTypes - 1];

  for(i = 0; i < LinesPercept::numberOfLineTypes - 1; ++i)
  {
    average[i] = 0;
    count[i] = 0;
  }

  for(i = 0; i < SAMPLES_MAX; ++i)
  {
    Sample& s = sampleSet[i];
    for(j = 0; j < LinesPercept::numberOfLineTypes - 1; ++j)
      if(s.quality[j] != 2)
      {
        average[j] += s.quality[j];
        ++count[j];
      }
  }

  double average2 = 1;
  for(i = 0; i < LinesPercept::numberOfLineTypes - 1; ++i)
  {
    if(count[i])
      average[i] /= count[i];
    else
      average[i] = 0.5;
    average2 *= average[i];
  }

  // move the quality of all samples towards the probability
  for(i = 0; i < SAMPLES_MAX; ++i)
    sampleSet[i].updateQuality(average);
    
  // swap sample arrays
  Sample* oldSet = sampleSet.swap();

  double probSum = 0;
  for(i = 0; i < SAMPLES_MAX; ++i)
    probSum += oldSet[i].getQuality();
  double prob2Index = (double) SAMPLES_MAX / probSum;

  double indexSum = 0;
  j = 0;
  for(i = 0; i < SAMPLES_MAX; ++i)
  {
    indexSum += oldSet[i].getQuality() * prob2Index;
    while(j < SAMPLES_MAX && j < indexSum - 0.5)
      sampleSet[j++] = oldSet[i];
  }

  for(i = 0; i < SAMPLES_MAX; ++i)
  {
    Sample& s = sampleSet[i];
    // if quality is too low, select a new random pose for the sample
    if(numberOfTypes && random() * average2 > s.getQuality())
    {
      Pose2D t = getTemplate();
      s.rotation = t.rotation;
      s.translation += (t.translation - s.translation) * 50 / (50 + speed);
    }
    else
    {
      COMPLEX_DRAWING(selfLocatorField,draw(s,Drawings::black));
    }
  }
}
  
void LinesSelfLocator::calcPose(Pose2D& pose,double& validity)
{
  Cell cells[GRID_MAX][GRID_MAX][GRID_MAX];
  double width = field.x.getSize(),
         height = field.y.getSize();

  // attach the samples to the corresponding grid cells
  for(int i = 0; i < SAMPLES_MAX; ++i)
  {
    Sample& p = sampleSet[i];
    if(p.isValid())
    {
      int r = static_cast<int>((p.getAngle() / pi2 + 0.5) * GRID_MAX),
          y = static_cast<int>((p.translation.y / height + 0.5) * GRID_MAX),
          x = static_cast<int>((p.translation.x / width + 0.5) * GRID_MAX);
      if(x < 0)
        x = 0;
      else if(x >= GRID_MAX)
        x = GRID_MAX-1;
      if(y < 0)
        y = 0;
      else if(y >= GRID_MAX)
        y = GRID_MAX-1;
      if(r < 0)
        r = GRID_MAX-1;
      else if(r >= GRID_MAX)
        r = 0;
      Cell& c = cells[r][y][x];
      p.next = c.first;
      c.first = &p;
      ++c.count;
    }
  }  
  
  // determine the 2x2x2 sub-cube that contains most samples
  int rMax = 0,
      yMax = 0,
      xMax = 0,
      countMax = 0,
      r;
  for(r = 0; r < GRID_MAX; ++r)
  {
    int rNext = (r + 1) % GRID_MAX;
    for(int y = 0; y < GRID_MAX - 1; ++y)
      for(int x = 0; x < GRID_MAX - 1; ++x)
      {
        int count = cells[r][y][x].count +
                    cells[r][y][x+1].count +
                    cells[r][y+1][x].count +
                    cells[r][y+1][x+1].count +
                    cells[rNext][y][x].count +
                    cells[rNext][y][x+1].count +
                    cells[rNext][y+1][x].count +
                    cells[rNext][y+1][x+1].count;
        if(count > countMax)
        {
          countMax = count;
          rMax = r;
          yMax = y;
          xMax = x;
        }
      }
  }

  if(countMax > 0)
  {
    // determine the average pose of all samples in the winner sub-cube
    double xSum = 0,
                  ySum = 0,
                  cosSum = 0,
                  sinSum = 0,
                  qualitySum = 0;
    for(r = 0; r < 2; ++r)
    {
      int r2 = (rMax + r) % GRID_MAX;
      for(int y = 0; y < 2; ++y)
        for(int x = 0; x < 2; ++x)
        {
          for(Sample* p = cells[r2][yMax + y][xMax + x].first; p; p = p->next)
          {
            xSum += p->translation.x * p->getQuality();
            ySum += p->translation.y * p->getQuality();
            cosSum += p->getCos() * p->getQuality();
            sinSum += p->getSin() * p->getQuality();
            qualitySum += p->getQuality();
          }
        }
    }  
    if(qualitySum)
    {
      pose = Pose2D(atan2(sinSum,cosSum),
                            Vector2<double>(xSum / qualitySum,ySum / qualitySum));
      validity = qualitySum / SAMPLES_MAX;
      double diff = field.clip(pose.translation);
      if(diff > 1)
        validity /= sqrt(diff);
      return;
    }
  }
  validity = 0;
}

///////////////////////////////////////////////////////////////////////
// Template samples

LinesSelfLocator::Sample LinesSelfLocator::getTemplate() const
{
  LinesPercept::LineType type = types[rand() % numberOfTypes];
  const Vector2<int>& point = linesPercept.points[type][rand() % linesPercept.numberOfPoints[type]];
  int index = type;
  if(getPlayer().getTeamColor() == Player::blue && type >= LinesPercept::yellowGoal) 
    index = LinesPercept::yellowGoal + LinesPercept::skyblueGoal - type;
  Pose2D t = templateTable[index].sample(point.abs()) + Pose2D(-atan2((double)point.y,(double)point.x));
  static const Drawings::Color color[4] =
  { 
    Drawings::red,
    Drawings::green,
    Drawings::yellow,
    Drawings::skyblue
  };
  COMPLEX_DRAWING(selfLocatorField,draw(t,color[type]));
  Pose2D pose = Pose2D::random(Range<double>(t.translation.x - 50,
                                             t.translation.x + 50),
                               Range<double>(t.translation.y - 50,
                                             t.translation.y + 50),
                               Range<double>(t.getAngle() - 0.1,
                                             t.getAngle() + 0.1));
  return Sample(pose,average);
}

///////////////////////////////////////////////////////////////////////
// Debug drawing

void LinesSelfLocator::draw(const Pose2D& pose,Drawings::Color color) const
{ 
  Pose2D p = pose + Pose2D(-100,0);
  LINE(selfLocatorField,
       pose.translation.x,
       pose.translation.y,
       p.translation.x,
       p.translation.y,
       1,
       Drawings::ps_solid,
       color);
  p = pose + Pose2D(-40,-40);
  LINE(selfLocatorField,
       pose.translation.x,
       pose.translation.y,
       p.translation.x,
       p.translation.y,
       1,
       Drawings::ps_solid,
       color);
  p = pose + Pose2D(-40,40);
  LINE(selfLocatorField,
       pose.translation.x,
       pose.translation.y,
       p.translation.x,
       p.translation.y,
       1,
       Drawings::ps_solid,
       color);
}

void LinesSelfLocator::draw(const Vector2<int>& point,LinesPercept::LineType type) const
{ 
  static const Drawings::Color colors[] =
  {
    Drawings::red,
    Drawings::green,
    Drawings::yellow,
    Drawings::skyblue
  };

  CameraInfo cameraInfo;

  Vector2<int> p;
  Geometry::calculatePointInImage(
    point,
    cameraMatrix, cameraInfo,
    p);
  CIRCLE(selfLocator,p.x,p.y,3,1,Drawings::ps_solid,colors[type]);
}

bool LinesSelfLocator::handleMessage(InMessage& message)
{
  switch(message.getMessageID())
  {
    case idLinesSelfLocatorParameters:
      GenericDebugData d;
      message.bin >> d;
      paramUp = d.data[0];
      paramDown = d.data[1];
      paramDelay = d.data[2];
      paramHeight = d.data[3];
      paramZ = d.data[4];
      paramY = d.data[5];
      paramTrans = d.data[6];
      paramRot = d.data[7];
      return true;
  }
  return false;
}

/*
 * Change log :
 * 
 * $Log: LinesSelfLocator.cpp,v $
 * Revision 1.6  2004/03/08 02:11:47  roefer
 * Interfaces should be const
 *
 * Revision 1.5  2004/03/08 01:09:33  roefer
 * Use of LinesTables restructured
 *
 * Revision 1.4  2003/12/15 11:49:06  juengel
 * Introduced CameraInfo
 *
 * Revision 1.3  2003/11/14 19:02:26  goehring
 * frameNumber added
 *
 * Revision 1.2  2003/11/13 16:47:38  goehring
 * frameNumber added
 *
 * Revision 1.1  2003/10/06 14:10:14  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.4  2003/09/26 11:38:52  juengel
 * - sorted tools
 * - clean-up in DataTypes
 *
 * Revision 1.3  2003/09/01 10:20:11  juengel
 * DebugDrawings clean-up 2
 * DebugImages clean-up
 * MessageIDs clean-up
 * Stopwatch clean-up
 *
 * Revision 1.2  2003/07/06 12:17:12  roefer
 * Challenge 2 finished, some parts not checked in
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.27  2003/05/24 21:09:10  roefer
 * Sensor resetting improved
 *
 * Revision 1.26  2003/05/22 07:53:05  roefer
 * GT2003SelfLocator added
 *
 * Revision 1.25  2003/05/12 14:08:40  brunn
 * renamed selfLocationSampleSetProxy to selfLocatorSamples
 * "Ha, glad am I that no one knew that Rumpelstiltskin I am styled"
 *
 * Revision 1.24  2003/05/12 12:28:10  brunn
 * renamed sampleSetProxy to selfLocationSampleSetProxy
 * added selfLocationSampleSetProxy to BehaviourControl-Interfaces
 *
 * Revision 1.23  2003/05/12 12:03:37  roefer
 * New penalty area size, corrected sample poses for blue players
 *
 * Revision 1.22  2003/05/08 23:52:24  roefer
 * SampleSet and SampleSetProxy added
 *
 * Revision 1.21  2003/04/27 08:14:02  roefer
 * static vars -> static const vars
 *
 * Revision 1.20  2003/04/16 07:00:16  roefer
 * Bremen GO checkin
 *
 * Revision 1.22  2003/04/11 07:20:54  roefer
 * Bug fixed!!
 *
 * Revision 1.21  2003/04/10 23:57:59  roefer
 * no message
 *
 * Revision 1.20  2003/04/10 21:38:35  roefer
 * Improvements in self localization
 *
 * Revision 1.19  2003/04/05 09:57:31  roefer
 * Correct position also for blue players
 *
 * Revision 1.18  2003/04/01 17:15:02  roefer
 * Observation is only used if CameraMatrix is valid
 *
 * Revision 1.17  2003/03/22 18:11:24  roefer
 * Warnings removed
 *
 * Revision 1.16  2003/03/22 11:07:02  roefer
 * Only goals are used to generate candidate positions
 *
 * Revision 1.15  2003/02/21 22:20:12  roefer
 * LinesSelfLocator is working
 *
 * Revision 1.14  2003/02/19 15:01:23  roefer
 * LinesPerceptor2 added
 *
 * Revision 1.13  2003/02/17 10:55:59  dueffert
 * greenhills backport
 *
 * Revision 1.12  2003/01/15 13:47:28  roefer
 * Progress in LinesSelfLocator, new debug drawing
 *
 * Revision 1.11  2002/12/17 20:58:38  roefer
 * COMPLEX_DRAWING changed
 *
 * Revision 1.10  2002/12/15 18:15:13  roefer
 * Using less points per cycle
 *
 * Revision 1.9  2002/12/14 21:21:42  roefer
 * Drawing simplified
 *
 * Revision 1.8  2002/12/12 22:09:41  roefer
 * Progress in LinesSelfLocator
 *
 * Revision 1.7  2002/12/04 12:20:37  juengel
 * Changed the data type of the points in the LinesPercept from Vector2<double> to Vector2<int>.
 *
 * Revision 1.6  2002/11/12 10:49:02  juengel
 * New debug drawing macros - second step
 * -moved /Tools/Debugging/PaintMethods.h and . cpp
 *  to /Visualization/DrawingMethods.h and .cpp
 * -moved DebugDrawing.h and .cpp from /Tools/Debugging/
 *  to /Visualization
 *
 * Revision 1.5  2002/09/23 13:54:49  risler
 * abs replaced by fabs/labs
 *
 * Revision 1.4  2002/09/22 18:40:53  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.3  2002/09/17 23:55:21  loetzsch
 * - unraveled several datatypes
 * - changed the WATCH macro
 * - completed the process restructuring
 *
 * Revision 1.2  2002/09/12 12:24:09  juengel
 * continued change of module/solution mechanisms
 *
 * Revision 1.1  2002/09/10 15:36:15  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.11  2002/08/31 13:40:06  roefer
 * MonteCarlo added
 *
 * Revision 1.10  2002/08/30 14:34:06  dueffert
 * removed unused includes
 *
 * Revision 1.9  2002/07/23 13:33:43  loetzsch
 * new streaming classes
 *
 * removed many #include statements
 *
 * Revision 1.8  2002/07/09 16:26:48  roefer
 * OUTPUT without braces fixed
 *
 * Revision 1.7  2002/06/28 10:26:19  roefer
 * OUTPUT is possible in constructors
 *
 * Revision 1.6  2002/06/20 00:40:20  Thomas Rfer
 * WLan crash removed
 *
 * Revision 1.5  2002/06/12 11:34:53  roefer
 * Improvements in LinesSelfLocator
 *
 * Revision 1.4  2002/06/11 10:22:21  kallnik
 * Pose2D neue Version (RotationVector) fertig
 * PLEASE DON'T  READ ROTATION DIRECTLY USE getAngle()
 *
 * Revision 1.3  2002/06/06 17:02:03  roefer
 * LinesSelfLocator, first approach
 *
 * Revision 1.2  2002/06/02 23:21:09  roefer
 * Single color table and progress in LinesSelfLocator
 *
 * Revision 1.1.1.1  2002/05/10 12:40:15  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.3  2002/04/03 17:27:45  roefer
 * Reduced memory usage
 *
 * Revision 1.2  2002/04/02 13:10:20  dueffert
 * big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
 *
 * Revision 1.1  2002/04/01 08:28:32  roefer
 * Dummy LinesSelfLocator added
 *
 *
 */
