/**
* @file SingleLandmarkSelfLocator.cpp
* 
* This file contains a class for self-localization based on measurements to a single landmark
* @author <a href=mailto:brunn@sim.tu-darmstadt.de>Ronnie Brunn</a>
* @author <a href=mailto:mkunz@sim.tu-darmstadt.de>Michael Kunz</a>
*/

#include "MRSingleLandmarkSelfLocator.h"
#include "Tools/FieldDimensions.h"
//#include "Tools/Debugging/Debugging.h"


/**
 * Constructor
 */
MRSingleLandmarkSelfLocator::MRSingleLandmarkSelfLocator(SelfLocatorInterfaces& interfaces)
: SelfLocator(interfaces)
{
  lastOdometry.translation.x=0; // let's set some starting values
  lastOdometry.translation.y=0;
  lastOdometry.rotation=0;
  lastPose.setValidity(0);
  lastPose.translation.x = -1500;
  lastPose.translation.y = 0;
  lastPositionValidity=0;
  lastRotationValidity=0;
}

const double PROGRESSIVENESS=0.10; /**< parameter used to control the relation between smooth velues and fast rrelocalization */

/**
 * Execute
 */ 
void MRSingleLandmarkSelfLocator::execute()
{
  robotPose.setFrameNumber(landmarksPercept.frameNumber);

  const double MINIMUM_FLAG_VALIDITY=0.39; /*< the minimum validity value for a landmark to be used. At 0.5 no landmarks that are far than 3000 according to Landmarksperceptor - to bad distances on real field. */
  const double MINIMUM_GOAL_VALIDITY=0.10; /*< same for goals */

  double odometryDiff = (odometryData - lastOdometry).translation.abs(); // estimated movement since last cycle
  
  lastPose += odometryData - lastOdometry; // odometry update
  lastOdometry = odometryData;

  double poseAgingFactor=0.99; // no aging at 1.0
  if (odometryDiff == 0) poseAgingFactor=0.999;  // more aging at more movement
	//  else if (odometryDiff <= 5) poseAgingFactor=0.999;

  lastPositionValidity*=poseAgingFactor;
  lastRotationValidity*=poseAgingFactor;

//  lastPose.setValidity(lastPose.getValidity()*poseAgingFactor);

  double bestValidity=MINIMUM_FLAG_VALIDITY;
  int bestObject=-1;
  int secondBestObject=-1;
  int i;

  for (i=0; i<landmarksPercept.numberOfFlags; i++) {  // let's find the two best seen landmarks
    if (landmarksPercept.flags[i].distanceValidity>=bestValidity)
    {
        secondBestObject=bestObject;
        bestValidity=landmarksPercept.flags[i].distanceValidity;
        bestObject=i;
    }
  }

  if (secondBestObject>=0) {
    lastPose=nextPose(landmarksPercept.flags[secondBestObject],lastPose); // pose update with the second best landmark
  }

  if (bestObject>=0) {
    lastPose=nextPose(landmarksPercept.flags[bestObject],lastPose); // pose update with the best landmark
  }

  bestObject=-1;
  bestValidity=MINIMUM_GOAL_VALIDITY; 
  
  for (i=0; i<landmarksPercept.numberOfGoals; i++) { // let's find the best seen goal
    if (landmarksPercept.goals[i].angleValidity>=bestValidity)
    {
        bestValidity=landmarksPercept.goals[i].angleValidity;
        bestObject=i;
    }
  }

  if (bestObject>=0) {
    lastPose=nextPose(landmarksPercept.goals[bestObject],lastPose); // pose update with the best goal
  }

  // simple orthogonal clipping
  if (lastPose.translation.x<xPosOwnGoal) lastPose.translation.x=xPosOwnGoal;
  if (lastPose.translation.x>xPosOpponentGoal) lastPose.translation.x=xPosOpponentGoal;
  if (lastPose.translation.y<yPosRightSideline) lastPose.translation.y=yPosRightSideline;
  if (lastPose.translation.y>yPosLeftSideline) lastPose.translation.y=yPosLeftSideline;

  lastPose.setValidity(lastPositionValidity*lastRotationValidity); // calculate new reliability value
  robotPose=lastPose; // save pose for next cycle

  selfLocatorSamples.link(NULL, 0,0);
  }

/**
 * Calculate a new pose based on a detected flag 
 */
RobotPose MRSingleLandmarkSelfLocator::nextPose(const Flag& flag, RobotPose oldPose)
{
  RobotPose result;
    
  double distanceCorrectionFactor=flag.distanceValidity/(flag.distanceValidity+lastPositionValidity); // calculation of the amount of position correction according to the validity values
  distanceCorrectionFactor*=PROGRESSIVENESS; // this is for smoothing control

  Vector2<double> oldV=oldPose.translation - flag.position ; // vector from flag to old position
  double oldDistance=oldV.abs(); // distance from old position to flag
  
  result.translation=flag.position+
    (
    (oldV*(oldDistance+((flag.distance-oldDistance)*distanceCorrectionFactor))) 
    / oldDistance); // corrected position according to validity of flag

  double angleCorrectionFactor=flag.angleValidity/(flag.angleValidity+lastRotationValidity); // calculation of the amount of orientation correction according to the validity values
  angleCorrectionFactor*=PROGRESSIVENESS; // this is for smoothing control

  double flagA=atan2(-oldV.y,-oldV.x); // angle between oldV and worlds X-Axis
  double angleDiff=normalize((flagA-flag.angle)-oldPose.getAngle()); // difference between current orientation and the orientation proposed by the landmark

  result.rotation=oldPose.rotation+angleDiff*angleCorrectionFactor;
  
  // calulation of new reliability values depending on the amount of necessary correction
  lastPositionValidity = (1-distanceCorrectionFactor)*lastPositionValidity+distanceCorrectionFactor*getValidityDistance(flag.distance-oldDistance);
  lastRotationValidity = (1-angleCorrectionFactor)*lastRotationValidity+angleCorrectionFactor*getValidityAngle(angleDiff);

// old validity calculation
//  result.setValidity(((1-distanceCorrectionFactor)*oldPose.getValidity()+distanceCorrectionFactor*flag.distanceValidity));

  return result;
}


/**
 * Calculate a new pose based on a detected goal
 */
RobotPose MRSingleLandmarkSelfLocator::nextPose(const Goal& goal, RobotPose oldPose)
{
  RobotPose result;

  Vector2<double> goalPosition = (goal.leftPost+goal.rightPost)/2.0; // goal position in the middle between the two posts

  Vector2<double> oldV=oldPose.getPose().translation - goalPosition; // vector from goal to old position
  double oldDistance=oldV.abs(); // distance from old position to goal

  if (goal.distanceValidity >= 0.80) // only use goals for position correction, if distance validity is high enough
  {
    double distanceCorrectionFactor=goal.distanceValidity/(goal.distanceValidity+lastPositionValidity); // calculation of the amount of position correction according to the validity values
    distanceCorrectionFactor*=PROGRESSIVENESS; // this is for smoothing control
  
    result.translation=goalPosition+
      (
      (oldV*(oldDistance+((goal.distance-oldDistance)*distanceCorrectionFactor))) 
      / oldDistance); // corrected position according to validity of flag
      
    lastPositionValidity = (1-distanceCorrectionFactor)*lastPositionValidity+distanceCorrectionFactor*getValidityDistance(goal.distance-oldDistance); // calulation of new reliability values depending on the amount of necessary correction
  }
  else
    result.translation=oldPose.getPose().translation; // no position update, if distance validity is too bad
  
  double angleCorrectionFactor=goal.angleValidity/(goal.angleValidity+lastRotationValidity); // calculation of the amount of orientation correction according to the validity values
  angleCorrectionFactor*=PROGRESSIVENESS; // this is for smoothing control

  double goalA=atan2(-oldV.y,-oldV.x); // angle between oldV and worlds X-Axis
  double angleDiff=normalize((goalA-goal.angle)-oldPose.getPose().getAngle()); // difference between current orientation and the orientation proposed by the landmark

  result.rotation=oldPose.getPose().rotation+angleDiff*angleCorrectionFactor;

 // calulation of new reliability values depending on the amount of necessary correction
 lastRotationValidity = (1-angleCorrectionFactor)*lastRotationValidity+angleCorrectionFactor*getValidityAngle(angleDiff);

// old validity calculation
//  result.setValidity(((1-angleCorrectionFactor)*oldPose.getValidity()+angleCorrectionFactor*goal.angleValidity));

  return result;
}

/**
 * part of the calculation of the new position reliability value according to the distance difference
 */
double MRSingleLandmarkSelfLocator::getValidityDistance(double difference) 
{
  difference = fabs(difference);
  if (difference <= 1000)
    return 1.0-(difference/1000.0);
  return 0.0;
}

/**
 * part of the calculation of the new orientation reliability value according to the angle difference
 */
double MRSingleLandmarkSelfLocator::getValidityAngle(double difference) 
{
  difference = fabs(difference);
  if (difference <= pi_2)
    return 1.0-(difference/pi_2);
  return 0.0;
}


/*
 * Change log :
 *
 */
