/**
* @file BB2004SensorDataProcessor.h
* 
* This file contains a class for Sensor Data Processing.
*/

#ifndef __BB2004SensorDataProcessor_h_
#define __BB2004SensorDataProcessor_h_

#include "DefaultSensorDataProcessor.h"
#include "Tools/Actorics/Kinematics.h"

const double bb2004SensorDataProcessorFalldownTiltAngleBack = fromDegrees(-40);

/**
* @class BB2004SensorDataProcessor
* 
* The Bremen Byters SensorDataProcessor.
*/
class BB2004SensorDataProcessor : public DefaultSensorDataProcessor
{
public:
  /** 
  * Constructor
  * @param interfaces The paramters of the HeadControl module.
  */
  BB2004SensorDataProcessor(const SensorDataProcessorInterfaces& interfaces)
    : DefaultSensorDataProcessor(interfaces) {}

protected:
  /** calculates the offset and the rotation of the camera matrix */
  virtual void buildCameraMatrix(const SensorData& sensorData, CameraMatrix& cameraMatrix)
  {
    Kinematics::calculateCameraMatrix(sensorData, headState, cameraMatrix);
  }

  /** detects whether the robot fell down. */
  virtual bool detectFallDown()
  {
    double bodyTiltAngle = atan2((double)gravity.x,(double)-gravity.z);
    double bodyRollAngle = atan2((double)-gravity.y,(double)-gravity.z);

    return
      (bodyTiltAngle >  defaultSensorDataProcessorFalldownTiltAngle  ||
      bodyTiltAngle < bb2004SensorDataProcessorFalldownTiltAngleBack  ||
      bodyRollAngle >  defaultSensorDataProcessorFalldownRollAngle  ||
      bodyRollAngle < -defaultSensorDataProcessorFalldownRollAngle);
  }
};

#endif// __BB2004SensorDataProcessor_h_

/*
 * Change log :
 * 
 * $Log: BB2004SensorDataProcessor.h,v $
 * Revision 1.4  2004/04/09 11:35:53  roefer
 * Bremen Byters German Open check-in
 *
 * Revision 1.3  2004/03/20 09:55:25  roefer
 * Preparation for improved odometry
 *
 * Revision 1.2  2004/03/08 02:11:51  roefer
 * Interfaces should be const
 *
 * Revision 1.1  2004/02/04 13:41:33  roefer
 * Some place holders for BB2004 modules added
 *
 */
