/**
* @file DefaultTeamBallLocator.cpp
* 
* Implementation of class TeamBallLocator
*
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
* @author Martin Ltzsch
*/

#include "DefaultTeamBallLocator.h"
#include "Platform/SystemCall.h"

DefaultTeamBallLocator::DefaultTeamBallLocator(const TeamBallLocatorInterfaces& interfaces)
: TeamBallLocator(interfaces)
{
}

void DefaultTeamBallLocator::execute()
{
  double validityMax = 0;
  int bestIndex = -1;

  for(int i = 0; i < teamMessageCollection.numberOfTeamMessages; ++i)
  {
    // get the time until the ball was seen consecutively in own time
    unsigned long timeUntilSeenConsecutively
      = teamMessageCollection[i]
      .getTimeInOwnTime(teamMessageCollection[i].seenBallPosition.timeUntilSeenConsecutively);

    if(SystemCall::getTimeSince(timeUntilSeenConsecutively) < 2000)
    {
      // The ball was seen consecutively by the other robot in the last 2 seconds

      double v = teamMessageCollection[i].robotPose.getValidity();

      if(v > validityMax)
      {
        // take the ball of the robot with the highest validity of the robot's pose
        validityMax = v;
        bestIndex = i;
      }
    }
  }

  if(bestIndex >= 0)
  {
    // copy the seen ball position of the best other robot to
    // the communicated ball position
    communicatedBallPosition.x = teamMessageCollection[bestIndex].seenBallPosition.x;
    communicatedBallPosition.y = teamMessageCollection[bestIndex].seenBallPosition.y;
    communicatedBallPosition.timeWhenLastObserved
      = teamMessageCollection[bestIndex].getTimeInOwnTime
      (teamMessageCollection[bestIndex].seenBallPosition.timeUntilSeenConsecutively);
  }
}


/*
* Change log :
* 
* $Log: DefaultTeamBallLocator.cpp,v $
* Revision 1.2  2004/03/08 02:11:53  roefer
* Interfaces should be const
*
* Revision 1.1  2003/10/06 14:10:14  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.7  2003/05/12 16:05:57  risler
* bugfix: wrong sign
*
* Revision 1.6  2003/05/12 15:26:08  risler
* bugfix: replaced timeWhenFirstSeenConsecutively with timeUntilSeenConsecutively
*
* Revision 1.5  2003/05/03 13:04:54  roefer
* Warning removed
*
* Revision 1.4  2003/05/02 12:57:13  loetzsch
* TeamMessage now contains a SeenBallPosition instead of a BallPercept
*
* Revision 1.3  2003/05/01 19:32:04  roefer
* Ball position assignment issues solved
*
* Revision 1.2  2003/05/01 18:10:03  loetzsch
* renamed CommunicatedBallPosition::timeWhenLastReceived to CommunicatedBallPosition::timeWhenLastObserved
*
* Revision 1.1  2003/05/01 17:09:08  loetzsch
* Redesign of ball modeling:
* - Modularized class BallPosition
* - splitted up module "BallLocator" into "BallLocator" for modeling of percepts
*   and "TeamBallLocator" for modelling communicated positions
* - Removed solution JumpingBallLocator
* - Splitted Solution DefaultBallLocator into DefaultBallLocator and DefaultTeamBallLocator
* - Renamed SensorFusionBallLocator to GaussBellTeamBallLocator
*/
