/**
* @file Platform/Aperios1.2.1/File.h
* 
* Declaration of class File for Aperios 1.2.1
*/

#ifndef __File_h__
#define __File_h__

#include <FileSystem/futil.h>

/**
* This class provides basic file input/output capabilies. 
*/
class File
{
private:
  bool mEof;
  FATFS::FILE* mFileHandle; /**< File handle. */
  char buf[65536]; /**< A buffer to reduce the number of operation system calls. */
  unsigned int bufSize, /**< The number of bytes in the buffer while reading. */
    index; /**< The next byte in the buffer to read/write. */
  bool isWrite; /**< Is this a write stream? */
public:
/**
* Constructor.
* @param name File name or path. If it is a relative path, it is assumed
*             to be relative to the path for configuration files. Otherwise,
*             the path is used directly.
* @param mode File open mode as used by fopen defined in stdio.h.
  */
  File(const char* name,const char* mode);
  
  /**
  * Destructor.
  */
  ~File();
  
  /**
  * The function read a number of bytes from the file to a certain
  * memory location.
  * @param p The start of the memory space the data is written to.
  * @param size The number of bytes read from the file.
  */
  void read(void* p,unsigned size);
  
  /**
  * The function writes a number of bytes from a certain memory 
  * location into the file.
  * @param p The start of the memory space the data is read from.
  * @param size The number of bytes written into the file.
  */
  void write(const void *p,unsigned size); 
  
  /**
  * The function implements printf for the stream represented by
  * instances of this class.
  * @param format Format string as used by printf defined in stdio.h.
  * @param ... See printf in stdio.h.
  */
  void printf(const char* format, ...);
  
  /**
  * The function returns whether the file represented by an
  * object of this class actually exists.
  * @return The existence of the file.
  */
  bool exists() const {return mFileHandle >= 0;}
  
  /**
  * The function returns whether the end of the file represented 
  * by an object of this class was reached.
  * @return End of file reached?
  */
  bool eof() const; 

  /**
  * The function returns the current GT directory,
  * e.g. /MS/OPENR/APP or <...>/GT2003 or /usr/local/GT2003
  * @return The current GTDir
  */
  static char* getGTDir(); 
};

#endif // __FILE_H__

/*
* Change log :
* 
* $Log$
*
*/
