/**
 * @file
 * This file contains a sender for motor commands.
 */
#ifndef __MOTORCOMMANDSSENDER_H__
#define __MOTORCOMMANDSSENDER_H__

#ifndef __ProcessFramework_h__
#error Never include this file directly. Include ProcessFramework.h instead.
#endif

#include "Platform/ProcessFramework.h"
#include "Representations/Motion/MotorCommands.h"
#include <OPENR/OPENR.h>

/**
 * This class implements a sender for motor commands.
 */
class MotorCommandsSender : public SenderBase<MotorCommands>
{
private:
  OPrimitiveID jointId[JointData::numOfJoint]; /**< The Open-R IDs of all joints. */
  bool jointGainsSet; /**< A flag that states whether the joint gains were initialized. */
  MemoryRegionID memID; /**< The memory region of the only package instance. */
  OCommandVectorData* cmdVec; /**< A pointer to the data in the package. */
  //RCRegion* package; /**< The only instance of a package. */
  PIDData lastPidData; /**< The last joint gains sent. */
  int numOfLED; /**< The number of LEDs of the robot. */

  /**
    * Returns OPENR Primitive Joint Name
    */
  static const char* getPrimitiveJointName(int i) 
  {
    // SystemCall::getRobotDesign() == RobotDesign::ERS110
      switch (i) 
      {
        case 0: return "PRM:/r1/c1-Joint2:j1";            // head tilt
        case 1: return "PRM:/r1/c1/c2-Joint2:j2";         // head pan
        case 2: return "PRM:/r1/c1/c2/c3-Joint2:j3";      // head roll
        case 3: return "PRM:/r1/c1/c2/c3/c4-Joint2:j4";   // mouth
        case 4: return "PRM:/r1/c1/c2/c3/e1-Joint3:j5";   // ear L
        case 5: return "PRM:/r1/c1/c2/c3/e2-Joint3:j6";   // ear R
        case 6: return "PRM:/r4/c1-Joint2:j1";            // leg FR joint
        case 7: return "PRM:/r4/c1/c2-Joint2:j2";         // leg FR shoulder
        case 8: return "PRM:/r4/c1/c2/c3-Joint2:j3";      // leg FR knee
        case 9: return "PRM:/r2/c1-Joint2:j1";            // leg FL joint
        case 10: return "PRM:/r2/c1/c2-Joint2:j2";        // leg FL shoulder
        case 11: return "PRM:/r2/c1/c2/c3-Joint2:j3";     // leg FL knee
        case 12: return "PRM:/r5/c1-Joint2:j1";           // leg RR joint
        case 13: return "PRM:/r5/c1/c2-Joint2:j2";        // leg RR shoulder
        case 14: return "PRM:/r5/c1/c2/c3-Joint2:j3";     // leg RR knee
        case 15: return "PRM:/r3/c1-Joint2:j1";           // leg RL joint
        case 16: return "PRM:/r3/c1/c2-Joint2:j2";        // leg RL shoulder
        case 17: return "PRM:/r3/c1/c2/c3-Joint2:j3";     // leg RL knee
        case 18: return "PRM:/r6/c1-Joint2:j1";           // tail pan
        case 19: return "PRM:/r6/c2-Joint2:j2";           // tail tilt
        default: return ""; // not defined
      }
  }

  /**
    * Return OPENR Primitive LED Name
    */
  static const char* getPrimitiveLEDName(int i) 
  {
    // SystemCall::getRobotDesign() == RobotDesign::ERS110
      switch (i) 
      {
        case 0: return "PRM:/r1/c1/c2/c3/l1-LED2:l1";
        case 1: return "PRM:/r1/c1/c2/c3/l2-LED2:l2";
        case 2: return "PRM:/r1/c1/c2/c3/l3-LED2:l3";
        case 3: return "PRM:/r1/c1/c2/c3/l4-LED2:l4";
        case 4: return "PRM:/r1/c1/c2/c3/l5-LED2:l5";
        case 5: return "PRM:/r1/c1/c2/c3/l6-LED2:l6";
        case 6: return "PRM:/r1/c1/c2/c3/l7-LED2:l7";
        case 7: return "RPM:/r6/l1-LED2:l1"; //Blue LED
        case 8: return "RPM:/r6/l2-LED2:l2"; //Orange LED
        default: return ""; //not defined
      }
  }

protected:
  /**
    * The function prepares a package.
    */
  virtual void preparePackage();

  /**
    * The functions sets a package for a receiver.
    * @param receiver The receive the package will be sent to.
    */
  virtual void setPackage(const ObserverID& receiver);

  /**
    * The function frees the package.
    */
  virtual void freePackage();

public:
  /**
    * The constructor.
    * @param process The process this sender is associated with.
    * @param blocking Decides whether this sender blocks the execution of the next frame
    *                 until all connected receivers have requested a new package.
    */
  MotorCommandsSender(PlatformProcess* process,bool blocking);

  /**
    * Destructor.
    */
  ~MotorCommandsSender();
};

/**
 * The macro declares a sender for motor commands.
 * It must be used inside a declaration of a process, after the macro DEBUGGING.
 */
#define SENDER_MOTORCOMMANDS \
  MotorCommandsSender theMotorCommandsSender

/**
 * The macro instantiates a sender for MotorCommands.
 * @param blocking Decides whether this sender blocks the execution of the next frame
 *                 until all connected receivers have requested a new package.
 */
#define INIT_SENDER_MOTORCOMMANDS(blocking) \
  theMotorCommandsSender(this,blocking)

#endif

/*
 * Change log :
 * 
 * $Log$
 *
*/
