/** 
* @file  Platform/Aperios1.3.2/IPEndpoint.cpp
*
* Implementation for IP Communication Classes
* @attention this is the Aperios implementation
*
* @author <A href=mailto:robocup@m-wachter.de>Michael Wachter</A>
* 
*/

#include "IPEndpoint.h"
#include <stdio.h>
#include <iostream.h>

/*
 * Constructor for an enpoint - creates shared memory buffers
 * and gets pointers for entry-points
 */

IPEndpoint::IPEndpoint(int sbs, int rbs) :
    sharedSendBufferSize(sbs),
    sharedReceiveBufferSize(rbs)
{
     ipStackRef = antStackRef("IPStack");

     // Get the OID of the process because we need it to send the 
     // messages
     myOID_ = ProcessBase::theInstance->getOID();

     // Get selectors of the entrypoints to send messages
     // (passed as pointers because they change if the endpoint
     // is created in the constructor of the process)
     ProcessBase::theInstance->
           getAntInformation(listenContSelector, 
                 sendContSelector, receiveContSelector,
                 closeContSelector, connectContSelector);
         
     // Allocate send buffer
     antEnvCreateSharedBufferMsg sendBufferMsg(sharedSendBufferSize);
     sendBufferMsg.Call(ipStackRef, sizeof(sendBufferMsg));

     if (sendBufferMsg.error != ANT_SUCCESS) 
     {
       cout << "IPEndpoint: ANT Error creating send buffer \n";
     }

     sendBuffer = sendBufferMsg.buffer;
     sendBuffer.Map();
     sharedSendBuffer = static_cast<byte*>(sendBuffer.GetAddress());


     // Allocate receive buffer
     antEnvCreateSharedBufferMsg receiveBufferMsg(sharedReceiveBufferSize);
     receiveBufferMsg.Call(ipStackRef, sizeof(receiveBufferMsg));

     if (receiveBufferMsg.error != ANT_SUCCESS)
     {
       cout << "IPEndpoint: ANT Error creating receive buffer \n";
     }

     receiveBuffer = receiveBufferMsg.buffer;
     receiveBuffer.Map();
     sharedReceiveBuffer = static_cast<byte*>(receiveBuffer.GetAddress());

     cout << "IPEndpoint(): created sendBuffer with " << sharedSendBufferSize 
       << " and receiveBuffer with " << sharedReceiveBufferSize << " bytes \n";
}

IPEndpoint::~IPEndpoint()
{
    sendBuffer.UnMap();
    receiveBuffer.UnMap();

    // From InternetProtocolVersion4.pdf
    // antError UnMap();
    // Description
    // Removes a shared memory buffer from an objects address space.
    // This operation is required before the shared buffer can be destroyed.
}

Out& operator<<(Out& stream, const IPAddress& ipAddress)
{
    unsigned int address = ipAddress.Address();

    stream << (address >> 24) << "." 
           << (address >> 16 & 255) << "." 
           << (address >> 8 & 255) << "."
           << (address & 255);

    return(stream);
}

/*
 * Change log :
 * 
 * $Log: IPEndpoint.cpp,v $
 * Revision 1.8  2004/01/26 13:44:07  wachter
 * shared-memory-buffers now have variable sizes
 *
 * Revision 1.7  2004/01/16 16:23:58  wachter
 * Bugfixes
 *
 * Revision 1.6  2004/01/09 13:57:23  wachter
 * Addes streaming-operator for Open-R IPAddress
 *
 * Revision 1.5  2004/01/03 16:18:24  wachter
 * debug-communication mostly working now
 *
 * Revision 1.4  2003/12/11 15:02:37  wachter
 * Low-level TCP and UDP functions for aperios now working.
 *
 * Revision 1.3  2003/12/03 14:21:52  wachter
 * Splitted IPEndpoint.*
 *
 * Revision 1.2  2003/12/02 20:52:41  wachter
 * Added IP-networking to aperios process-framework.
 *
 * Revision 1.1  2003/11/29 23:01:06  wachter
 * idea of an ip-implementation in the process-framework
 *
 * 
 */

