/**
 * This file contains classes related to senders.
 * @author Thomas Rfer
 */
#ifndef __SENDER_H__
#define __SENDER_H__

#ifndef __ProcessFramework_h__
#error Never include this file directly. Include ProcessFramework.h instead.
#endif

#include "Tools/Streams/OutStreams.h"

const int RECEIVERS_MAX = 20; /**< The maximum number of receivers connected to a single sender */

/**
 * The class is the base class for senders.
 * A sender is an object that sends packages to an Aperios queue.
 * The class manages a global list of all senders in an Aperios process.
 */
class SenderList : public OSubject
{
  private:
    PlatformProcess* process;   /**< The process this sender is associated with. */
    SenderList* next;           /**< The successor of the current sender. */
    char name[NAME_LENGTH_MAX]; /**< The name of a sender without the module's name. */
  
  protected:
    int eventId;                /**< The id of the current sender in the range [0..30]. */
    bool blocking;              /**< Stores whether this is a blocking sender. */

    /**
     * The function sends a package to all receivers that requested it.
     */
    virtual void sendPackage() = 0;

  public:
    /**
     * The constructor.
     * @param process The process this sender is associated with.
     * @param senderName The Aperios connection name of the sender without the process name.
     * @param blocking Decides whether this sender blocks the execution of the next frame
     *                 until all connected receivers have requested a new package.
     */
    SenderList(PlatformProcess* process,const char* senderName,bool blocking);

    /**
     * Returns the begin of the list of all senders.
     * @return The first sender in the list, or 0 if the list ist empty.
     */
    SenderList*& getFirst();

    /**
     * Returns the next sender in the list.
     * @return The next sender in the list, or 0 if this sender is the last one.
     */
    SenderList* getNext() const {return next;}

    /**
     * Returns the Aperios connection name of the sender.
     * @return The Aperios connection name without the process name ("Sender.type.O")
     */
    const char* getName() const {return name;}

    /**
     * The function must be called to finish the current frame.
     */
    void finishFrame();

    /**
     * The functions sets or resets a bit in the blocking mask.
     * After a bit is set in the blocking mask for a certain 
     * sender or receiver, a new frame will not be started before
     * this sender or receiver received an event.
     * @param id The id of the sender or receiver.
     * @param block Should it block or not?
     */
    static void setBlockingId(int id,bool block = true);

    /**
     * The function is called when an event was received. 
     * If this was the last event the process was waiting for, the next
     * frame is started, i.e. NextFrame() is called.
     * @param id The id of the sender or receiver that received an event.
     */
    static void setEventId(int id);

    /**
     * The function is called when a receiver announces that it is ready to receive a package.
     * @param msg An Open-R message that contains the object id of the receiver.
     */    
    virtual void handleMessage(OReadyMessage& msg) = 0;
};

/**
 * The class implements a sender.
 * A sender is an object that sends packages to an Aperios queue.
 */
template<class T> class Sender : public SenderList, public T
{
  private:
    ObserverID receiver[RECEIVERS_MAX],        /**< A list of all receivers that are ready to receive a package. */
               alreadyReceived[RECEIVERS_MAX]; /**< A list of all receivers that have already received the current package. */
    int numOfReceivers,                        /**< The number of entries in the receiver list. */
        numOfAlreadyReceived;                  /**< The number of entries in the alreadyReceived list. */

    /**
     * The function is called when a receiver announces that it is ready to receive a package.
     * @param msg An Open-R message that contains the object id of the receiver.
     */    
    void handleMessage(OReadyMessage& msg)
    {
      ReadyHandler(msg);
      ASSERT(numOfReceivers < RECEIVERS_MAX);
      receiver[numOfReceivers++] = msg.observerID;
      if(numOfReceivers == NumberOfObservers())
        setEventId(eventId);
    }

  protected:
    /**
     * The function sends a package to all receivers that requested it.
     */
    virtual void sendPackage()
    {
      if(numOfAlreadyReceived != -1)
      { // send() has been called at least once
        int i;
        for(i = 0; i < numOfReceivers; ++i)
        {
          int j;
          for(j = 0; j < numOfAlreadyReceived; ++j)
            if(receiver[i] == alreadyReceived[j])
              break;
          if(j == numOfAlreadyReceived)
            break; // receiver[i] has not received its requested package yet
        }
        if(i < numOfReceivers)
        { // at least one receiver has not received its requested package, so prepare one
          const T& data = *static_cast<const T*>(this);
          OutBinarySize size;
          size << data;
          RCRegion* r = new RCRegion(size.getSize());
          ASSERT(r);
          OutBinaryMemory memory(r->Base());
          memory << data;

          // Send this package to all the receivers that requested it and that have not received it yet
          i = 0;
          while(i < numOfReceivers)
          {
            int j;
            for(j = 0; j < numOfAlreadyReceived; ++j)
              if(receiver[i] == alreadyReceived[j])
                break;
            if(j == numOfAlreadyReceived)
            { // receiver[i] has not received its requested package yet
              VERIFY(SetData(receiver[i],r) == oSUCCESS);
              VERIFY(NotifyObserver(receiver[i]) == oSUCCESS);
              // note that receiver[i] has received the current package
              ASSERT(numOfAlreadyReceived < RECEIVERS_MAX);
              alreadyReceived[numOfAlreadyReceived++] = receiver[i];
              // remove receiver[i] from the list 
              receiver[i] = receiver[--numOfReceivers];
            }
            else 
              ++i;
          }
          // The sender does not need the package anymore
          r->RemoveReference();
        }
      }
    }
    
  public:
    /**
     * The constructor.
     * @param process The process this sender is associated with.
     * @param senderName The Aperios connection name of the sender without the process name.
     * @param blocking Decides whether this sender blocks the execution of the next frame
     *                 until all connected receivers have requested a new package.
     */
    Sender(PlatformProcess* process,const char* senderName,bool blocking)
    : SenderList(process,senderName,blocking)
    {
      numOfReceivers = 0;
      numOfAlreadyReceived = -1;
    }

    /**
     * Returns whether a new package was requested from the sender.
     * This is always true if this is a blocking sender.
     * @return Has a new package been requested?
     */
    bool requestedNew() const {return numOfReceivers > 0;}

    /**
     * Marks the package for sending and transmits it to all receivers that already requested for it.
     * All other receiver may get it later if they request for it before the package is changed.
     */
    void send()
    {
      setBlockingId(eventId,blocking);
      numOfAlreadyReceived = 0;
      sendPackage();
    }
};

#endif

/*
 * Change log :
 * 
 * $Log: Sender.h,v $
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.2  2003/04/04 21:28:35  roefer
 * Communication protocol changed
 *
 * Revision 1.1  2002/09/10 15:40:04  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.3  2002/08/14 17:11:23  dueffert
 * adapted by Thomas Roefer to OPENR_SDK-1.1.3-r1 and OPENR_SYS-007
 *
 * Revision 1.2  2002/07/23 16:40:11  roefer
 * Router and SimGT2002 adapted to new message queue and streams
 *
 * Revision 1.1.1.1  2002/05/10 12:40:18  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.1  2002/05/02 17:04:09  roefer
 * New router
 *
 * Revision 1.9  2002/04/21 16:09:02  roefer
 * Anchor removed
 *
 * Revision 1.8  2001/12/15 20:32:08  roefer
 * Senders and receivers are now part of the processes
 *
 * Revision 1.7  2001/12/12 13:25:02  roefer
 * Blocking sender fixed
 *
 * Revision 1.6  2001/12/10 17:47:08  risler
 * change log added
 *
 */
