/** 
* @file Platform/Win32/ForRobotControl/OVirtualRobotComm.h
*
* Declaration of OVirtualRobotComm.
*
* @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
*/

#ifndef __OVirtualRobotComm_H__
#define __OVirtualRobotComm_H__

#include "RobotControl/SimRobot/Controller.h"
#include "SimRob95/SimRobot/sim3DMoveable.h"

#include "Tools/Process.h"
#include "Tools/Math/Common.h"
#include "Platform/Sensors.h"
#include "Representations/Motion/OdometryData.h"
#include "Representations/Cognition/RobotPose.h"
#include "Representations/Cognition/BallModel.h"
#include "Representations/Cognition/PlayerPoseCollection.h"
#include "Tools/RingBuffer.h"

class RoboCupCtrl;

/**
* @class OVirtualRobotComm
*
* A process that simulates the Aperios OVirtualRobotComm.
*/
class OVirtualRobotComm : public Process
{
  
public:
  DEBUGGING;
  DECLARE_SYNC; /**< Make this object synchronizable. */
private:
  RECEIVER(MotorCommands);
  RECEIVER(OdometryData);
  
  SENDER(SensorDataBuffer);
  SENDER(Image);
  
  
  unsigned frameNumber, /**< The frame counter. */
    lastFrameNumber; /**< The last frame already processed. */
  MotorCommands motorCommands; /**< A buffer for the motor commands. */
  RoboCupCtrl* ctrl; /** A pointer to the controller object. */
  SIM3DOBJECT objRobot; /**< A pointer to the associated SimRobot robot. */
  SENSORPORT spCamera; /**< The sensor port of the robot's camera. */
  PSHORTREAL imageBuffer; /**< A buffer for the camera image. */
  SENSORPORT spPsd; /**< The sensor port of the robot's psd sender. */
  SIM3DMoveable* objJoint[JointData::numOfJoint]; /**< Pointers to all joints. */
  SIM3DElement* objLed[LEDValue::numOfLED_ERS210]; /**< Pointers to all leds. */
  /** @todo double?*/
  double target[JointData::numOfJoint]; /**< Buffers for joint angles. */
  long overruleSensorData[SensorData::numOfSensor_ERS210]; /**< Allows to set sensor values from the user interface. */
  OdometryData odometryData, /**< A buffer for the odometry data. */
               lastOdometry; /**< The last odometry pose. */
  double lastHeight, /**< The previous height of the body of the robot. */
          lastSpeed;  /**< The previous vertical speed of the body of the robot. */
  static const char* jointName[JointData::numOfJoint]; /**< A table of the SimRobot names of the joints. */
  static const char* ledName[LEDValue::numOfLED_ERS210]; /**< A table of the SimRobot names of the leds. */
  
  /** a RobotPose that was calculated by the oracle */
  RobotPose oracledRobotPose;
  
  /** a BallModel that was calculated by the oracle */
  BallModel oracledBallPosition;
  
  /** a PlayerPoseCollection that was calculated by the oracle */
  PlayerPoseCollection oracledPlayerPoseCollection;
  
  /**
  * The function returns the angle of a joint.
  * @param j The index of the joint.
  * @return The angle in radians.
  */
  double getJointAngle(int j) const
  {ASSERT(objJoint[j]); return fromDegrees(objJoint[j]->ActValue);}
  
  /**
  * The function sets the angle of a joint.
  * @param j The index of the joint.
  * @param angle The angle in radians.
  */
  void setJointAngle(int j,double angle);
  
  /**
  * The function sets the on/off state of an led.
  * @param l The index of the led.
  * @param state The new state of the led.
  */
  void setLedState(int l,bool state) 
  {ASSERT(objLed[l]); objLed[l]->DetailLevel = state ? 2 : 10;}

  /**
  * Buffer for PSD values to simulate PSD sensor delay
  */
  RingBuffer<SHORTREAL,10> psdBuffer;
  
public:
/**
* Constructor.
  */
  OVirtualRobotComm();
  
  /**
  * Destructor.
  */
  ~OVirtualRobotComm() {delete [] imageBuffer;}
  
  /**
  * The function is called from the framework once in every frame 
  */
  virtual int main();
  
  /**
  * The function must be called to exchange data with SimRobot.
  * It sends the motor commands to SimRobot and acquires new sensor data.
  */
  void update();
  
  /**
  * the number of the robot [0..7]
  */
  int robotNumber;
  
  /**
  * Activates or diactivates a certain switch on the robot.
  * @param sensor The switch. Allowed values are: headBack, headFront,
  *               mouth, chin, and back.
  * @param activate Decides whether to activate or deactivate the switch.
  */
  void activateSwitch(SensorData::sensors sensor, bool activate = true);
  
  /**
  * The function checks whether a certain switch is activated.
  * @param sensor The switch to check.
  * @return Activated or not?
  */
  bool isSwitchActivated(SensorData::sensors sensor) const
  {return overruleSensorData[sensor] != jointDataInvalidValue;}
  
  /** 
  * The function is called for every incoming debug message.
  */
  virtual bool handleMessage(InMessage& message);
};

#endif // __OVirtualRobotComm_H__


/*
* Change log :
*
* $Log: OVirtualRobotComm.h,v $
* Revision 1.5  2004/02/03 13:19:49  spranger
* renamed all references to  class BallPosition to BallModel
*
* Revision 1.4  2003/12/31 20:16:13  roefer
* SensorData for ERS-7
*
* Revision 1.3  2003/12/31 14:29:19  roefer
* Joints and LEDs for ERS-7
*
* Revision 1.2  2003/12/06 21:03:16  roefer
* Improved simulation of z-acceleration
*
* Revision 1.1  2003/10/07 10:06:59  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 15:30:28  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1  2003/09/26 11:34:07  juengel
* no message
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.11  2003/06/26 17:07:22  risler
* correct psd sensor delay in simulator
*
* Revision 1.10  2003/05/02 18:26:18  risler
* SensorDataBuffer added
* replaced SensorData with SensorDataBuffer
* full SensorData resolution now accessible
*
* Revision 1.9  2003/04/16 14:20:19  loetzsch
* removed Xabsl 1 from GT2003
*
* Revision 1.8  2003/03/22 22:37:51  loetzsch
* finished GUI and message routing for 8 simulated robots
* almost all functionallity for simulated robots now is encapsulated in class
*   CRobotControlSimulatedRobots
*
* Revision 1.7  2003/03/05 17:09:27  loetzsch
* redesign of the queues and debug key tables in RobotControl
*
* Revision 1.6  2003/03/05 11:09:49  loetzsch
* added class GlobalGameControlData
* the Game Toolbar now sents GlobalGameControlData
*
* Revision 1.5  2002/12/18 16:22:55  dueffert
* doxygen docu corrected
*
* Revision 1.4  2002/09/22 18:40:55  risler
* added new math functions, removed GTMath library
*
* Revision 1.3  2002/09/18 19:52:36  loetzsch
* the head state is now sent from Motion to Cognition using the package.
*
* Revision 1.2  2002/09/17 23:55:22  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.1  2002/09/10 15:40:05  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.7  2002/08/29 14:02:08  dueffert
* includes in correct case, system includes in <>
*
* Revision 1.6  2002/08/22 15:20:10  loetzsch
* debug messages to the local processes are now sent in a better synchronized way.
*
* Revision 1.5  2002/07/23 13:39:39  loetzsch
* - new streaming classes
* - removed many #include statements
* - new design of debugging architecture
* - exchanged StaticQueue with MessageQueue
*
* Revision 1.4  2002/07/13 11:54:46  roefer
* New command and sound sender
*
* Revision 1.3  2002/06/04 23:27:02  loetzsch
* 4-robots-in-RobotControl related bug fixes and improvements
*
* Revision 1.2  2002/06/04 00:15:36  loetzsch
* RobotControl now can simulate four robots.
*
* Revision 1.1.1.1  2002/05/10 12:40:18  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.12  2002/04/25 14:50:36  kallnik
* changed double/float to double
* added several #include GTMath
*
* PLEASE use double
*
* Revision 1.11  2002/04/20 15:52:20  roefer
* Project simpified, WATCH and WATCH_PART added
*
* Revision 1.10  2002/04/06 09:55:53  roefer
* Image and SensorData path through DebugQueues changed
*
* Revision 1.9  2002/03/24 18:15:00  loetzsch
* continued change to blocking sensor data receivers
*
* Revision 1.8  2002/03/11 12:51:00  loetzsch
* BerlinGermanOpenBehaviorControl added
*
*/
