/**
* @file SensorView.h
*
* Declaration of class SensorView
*
* @author <a href="mailto:roefer.tzi.de">Thomas Rfer</a>
*/

#ifndef __SensorView_h_
#define __SensorView_h_

#include "RobotConsole.h"

/**
* @class SensorView 
* 
* A class to represent a view with information about the timing of modules.
*
* @author <a href="mailto:roefer.tzi.de">Thomas Rfer</a>
*/
class SensorView : public DirectView
{
  private:
    enum Type
    {
      angle,
      distance,
      pressure,
      button,
      temperature,
      acceleration
    };

    struct ViewInfo
    {
      SensorData::sensors sensor; /**< The id of the sensor. */
      Type type;  /**< The type of measurements the sensor takes. */
      bool endOfSection; /**< Is this entry the last in a group of sensors? */
    };

    RobotConsole& console; /**< A reference to the console object. */
    const SensorData& sensorData; /**< The most current sensor data. */
    static double leftOfColumnPercent[3]; /**< Ratios for the beginning of the columns. */
    CFont normalFont, /**< Normal font. */
          boldFont; /**< Bold font. */
    CDC* dc; /**< The current device context. */
    int line, /**< A line counter in pixels. */
        width, /**< The width of the view in pixels. */
        lineHeight; /**< The height of a line in pixels. */
    static const ViewInfo viewInfoERS210[], /**< Information about the sensors of the ERS-210. */
                          viewInfoERS7[]; /**< Information about the sensors of the ERS-7. */
  
    /**
    * The function prints the measurement and the unit of a sensor.
    * @param vi Information about the sensor to output.
    */
    void printSensorData(const ViewInfo& vi);

    /**
    * The function prints a string to a certain column in the view.
    * @param s The string.
    * @param column A column index.
    * @param bold Should the string be printed in boldface?
    */    
    void print(const std::string& s, int column, bool bold = false);

    /**
    * The function sets the print position to the next line.
    * @param drawLine Should a line be drawn between the lines?
    */
    void newLine(bool drawLine = false);

  public:
    /**
    * Constructor.
    * @param c The console object.
    * @param s The most current sensor data to be visualized.
    */
    SensorView(RobotConsole& c, const SensorData& s);

    /**
    * Destructor.
    */
    ~SensorView();

    /**
    * The function is called to draw the view.
    * @param dc The device context to draw to.
    */
    void draw(CDC& dc);
};

#endif //__SensorView_h_

/*
 * Change log :
 * 
 * $Log: SensorView.h,v $
 * Revision 1.1  2004/04/12 18:00:45  roefer
 * SensorView added
 *
 * Revision 1.1  2004/04/09 18:55:16  roefer
 */
