/**
 * @file Platform/Win32/ProcessFramework.h
 *
 * This file declares classes corresponding to the process framework.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */
#ifndef __ProcessFramework_h__
#define __ProcessFramework_h__

#include "Thread.h"
#include "Tools/List.h"
#include "Representations/Motion/MotorCommands.h"
#include "Representations/Sound/SoundData.h"
#include "SystemCall.h"

#pragma warning(disable:4355) // VC++: "this" in initializer list is ok

const int NAME_LENGTH_MAX = 100, /**< The maximum length of Aperios connection names. */
          PROCESS_MAX = 100,     /**< The maximum number of processes in this address context. */
          ROBOT_MAX = 8;         /**< The maximum number of robots in this address context. */

class PlatformProcess;

#include "Receiver.h"
#include "Sender.h"

class Robot;

/**
 * This class is the platform dependent base class for all processes.
 */
class PlatformProcess
{
  private:
    SenderList* firstSender; /**< The begin of the list of all senders of this process. */
    ReceiverList* firstReceiver; /**< The begin of the list of all receivers of this process. */
    int blockMask, /**< A mask with bits set for all blocking senders and receivers. */
        eventMask; /**< A mask with bits set for all senders and receivers that received an event */
    int lastTime; /**< The last time when Process::Main() was finished. */
    int id; /**< A counter that is used to provide different ids to senders and receivers. */
    unsigned sleepUntil; /**< The process should sleep until this point in time is reached. */

  protected:
    /**
     * The function will be call from the process framework in each frame.
     * It shall provide the main functionality of the process.
     * @return If 0 is returned, the function main will not be called on a timed basis
     *         again. A positive number specifies the minimum number of microseconds to
     *         wait before the next frame starts. A negative number specifies a cycle 
     *         time, i.e. the time between two activations of the processMain() function.
     */
    virtual int processMain() = 0;

  public:
    /**
     * Constructor.
     */
    PlatformProcess()
    {
      firstSender = 0;
      firstReceiver = 0;
      blockMask = 0;
      eventMask = 0;
      lastTime = 0;
      id = 0;
      sleepUntil = 0;
    }

    /**
     * The function returns a new id each time it is called.
     * @return The next unused id.
     */
    int getNextId() {return id++;}

    /**
     * The function returns the begin of list of all senders.
     * Note that the function returns a reference that can be changed.
     * @return A reference to the address of the first element. 
     */
    SenderList*& getFirstSender() {return firstSender;}

    /**
     * The function returns the begin of list of all receivers.
     * Note that the function returns a reference that can be changed.
     * @return A reference to the address of the first element. 
     */
    ReceiverList*& getFirstReceiver() {return firstReceiver;}

    /**
     * The functions sets or resets a bit in the blocking mask.
     * After a bit is set in the blocking mask for a certain 
     * sender or receiver, a new frame will not be started before
     * this sender or receiver received an event.
     * @param id The id of the sender or receiver.
     * @param block Should it block or not?
     */
    void setBlockingId(int id,bool block = true);

    /**
     * The function is called when an event was received. 
     * If this was the last event the process was waiting for, the next
     * frame is started, i.e. NextFrame() is called.
     * @param id The id of the sender or receiver that received an event.
     */
    void setEventId(int id);

    /**
     * Returns the event mask.
     * @return The event mask.
     */
    int getEventMask() {return eventMask;}

    /**
     * Resets the event mask.
     */
    void resetEventMask() {eventMask = 0;}

    /**
     * The function is called once for each frame.
     */
    void nextFrame()
    {
      int frameTime = processMain();
      if(getFirstSender())
        getFirstSender()->finishFrame();
      if(getFirstReceiver())
        getFirstReceiver()->finishFrame();
      resetEventMask();
      int currentTime = SystemCall::getCurrentSystemTime();
      if(frameTime < 0)
        frameTime = lastTime - frameTime - currentTime;
      if(frameTime)
        sleepUntil = SystemCall::getCurrentSystemTime() + frameTime;
      else
        sleepUntil = 0;
      setBlockingId(31,frameTime != 0);
      lastTime = currentTime + frameTime;
    }

    /**
     * The function checks whether a timeout condition is satisfied if one was defined.
     * If this is the case, the corresponding eventId is set.
     */
    void checkTime()
    {
      if(sleepUntil && sleepUntil <= SystemCall::getCurrentSystemTime())
      {
        sleepUntil = 0;
        setEventId(31);
      }
    }

    /**
     * The function returns the index of the current process.
     * @return A number between 0 and PROCESS_MAX-1.
     */    
    static int getIndex();

    /**
     * The function returns the index of the robot associated to the current process.
     * @return A number between 0 and ROBOT_MAX-1.
     */    
    static int getRobotIndex();
};

/**
 * The class is a helper that allows to instantiate a class as an Win32 process.
 * ProcessBase contains the parts that need not to be implemented as a template.
 * It will only be used by the macro MAKE_PROCESS and should never be used directly.
 */
class ProcessBase : public Thread<ProcessBase>
{
  private:
    Robot& robot; /**< A reference to the robot this process corresponds to. */

  protected:
    /**
     * The main function of this Windows thread.
     */
    virtual void main() = 0;

  public:
    /**
     * Constructor.
     * @param r The robot this process corresponds to.
     */
    ProcessBase(Robot* r) : robot(*r) {}
    
    /**
     * Virtual destructor.
     */
    virtual ~ProcessBase() {}

    /**
     * The function starts the process by starting the Windows thread.
     */
    void start() {Thread<ProcessBase>::start(this,main);}

    /**
     * The functions searches for a sender with a given name.
     * @param name The name of the sender.
     * @return If the sender was found, a pointer to it is returned.
     *         Otherwise, the function returns 0.
     */ 
    virtual SenderList* lookupSender(const char* name) = 0;

    /**
     * The functions searches for a receiver with a given name.
     * @param name The name of the receiver.
     * @return If the receiver was found, a pointer to it is returned.
     *         Otherwise, the function returns 0.
     */ 
    virtual ReceiverList* lookupReceiver(const char* name) = 0;

    /**
     * The function returns the name of the process.
     * @return The name of the process that normally is its class name.
     */
    virtual const char* getName() const = 0;
    
    /**
     * The function returns a pointer to the process if it has the given name.
     * @param processName The name of the process that is searched for.
     * @return If the process has the required name, a pointer to it is
     *         returned. Otherwise, the function returns 0.
     */
    virtual PlatformProcess* getProcess(const char* processName) = 0;
};    

/**
 * The class is a helper that allows to instantiate a class as an Win32 process.
 * ProcessCreator contains the parts that need to be implemented as a template.
 * It will only be used by the macro MAKE_PROCESS and should never be used directly.
 */
template<class T> class ProcessFrame : public ProcessBase
{
  private:
    T process; /**< The process. */
    const char* name; /**< The name of the process. */

  protected:
    /**
     * The main function of this Windows thread.
     */
    virtual void main()
    {
      // Call process.nextFrame if no blocking receivers are waiting
      process.setBlockingId(31);
      process.setEventId(31);
      Sleep(4); // always leave processing time to other threads
      while(isRunning())
      {
        if(process.getFirstSender())
          process.getFirstSender()->checkAllForRequests();
        if(process.getFirstReceiver())
          process.getFirstReceiver()->checkAllForPackages();
        process.checkTime();
        Sleep(4); // always leave processing time to other threads
      }
    }

  public:
    /**
     * Constructor.
     * @param r The robot this process corresponds to.
     * @param name The name of the process.
     */
    ProcessFrame(Robot* r,const char* name)
    : ProcessBase(r)
    {
      this->name = name;
    }

    /**
     * The functions searches for a sender with a given name.
     * @param senderName The name of the sender.
     * @return If the sender was found, a pointer to it is returned.
     *         Otherwise, the function returns 0.
     */ 
    virtual SenderList* lookupSender(const char* senderName) 
      {return process.getFirstSender()->lookup(name,senderName);}

    /**
     * The functions searches for a receiver with a given name.
     * @param receiverName The name of the receiver.
     * @return If the receiver was found, a pointer to it is returned.
     *         Otherwise, the function returns 0.
     */ 
    virtual ReceiverList* lookupReceiver(const char* receiverName)
      {return process.getFirstReceiver()->lookup(name,receiverName);}

    /**
     * The function returns the name of the process.
     * @return the name of the process.
     */
    virtual const char* getName() const {return name;}

    /**
     * The function returns a pointer to the process if it has the given name.
     * @param processName The name of the process that is searched for.
     * @return If the process has the required name, a pointer to it is
     *         returned. Otherwise, the function returns 0.
     */
    virtual PlatformProcess* getProcess(const char* processName)
    {
      if(!strcmp(name,processName))
        return &process;
      else
        return 0;
    }
};

class ProcessList;

/**
 * The class is a base class for process creators.
 */
class ProcessCreatorBase
{
  private:
    static List<ProcessCreatorBase*> list; /**< The list of all process creators. */

  protected:
    /**
     * The function creates a process.
     * @param robot The robot the new process will be associated with.
     * @return A pointer to the new process.
     */
    virtual ProcessBase* create(Robot* robot) = 0;

  public:
    /**
     * Constructor.
     */
    ProcessCreatorBase() {list.insert(this);}

  friend class ProcessList;
};

/**
 * The template class instatiates creators for processes of a certain type.
 */
template <class T> class ProcessCreator : public ProcessCreatorBase
{
  private:
    const char* name; /**< The name of the process that will be created. */

  protected:
    /**
     * The function creates a process.
     * @param robot The robot the new process will be associated with.
     * @return A pointer to the new process.
     */
    ProcessBase* create(Robot* robot) {return new T(robot,name);}

  public:
    /**
     * Constructor.
     * @param name The name of the process that will be created.
     */
    ProcessCreator(const char* name) {this->name = name;}
};

/**
 * The class implements a list of processes.
 */
class ProcessList : public List<ProcessBase>
{
  public:
    static int numberOfProcesses; /**< The number of processes already constructed for one robot. */

    /**
     * Constructor.
     * Creates a process for each process constructor and inserts them
     * into the list.
     */
    ProcessList()
    {
      numberOfProcesses = 0;
      for(List<ProcessCreatorBase*>::Pos i = ProcessCreatorBase::list.getFirst(); i; ++i)
      {
        insert(ProcessCreatorBase::list[i]->create((Robot*) this));
        ++numberOfProcesses;
      }

    }

    void announceStop()
    {
      for(Pos i = getFirst(); i; ++i) 
        (*this)[i].announceStop();
    }

    /**
     * The function starts all processes in the list.
     */
    void start() 
    {
      for(Pos i = getFirst(); i; ++i) 
        (*this)[i].start();
    }
};

/**
 * The macro MAKE_PROCESS instatiates a process creator.
 * As a convention, it should be used in the last line of the 
 * source file. For each process, MAKE_PROCESS must exactly be used
 * once.
 * @param className The type of the class that will later be instantiated
 *                 as a process.
 */
#define MAKE_PROCESS(className) \
  ProcessCreator<ProcessFrame<className> > _create##className(#className)

/**
 * The macro declares a sender for motor commands.
 * It must be used inside a declaration of a process, after the macro DEBUGGING.
 */
#define SENDER_MOTORCOMMANDS \
  Sender<MotorCommands> theMotorCommandsSender

/**
 * The macro initializes a sender for motor commands. It must be part of the 
 * initializer list of the constructor of the process.
 * @param blocking Decides whether this sender blocks the execution of the next frame
 *                 until all connected receivers have requested a new package.
 */
#define INIT_SENDER_MOTORCOMMANDS(blocking) \
  theMotorCommandsSender(this,"Sender.OCommandVectorData.S",blocking)

/**
 * Streaming operator that converts the portable class MotorCommands into system-dependent motor commands.
 * @param stream The stream to which is written.
 * @param motorCommands The system independent motor commands object.
 * @return The stream.
 */ 
inline Out& operator<<(Out& stream,const MotorCommands& motorCommands)
{
  stream.write(&motorCommands,sizeof(MotorCommands));
  return stream;
}

/**
 * The macro declares a sender for sound data.
 * It must be used inside a declaration of a process, after the macro DEBUGGING.
 */
#define SENDER_SOUNDDATA \
  Sender<SoundData> theSoundDataSender

/**
 * The macro initializes a sender for sound data. It must be part of the 
 * initializer list of the constructor of the process.
 * @param blocking Decides whether this sender blocks the execution of the next frame
 *                 until all connected receivers have requested a new package.
 */
#define INIT_SENDER_SOUNDDATA(blocking) \
  theSoundDataSender(this,"Sender.OSoundVectorData.S",blocking)

inline Out& operator<<(Out& stream,const SoundData& soundData)
{
  stream.write(&soundData,sizeof(SoundData));
  return stream;
}

#endif

/*
 * Change Log:
 *
 * $Log: ProcessFramework.h,v $
 * Revision 1.1  2003/10/07 10:06:59  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.3  2003/09/26 15:30:28  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.2  2003/09/26 11:34:07  juengel
 * no message
 *
 * Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.3  2002/12/18 16:22:55  dueffert
 * doxygen docu corrected
 *
 * Revision 1.2  2002/11/27 13:50:06  dueffert
 * doxygen docu added
 *
 * Revision 1.1  2002/09/10 15:40:05  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.6  2002/07/23 13:39:39  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - new design of debugging architecture
 * - exchanged StaticQueue with MessageQueue
 *
 * Revision 1.5  2002/07/13 11:46:18  roefer
 * New command and sound senders
 *
 * Revision 1.4  2002/07/13 10:54:58  roefer
 * New command and sound sender
 *
 *
 */