/**
 * @file Platform/Win32/RoboCupCtrl2.h
 *
 * This file declares the class RoboCupCtrl.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */
#ifndef __RoboCupCtrl_H__
#define __RoboCupCtrl_H__

#include "SimRobXP/Controller/Controller.h"
#include "Robot2.h"
#include "Representations/Perception/SensorData.h"
#include "Tools/List.h"
#include "Tools/Math/Pose2D.h"

/**
 * The class implements the SimRobot controller for RoboCup.
 */
class RoboCupCtrl : public Controller
{
  private:
    static RoboCupCtrl* controller; /**< A pointer to the SimRobot controller. */

  protected:
    static std::string robotName; /**< The name of the robot currently constructed. */
    List<Robot> robots; /**< The list of all robots. */
    unsigned int currentRobot; /**< The robot currently handled by the controller. */
    RobotDesign::Design currentDesign; /**< The design of the robot currently constructed. */
    SimObject* obj; /**< The current robot constructed. */
    bool simTime; /**< Switches between simulation time mode and real time mode. */
    int time; /**< The simulation time. */
    unsigned lastTime; /**< The last time execute was called. */

    /**
     * The function returns the current system time.
     * @return The time in milliseconds.
     */
    unsigned getCurrentSystemTime() const;

    /**
     * The function connects all senders of a robot.
     * @param robot The robot.
     */
    void connect(Robot& robot);

   private:
    /**
     * The function searches for a sender that sends to a certain receiver.
     * @param fileName The name of the file to search.
     * @param searchedReceiverName The name of the receiver.
     * @return The name of the sender or "" if it has not been found.
     */
    std::string getSenderName(const char* fileName, const char* searchedReceiverName);

    /**
     * The function adapts the colors of the simulation to the color table.
     */
    void adaptColors();

    /**
     * The function determines the color class for a RGB color.
     * @param colorTable The color table used for segmentation.
     * @param r The red component of the color.
     * @param g The green component of the color.
     * @param b The blue component of the color.
     * @return The color class.
     */
    colorClass getColorClass(const ColorTable& colorTable, int r, int g, int b) const;

  public:
    /**
     * Constructor.
     */
    RoboCupCtrl();

    /**
     * Destructor.
     */
    ~RoboCupCtrl();

    /**
     * The function is called in each simulation step.
     */
    void execute();

    /**
     * The function returns a pointer to the SimRobot controller.
     * @return The pointer to the SimRobot controller.
     */
    static RoboCupCtrl* getController() {return controller;}

    /**
     * The function returns the full name of the robot currently constructed.
     * @return The name of the robot.
     */
    static std::string& getRobotFullName() {return robotName;}

    /**
     * The function returns the name of the robot associated to the current thread.
     * @return The name of the robot.
     */
    const char* getRobotName() const;

    /**
     * The function returns the index of the robot associated to the current thread.
     * @return The index of the robot. It refers to the sequence in which the robots where defined.
     */
    int getRobotIndex() const;

    /**
     * The function returns the index of the current process.
     * @return The index of the process. It refers to the sequence in which the processes where defined.
     */
    int getProcessIndex() const;

    /**
     * The function return the robot design of the robot associated to the current thread.
     * @return The design.
     */
    RobotDesign::Design getRobotDesign() const;

     /**
     * The function determines the 2-D pose of a SimRobot object.
     * @param obj The object of which the pose will be determined.
     * @return The 2-D pose of the specified object.
     */
    Pose2D getPose2D(SimObject* obj);

    /**
     * The function sets the 2-D pose of a SimRobot object.
     * @param obj The object of which the pose will be determined.
     * @param pose The desired 2-D pose of the specified object.
     * @param sensorData The sensor data the body tilt is calculated from.
     *                   If not specified, the body tilt is not changed.
     */
    void setPose2D(SimObject* obj, const Pose2D& pose, const SensorData* sensorData = 0);

    /**
     * The function returns the simulation time.
     * @return The pseudo-time in milliseconds.
     */
    unsigned getTime() const;
};

#endif

/*
 * Change log :
 *
 * $Log: RoboCupCtrl2.h,v $
 * Revision 1.9  2004/03/10 23:55:28  roefer
 * ERS7 support for log files
 *
 * Revision 1.8  2004/03/03 08:02:25  roefer
 * Missing comments added
 *
 * Revision 1.7  2004/02/29 14:54:40  roefer
 * Simulator performs a maximum of 25 simulation steps per second
 *
 * Revision 1.6  2004/01/28 21:55:50  roefer
 * RobotDimensions revised
 *
 * Revision 1.5  2004/01/17 20:09:03  roefer
 * Simulator calculates robot pose based on class Kinematics now
 *
 * Revision 1.4  2004/01/17 19:19:18  roefer
 * Simulator calculates robot pose based on class Kinematics now
 *
 * Revision 1.3  2003/11/17 17:42:29  dueffert
 * includes fixed
 *
 * Revision 1.2  2003/10/21 15:05:51  roefer
 * Added the oracle
 *
 * Revision 1.1  2003/10/14 07:34:16  roefer
 * Support files for SimRobXP added, not finished yet
 *
 */
