/**
* @file Platform/Win32Linux/MessageQueueBase.cpp
* 
* Implementation of class MessageQueueBase for Win32.
*
* @author Martin Ltzsch
*/

#include "MessageQueueBase.h"
#include "Platform/GTAssert.h"
#include "Tools/Debugging/Stopwatch.h"
#include "Tools/Debugging/DebugDrawings.h"
#include <stdlib.h>
#include <memory.h>

Message::Message()
: size(0), readPosition(0), data(0), timeStamp(0), id(undefined),
teamColor(Player::undefinedTeamColor), playerNumber(Player::undefinedPlayerNumber),
messageWasSentFromAPhysicalRobot(false)
{
}

Message::~Message()
{
  if(data != 0) 
    free(data);
}

int Message::getSize() const
{
  return size;
}

unsigned long Message::getTimeStamp() const
{
  return timeStamp;
}

void Message::setTimeStamp(unsigned long timeStamp)
{
  this->timeStamp = timeStamp;
}

MessageID Message::getMessageID() const
{
  return id;
}

void Message::setMessageID(MessageID id)
{
  this->id = id;
}

Player::teamColor Message::getTeamColor() const
{
  return (Player::teamColor)teamColor;
}

void Message::setTeamColor(Player::teamColor teamColor)
{
  this->teamColor=(char)teamColor;
}

Player::playerNumber Message::getPlayerNumber() const
{
  return (Player::playerNumber)playerNumber;
}

void Message::setPlayerNumber(Player::playerNumber playerNumber)
{
  this->playerNumber=(char)playerNumber;
}

bool Message::getMessageWasSentFromAPhysicalRobot() const
{
  return messageWasSentFromAPhysicalRobot;
}

void Message::setMessageWasSentFromAPhysicalRobot(bool b)
{
  this->messageWasSentFromAPhysicalRobot = b;
}

void Message::resetReadPosition()
{
  readPosition = 0;
}

void Message::write(const void* p,int s)
{
  data = (char*)realloc(data,size + s);
  ASSERT(data != 0);
  memcpy(data + size, p, s);
  size += s;
}

void Message::read(void* p,int s)
{
  ASSERT(s + readPosition <= size);
  memcpy(p,data + readPosition,s);
  readPosition += s;
}

const char* Message::getData() const
{
  return data;
}

bool Message::eof() const
{
  return readPosition == size;
}

MessageQueueBase::MessageQueueBase()
{
  clear();
}

MessageQueueBase::~MessageQueueBase()
{
  messages.clear();
}

void MessageQueueBase::clear()
{
  messages.clear();
  messages.insert(new Message);
  selectedMessageForReading = List<Message>::Pos();
}

int MessageQueueBase::getNumberOfMessages() const
{
  return messages.getSize() - 1;
}

void MessageQueueBase::write(const void* p,int size)
{
  messages[messages.getLast()].write(p,size);
}

void MessageQueueBase::finishMessage(MessageID id, unsigned long timeStamp,
                                     Player::teamColor teamColor,
                                     Player::playerNumber playerNumber,
                                     bool messageWasSentFromAPhysicalRobot)
{
  List<Message>::Pos pos = messages.getLast();
  ASSERT(messages[pos].getSize() > 0);
  messages[pos].setMessageID(id);
  messages[pos].setTimeStamp(timeStamp);
  messages[pos].setTeamColor(teamColor);
  messages[pos].setPlayerNumber(playerNumber);
  messages[pos].setMessageWasSentFromAPhysicalRobot(messageWasSentFromAPhysicalRobot);
  messages.insert(new Message);
}

void MessageQueueBase::removeRepetitions()
{
  int messagesPerType[numOfMessageIDs];
  int fieldDrawings[Drawings::numberOfFieldDrawings];
  int imageDrawings[Drawings::numberOfImageDrawings];
  int stopWatches[Stopwatch::numberOfStopwatchEventIDs];
  memset(messagesPerType,0,sizeof(messagesPerType));
  memset(fieldDrawings,0,sizeof(fieldDrawings));
  memset(imageDrawings,0,sizeof(imageDrawings));
  memset(stopWatches,0,sizeof(stopWatches));
  int i;
  List<Message>::Pos pos = messages.getFirst();
  for (i=0;i<messages.getSize()-1;i++)
  {
    if(messages[pos].getMessageID() == idDebugDrawingFinished)
    {
      if(Drawings::TypeOfDrawing(messages[pos].getData()[1]) == Drawings::drawingOnField)
        fieldDrawings[messages[pos++].getData()[0]]++;
      else
        imageDrawings[messages[pos++].getData()[0]]++;
    }
    else if(messages[pos].getMessageID() == idStopwatch)
      stopWatches[messages[pos++].getData()[0]]++;
    else
      messagesPerType[messages[pos++].getMessageID()]++;
  }
  pos = messages.getFirst();
  for (i=0;i<messages.getSize()-1;i++)
  {
    bool copy;
    switch (messages[pos].getMessageID())
    {
    case idDebugDrawingFinished:
      if(Drawings::TypeOfDrawing(messages[pos].getData()[1]) == Drawings::drawingOnField)
        copy = --fieldDrawings[messages[pos].getData()[0]] == 0;
      else 
        copy = --imageDrawings[messages[pos].getData()[0]] == 0;
      break;
    case idDebugDrawing2:
      if(Drawings::TypeOfDrawing(messages[pos].getData()[2]) == Drawings::drawingOnField)
        copy = fieldDrawings[messages[pos].getData()[1]] <= 1;
      else
        copy = imageDrawings[messages[pos].getData()[1]] <= 1;
      break;
    case idStopwatch:
      copy = stopWatches[messages[pos].getData()[0]] == 0;
      break;
    case idText:
      //allow up to 20 idTexts in realtime mode, because several of that kind may be send simultaneously:
      copy = --messagesPerType[messages[pos].getMessageID()] <= 20;
      break;
    default:
      copy = --messagesPerType[messages[pos].getMessageID()] == 0;
    }
    if(copy)
      ++pos;
    else
      messages.remove(pos);
  }
  selectedMessageForReading=messages.getFirst();
}

void MessageQueueBase::setSelectedMessageForReading(int message)
{
  ASSERT(message >= 0);
  ASSERT(message < messages.getSize() - 1);
  selectedMessageForReading = getPos(message);
  resetReadPosition();
}

void MessageQueueBase::read(void* p,int size)
{
  messages[selectedMessageForReading].read(p,size);
} 

const char* MessageQueueBase::getData() const
{
  return messages[selectedMessageForReading].getData();
}

int MessageQueueBase::getMessageSize() const
{
  return messages[selectedMessageForReading].getSize();
}

Player::teamColor MessageQueueBase::getTeamColor() const
{
  return messages[selectedMessageForReading].getTeamColor();
}

Player::playerNumber MessageQueueBase::getPlayerNumber() const
{
  return messages[selectedMessageForReading].getPlayerNumber();
}

bool MessageQueueBase::getMessageWasSentFromAPhysicalRobot() const
{
  return messages[selectedMessageForReading].getMessageWasSentFromAPhysicalRobot();
}

void MessageQueueBase::resetReadPosition()
{
  messages[selectedMessageForReading].resetReadPosition();
}

unsigned long MessageQueueBase::getTimeStamp() const
{
  return messages[selectedMessageForReading].getTimeStamp();
}

unsigned long MessageQueueBase::getTimeStamp(int message) const
{
  return messages[getPos(message)].getTimeStamp();
}

MessageID MessageQueueBase::getMessageID() const
{
  return messages[selectedMessageForReading].getMessageID();
}

bool MessageQueueBase::eof() const
{
  return messages[selectedMessageForReading].eof();
}

List<Message>::Pos MessageQueueBase::getPos(int message) const
{
  ASSERT(message>=0);
  ASSERT(message < messages.getSize() - 1);
  
  List<Message>::Pos pos = messages.getFirst();
  for (int i=1; i<=message; i++) pos++;
  return pos;
}


/*
* Change log :
* 
* $Log: MessageQueueBase.cpp,v $
* Revision 1.3  2004/03/01 14:58:46  dueffert
* allow up to 20 idTexts in realtime mode
*
* Revision 1.2  2003/12/06 23:23:55  loetzsch
* messages in a MessageQueue now contain
* - the team color of the robot which sent the message
* - the player number of the robot which sent the message
* - if the message was sent from a physical robot or not
*
* Revision 1.1  2003/10/07 10:07:00  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 11:36:26  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.7  2003/06/22 12:55:24  roefer
* Stopwatch RT compatible (2nd try)
*
* Revision 1.6  2003/06/13 21:46:47  roefer
* Real time mode now supports debug drawings
*
* Revision 1.5  2003/01/26 19:38:07  loetzsch
* In setSelectedMessageForReading resetReadPosition() now is called
*
* Revision 1.4  2002/12/16 23:50:16  roefer
* Minor changes in removeRepetitions
*
* Revision 1.3  2002/12/05 16:13:19  dueffert
* started implementing realtime sending
*
* Revision 1.2  2002/10/07 11:21:10  dueffert
* includes corrected for gcc3.2
*
* Revision 1.1  2002/09/10 15:40:05  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.2  2002/08/22 14:41:03  risler
* added some doxygen comments
*
* Revision 1.1  2002/08/01 12:52:27  roefer
* RouterCtrl and TcpConnection added
*
* Revision 1.3  2002/07/23 20:43:51  roefer
* selectedMessageForReading is List<Message>::Pos now (is faster)
*
* Revision 1.2  2002/07/23 16:40:12  roefer
* Router and SimGT2002 adapted to new message queue and streams
*
* Revision 1.1  2002/07/23 13:36:39  loetzsch
* - exchanged StaticQueue by MessageQueue with platform dependend
*   base classes
*
*/
