/**
 * @file Platform/Win32Linux/TcpConnection.h
 *
 * Declaration of class TcpConnection.
 *
 * @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
 */

#ifndef __TcpConnection_H__
#define __TcpConnection_H__

#ifdef _WIN32
#include <winsock.h>
#else
#include <netinet/in.h>
#endif

/**
 * @class TcpConnection
 * The class implements a tcp connection.
 */
class TcpConnection
{
  private:
    int createSocket, /**< The handle of the basic socket. */
        transferSocket; /**< The handle of the actual transfer socket. */
    sockaddr_in address; /**< The socket address. */
    bool ack,
         client;

    /**
     * The function tries to receive a package.
     * @param buffer A pointer that will be initialized with the address of
     *               the data received. It is valid and has to be freed only
     *               if the function returns a value larger than 0.
     * @return Success of the function: -1: failure, 0: nothing read, 
     *         > 0: success, size of data, and buffer points to data.
     */
    int receive(char*& buffer);

    /**
     * The function reads a number of bytes into a buffer.
     * @param buffer The buffer the data will be read into. 
     *               It must be large enough to contain "size" bytes.
     * @param size The number of bytes to be read.
     * @return Did the function succeed?
     */
    bool read(char* buffer,int size);

    /**
     * The function writes a number of bytes to a socket.
     * @param buffer The data that will be written. 
     * @param size The number of bytes to be written.
     * @return Did the function succeed?
     */
    bool send(const char* buffer,int size);

    /**
     * The function closes the transfer socket.
     */
    void closeTransferSocket();

  public:
    /**
     * Default constructor.
     */
    TcpConnection();

    /**
     * Constructor.
     * The constructor will first try to connect another process as
     * a client. If this does not work, it will open the port as a
     * server.
     * @param ip The ip address of the communication partner.
     * @param port The port under which will be communicated. 
     */
    TcpConnection(const char* ip,int port);

    /**
     * Destructor.
     */
    ~TcpConnection();

    /**
     * The function will first try to connect another process as
     * a client. If this does not work, it will open the port as a
     * server.
     * @param ip The ip address of the communication partner.
     * @param port The port under which will be communicated. 
     */
    void connect(const char* ip,int port);

    /** 
     * The function sends and receives data.
     * @param dataToSend The data to be send. The function will not free the buffer.
     * @param sendSize The size of data to send. If 0, no data is sent.
     * @param dataRead If data has been read, the parameter is initialzed with
     *                 the address of a buffer pointing to that data. The 
     *                 buffer has to be freed manually.
     * @param readSize The size of the block read. "dataRead" is only valid 
     *                 (and has to be freed) if this parameter contains a 
     *                 positive number after the call to the function.
     * @return Returns true if the data has been sent.
     */
    bool sendAndReceive(char* dataToSend,int sendSize,char*& dataRead,int& readSize);

    /**
     * The function states whether the connection is still established.
     * @return Does the connection still exist?
     */
    bool isConnected() const {return transferSocket > 0;}

    /**
     * The function states whether this system is the client in the connection.
     * @return Is it the client?
     */
    bool isClient() const {return client;}
};

#endif

/*
 * Change log :
 * 
 * $Log: TcpConnection.h,v $
 * Revision 1.2  2004/04/14 14:47:57  roefer
 * Only wait one second if a robot cannot be connected
 *
 * Revision 1.1  2003/10/07 10:07:00  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/08/17 18:35:37  roefer
 * Communication with router standardized and synchronized
 *
 * Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.1  2002/09/10 15:40:05  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.3  2002/08/22 14:41:03  risler
 * added some doxygen comments
 *
 * Revision 1.2  2002/08/04 17:53:18  roefer
 * SimGT2002 connection to physical robots added
 *
 * Revision 1.1  2002/08/01 12:52:27  roefer
 * RouterCtrl and TcpConnection added
 *
 */
