/** 
* @file Processes/CD/Cognition.cpp
* Implementation of the process Cognition
*
* @author Martin Ltzsch
*/

#include "Cognition.h"
#include "Representations/Perception/JPEGImage.h"
#include "Representations/Perception/LowResImage.h"
#include "Tools/Debugging/Stopwatch.h"
#include "Tools/RobotConfiguration.h"
#include "Platform/GTAssert.h"
#include "Tools/Location.h"


Cognition::Cognition() : 
INIT_DEBUGGING,
INIT_RECEIVER_SENSORDATA(false),
INIT_RECEIVER_IMAGE(false),
INIT_RECEIVER(GameControlData,false),
INIT_RECEIVER(TeamMessage1,false),
INIT_RECEIVER(TeamMessage2,false),
INIT_RECEIVER(TeamMessage3,false),

INIT_SENDER_MOTORCOMMANDS(true),
INIT_SENDER_SOUNDDATA(false),
INIT_SENDER(TeamMessage1,false),
INIT_SENDER(TeamMessage2,false),
INIT_SENDER(TeamMessage3,false),
INIT_SENDER(OdometryData,false)
{
  //  PRINT("Cognition::Cognition");
  debugOut.setSize(200000);
  debugIn.setSize(400000);
  
  SensorDataProcessorInterfaces sensorDataProcessorInterfaces(    
    headState,
    theSensorDataBufferReceiver,theImageReceiver.frameNumber,
    bodyPercept, cameraMatrix, psdPercept);
  pSensorDataProcessor = new SensorDataProcessorSelector(moduleHandler, sensorDataProcessorInterfaces);
  
  RobotStateDetectorInterfaces robotStateDetectorInterfaces(
    bodyPercept, collisionPercept, robotState);
  pRobotStateDetector = new RobotStateDetectorSelector(moduleHandler, robotStateDetectorInterfaces);
  
  LEDControlInterfaces ledControlInterfaces(
    ledRequest,
    wLanStatus1, 
    wLanStatus2, 
    wLanStatus3, 
    theMotorCommandsSender.ledValue);
  pLEDControl = new DefaultLEDControl(ledControlInterfaces);
  
  SoundControlInterfaces soundControlInterfaces(
    soundRequest, theSoundDataSender);
  pSoundControl = new SoundControlSelector(moduleHandler, soundControlInterfaces);
  
  SensorActorLoopInterfaces sensorActorLoopInterfaces(
    theImageReceiver,
    theSensorDataBufferReceiver,
    theMotorCommandsSender.pidData,
    theMotorCommandsSender.jointDataBuffer,
    ledRequest
    );
  pSensorActorLoop = new SensorActorLoopSelector(moduleHandler, sensorActorLoopInterfaces);

  teamMessageCollection.setInTeamMessages(theTeamMessage1Receiver,
    theTeamMessage2Receiver,
    theTeamMessage3Receiver);
  
  teamMessageCollection.setOutTeamMessages(theTeamMessage1Sender,
    theTeamMessage2Sender,
    theTeamMessage3Sender);
  
}

Cognition::~Cognition()
{
  delete pSensorDataProcessor;
  delete pRobotStateDetector;
  delete pLEDControl;
  delete pSoundControl;
  delete pSensorActorLoop;
}

int Cognition::main() 
{
  if (theSensorDataBufferReceiver.receivedNew())
  {
    WATCH(sendSensorData,idSensorData,bin,theSensorDataBufferReceiver);
    STOP_TIME_ON_REQUEST(sensorDataProcessor, pSensorDataProcessor->execute(); );
  }
  
  if (theImageReceiver.receivedNew())
  {
    WATCH(sendImage,idImage,bin,SEND_IMAGE(theImageReceiver,cameraMatrix));
    WATCH(sendSensorData,idSensorData,bin,theSensorDataBufferReceiver);
    teamMessageCollection.processMessages();
    
    STOP_TIME_ON_REQUEST(sensorDataProcessor, pSensorDataProcessor->execute(); );
    STOP_TIME_ON_REQUEST(robotStateDetector, pRobotStateDetector->execute(); );
    
    // hier neues Modul hin, wenn es mit 25 Hz mit Bildern arbeiten soll
  }
  pSensorActorLoop->execute();
  // hier neues Modul hin, wenn es mit 125 Hz ohne Bilder arbeiten soll
  
  theMotorCommandsSender.send();
  INFO(sendJointData,idJointData,bin,theMotorCommandsSender.jointDataBuffer.frame[0]);
  
  pLEDControl->execute();
  
  if(theSoundDataSender.requestedNew())
  {
    STOP_TIME_ON_REQUEST(soundControl,pSoundControl->execute(););
    if (theSoundDataSender.isInUse) theSoundDataSender.send();            
  }
  
  wLanStatus1 = theTeamMessage1Receiver.isActual();
  wLanStatus2 = theTeamMessage2Receiver.isActual();
  wLanStatus3 = theTeamMessage3Receiver.isActual();
  
  theDebugSender.send();
  
  teamMessageCollection.send (outgoingBehaviorTeamMessage);
  
  if ( (teamMessageCollection.processOutMessages())) { 
    theTeamMessage1Sender.send();
    theTeamMessage2Sender.send();
    theTeamMessage3Sender.send();
  }
  return 1;
}

void Cognition::init()
{
  // !!! don't remove that lines, it is needed to trigger the Debug process to send 
  // debug requests back.
  OUTPUT(idText,text,"player: " << Player::getTeamColorName(getPlayer().getTeamColor())
    << " " << Player::getPlayerNumberName(getPlayer().getPlayerNumber())
    << ", MAC Address: " << getRobotConfiguration().getMacAddressString());
}

bool Cognition::handleMessage(InMessage& message)
{
  switch (message.getMessageID())
  {
  case idSensorData:
    message.bin >> theSensorDataBufferReceiver;
    return true;
  case idImage:
    message.bin >> RECEIVE_IMAGE(theImageReceiver,cameraMatrix);
    return true;
  case idJPEGImage:
    {
      JPEGImage jpe;
      message.bin >> jpe >> cameraMatrix;
      jpe.toImage(theImageReceiver);
      return true;
    }
    return true;
  case idLowResImage:
    {
      LowResImage lrImage(theImageReceiver);
      message.bin >> lrImage >> cameraMatrix;
      return true;
    }
  case idLEDRequest:
    message.bin >> ledRequest;
    return true;
  case idGameControlData:
    message.bin >> theGameControlDataReceiver;
    return true;
  case idGlobalGameControlData:
    {
      GlobalGameControlData ggcd;
      message.bin >> ggcd;
      theGameControlDataReceiver.translateFromGlobalGameControlData(ggcd,getPlayer());
    }
    return true;
  case idSoundRequest:
    message.bin >> soundRequest;
    return true;
  case idCameraParameters:
    {
      message.bin >> cameraParameters;
      this->setCameraParameters(cameraParameters);
      return true;
    }
  case idBodyOffsets:
    getRobotConfiguration().handleMessage(message);
    return true;
  default:
    {
      return Process::handleMessage(message);
    }
  }
}

MAKE_PROCESS(Cognition);

/*
* Change log :
* 
* $Log: Cognition.cpp,v $
* Revision 1.9  2004/04/07 13:00:46  risler
* ddd checkin after go04 - second part
*
* Revision 1.2  2004/04/07 11:44:07  risler
* added sending low res images
* added Image::setCameraInfo
*
* Revision 1.1.1.1  2004/03/29 08:28:46  Administrator
* initial transfer from tamara
*
* Revision 1.8  2004/01/13 14:13:10  loetzsch
* bug fix
*
* Revision 1.7  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.6  2003/11/29 07:40:19  roefer
* Doxygen comments corrected
*
* Revision 1.5  2003/10/29 19:12:21  juengel
* Added module SensorActorLoop.
*
* Revision 1.4  2003/10/27 22:07:26  roefer
* Minor bugs fixed
*
* Revision 1.3  2003/10/27 19:47:30  loetzsch
* no message
*
* Revision 1.2  2003/10/27 19:18:25  loetzsch
* bug fix
*
* Revision 1.1  2003/10/27 19:02:20  loetzsch
* no message
*
*/
