/** 
* @file Processes/CD/Debug.cpp
* 
* Implementation of class Debug.
*
* @author Martin Ltzsch
*/
 
#include "Debug.h"
#include "Tools/RobotConfiguration.h"
#include "Tools/Debugging/Debugging.h"
#include "Platform/GTAssert.h"
#include "Platform/SystemCall.h"

Debug::Debug() : 
  INIT_DEBUGGING,
  INIT_DEBUG_RECEIVER(Cognition),

  INIT_DEBUG_SENDER(Cognition),

  mode(QueueFillRequest::immediateReadWrite),
  saveOrSendTime(0),
  fin(0)
{
  debugIn.setSize(400000);
  debugOut.setSize(2000000);

  theCognitionReceiver.setSize(200000);
  theCognitionSender.setSize(400000);
}

Debug::~Debug() 
{
  if (fin)
    delete(fin);
}

int Debug::main() 
{
  //if somebody attached messages to that file, process them now
  if ((fin->exists())&&(! fin->eof()))
  {
    *fin >> debugIn;
    debugIn.handleAllMessages(*this);
    debugIn.clear();
  }


  INFO(printRobotStatus,idText,text,"Free mem: "<<SystemCall::getFreeMem()<<" bytes"<<endl<<"Remaining capacity: "<<SystemCall::getRemainingPower()<<"%"<<endl << "MAC Address: " << getRobotConfiguration().getMacAddressString());

  // add the messages from the queues from the processes to the outgoing queue
  theCognitionReceiver.moveAllMessages(debugOut);
  
  switch (mode)
  {
  case QueueFillRequest::overwriteOlder:
    debugOut.removeRepetitions();
    break;
  case QueueFillRequest::rejectAll:
    debugOut.clear();
    break;
  case QueueFillRequest::toStickNSeconds:
    if ((SystemCall::getCurrentSystemTime() > saveOrSendTime)&&(saveOrSendTime != 0))
    {
      // save the outgoing queue to the memory stick
      if (!debugOut.isEmpty())
      {
        OutBinaryFile fout("Logfile.log");
        fout << debugOut;
      }
      saveOrSendTime = 0;
    }
    break;
  }
  
  // send the queues to the processes
  theCognitionSender.send();

#if defined(_WIN32) || defined(WLAN)
  // send the outgoing queue to RobotControl 
  // (direct in WIN32, via TcpGateway in WLAN)
  if ((mode==QueueFillRequest::immediateReadWrite)||
      ((mode==QueueFillRequest::collectNSeconds)&&(saveOrSendTime < SystemCall::getCurrentSystemTime()))||
      (mode==QueueFillRequest::overwriteOlder))
  {
#ifndef _WIN32
    if (messageWasReceived) // Hallo Hack
#endif
      theDebugSender.send();
  }
#endif

  return 1; // Wait at least 1 millisecond, then recall main()
}

void Debug::init()
{
  fin = new InBinaryFile("requests.dat");
  if ((fin->exists())&&(! fin->eof()))
  {
    *fin >> debugIn;
  }
  debugIn.handleAllMessages(*this);
  debugIn.clear();
  messageWasReceived = false;
}

bool Debug::handleMessage(InMessage& message)
{
  messageWasReceived = true;

  switch(message.getMessageID())
  {
  case idDebugKeyTable: 
    Process::handleMessage(message);
  case idSolutionRequest:
  case idGenericDebugData:
  case idAlLxComm:
  case idBodyOffsets:
  case idSensorData:
  case idImage:
  case idJPEGImage:
  case idLowResImage:
  case idPercepts: 
  case idHeadControlMode:   
  case idMotionRequest: 
  case idLEDRequest: 
  case idWorldState: 
  case idColorTable64:
  case idColorTableTSL:
  case idCameraParameters:
  case idOracledWorldState:
  case idXabsl2DebugRequest:
  case idXabsl2IntermediateCode:
  case idGT2003BehaviorConfiguration:
  case idGameControlData:
  case idGlobalGameControlData:
  case idSoundRequest:
  case idSpecialPercept:
  case idParametersForGridImageProcessor2:
  case idGridImageProcessor2DebugParameters:
  case idLinesImageProcessorParameters:
  case idLinesSelfLocatorParameters:
  case idJoystickData:
  case idHeadMotionRequest: 
  case idOdometryData: 
  case idJointDataSequence:
  case idFourierCoefficients:
  case idInvKinWalkingParameters:
  case idPIDData:
  case idGameSpeed:
    message >> theCognitionSender; 
    return true;
  case idQueueFillRequest:
  	{
    QueueFillRequest qfr;
    message.bin >> qfr;
    mode = qfr.mode;
    saveOrSendTime = 1000*qfr.seconds + SystemCall::getCurrentSystemTime();
    return true;
    }
  default:
    return Process::handleMessage(message);
  }
}

MAKE_PROCESS(Debug);

/*
 * Change log :
 * 
 * $Log: Debug.cpp,v $
 * Revision 1.4  2004/04/07 13:00:46  risler
 * ddd checkin after go04 - second part
 *
 * Revision 1.2  2004/04/07 11:44:07  risler
 * added sending low res images
 * added Image::setCameraInfo
 *
 * Revision 1.1.1.1  2004/03/29 08:28:46  Administrator
 * initial transfer from tamara
 *
 * Revision 1.3  2003/11/29 07:40:19  roefer
 * Doxygen comments corrected
 *
 * Revision 1.2  2003/11/17 14:56:04  urban
 * added SensorBehaviorControl-solution "AlLx RemotePresence" and RobotControl-dialogbar "AlLx Joystick"
 *
 * Revision 1.1  2003/10/27 19:02:20  loetzsch
 * no message
 *
 */
