/** 
* @file Processes/CMD/Cognition.h
* Declaration of Cognition
*
* @author <a href="mailto:dueffert@informatik.hu-berlin.de">Uwe Dffert</a>
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
* @author <a href="mailto:martin@martin-loetzsch.de">Martin Ltzsch</a>
* @author <a href="mailto:risler@sim.informatik.tu-darmstadt.de">Max Risler</a>
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/

#ifndef __Cognition_H__
#define __Cognition_H__

#include "Tools/Process.h"
#include "Tools/Module/SolutionRequest.h"

#include "Modules/SensorDataProcessor/SensorDataProcessorSelector.h"
#include "Modules/ImageProcessor/ImageProcessorSelector.h"
#include "Modules/RobotStateDetector/RobotStateDetectorSelector.h"
#include "Modules/TeamBallLocator/TeamBallLocatorSelector.h"
#include "Modules/BallLocator/BallLocatorSelector.h"
#include "Modules/PlayersLocator/PlayersLocatorSelector.h"
#include "Modules/ObstaclesLocator/ObstaclesLocatorSelector.h"
#include "Modules/SelfLocator/SelfLocatorSelector.h"
#include "Modules/BehaviorControl/BehaviorControlSelector.h"
#include "Modules/SensorBehaviorControl/SensorBehaviorControlSelector.h"
#include "Modules/SpecialVision/DefaultSpecialVision.h"
#include "Modules/CollisionDetector/CollisionDetectorSelector.h"
#include "Modules/ColorTableMod/ColorTableModSelector.h"

#include "Representations/Perception/ColorTable.h"
#include "Representations/Perception/ColorTable64.h"
#include "Representations/Perception/ColorTableTSL.h"
#include "Representations/Perception/ColorTable32K.h"
#include "Representations/Perception/LandmarksPercept.h"
#include "Representations/Perception/BallPercept.h"
#include "Representations/Perception/PlayersPercept.h"
#include "Representations/Perception/LinesPercept.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/JoystickData.h"

#include "Platform/Sensors.h"
#include "Platform/UDPHandler.h"

#include "PackageMotionCognition.h"
#include "PackageCognitionMotion.h"
#include "Modules/WalkingEngine/UDParameterSet.h"



/**
* @class Cognition
*
* A Process for cognition related modules in the HU1 process layout.
*/
class Cognition: public Process, public Sensors
{
private:
  DEBUGGING;
  RECEIVER(SensorDataBuffer);
  RECEIVER(Image);
  RECEIVER(GameControlData);
  RECEIVER(PackageMotionCognition);
  RECEIVER(OdometryData);
  NET_SENDER(TeamMessage1);
  NET_SENDER(TeamMessage2);
  NET_SENDER(TeamMessage3);
  NET_RECEIVER(TeamMessage1);
  NET_RECEIVER(TeamMessage2);
  NET_RECEIVER(TeamMessage3);
#ifdef FIVEDOGS
  NET_RECEIVER(TeamMessage4);
  NET_SENDER(TeamMessage4);
#endif
  SENDER(PackageCognitionMotion);
  UDP_HANDLER;
  
public:
  /** constructor */
  Cognition();
  
  /** destructor */
  ~Cognition();
  
  /** is called from the framework once in every frame */
  virtual int main();
  
  /** is called direct before the first call of main() */
  virtual void init();
  
  /** 
  * The function handles incoming debug messages.
  * @param message the message to handle
  * @return Has the message been handled?
  */
  virtual bool handleMessage(InMessage& message);
  
  /* time when image processing was started */
  unsigned long timeOfImageProcessing;
  
protected:
  /** If true, then a new image was received in the current frame */
  bool processImage;
  
  /** If true, then new sensor data were received in the current frame */
  bool processSensorData;
  
  /** If true, then percepts were received or generated in the current frame */
  bool processPercepts;
  
  //!@name modules
  //!@{
  
#ifndef NO_COLOR_TABLE_SELECTOR
  /** a reference to the colortable selector */
  ColorTableModSelector* pColorTableMod;
#endif
  
  /** a reference to the image processor */
  ImageProcessorSelector* pImageProcessor;
  
  /** a reference to the sensor data processor */
  SensorDataProcessorSelector* pSensorDataProcessor;
  
  /** a reference to the collision detector */
  CollisionDetectorSelector* pCollisionDetector;
  
  /** a reference to the robot state detector */
  RobotStateDetectorSelector* pRobotStateDetector;
  
  /** a reference to the ball-locator */
  BallLocatorSelector* pBallLocator;
  
  /** a reference to the team ball locator */
  TeamBallLocatorSelector* pTeamBallLocator;
  
  /** a reference to the players-locator */
  PlayersLocatorSelector* pPlayersLocator;
  
  /** a reference to the obstacles-locator */
  ObstaclesLocatorSelector* pObstaclesLocator;
  
  /** a reference to the self-locator */
  SelfLocatorSelector* pSelfLocator;
  
  /** a reference to the behavior control */
  BehaviorControlSelector* pBehaviorControl;
  
  /** a reference to sensor based behavior control */
  SensorBehaviorControlSelector* pSensorBehaviorControl;
  
  /** a reference to the special vison module*/
  SpecialVision* pSpecialVision;
  
  //!@}
  
  //!@name representations
  //!@{
  
  /** the color tables */
  char colorTableBuffer[sizeof(ColorTable64) > sizeof(ColorTableTSL) 
    ? sizeof(ColorTable64) > sizeof(ColorTable32K) ? sizeof(ColorTable64) 
    : sizeof(ColorTable32K)
    : sizeof(ColorTableTSL) > sizeof (ColorTable32K)
    ? sizeof(ColorTableTSL) : sizeof(ColorTable32K)];
  /** the color table */
  ColorTable& colorTable;
  
  /** The offset and rotation of the camera */
  CameraMatrix cameraMatrix;
  
  /** Detected landmarks */
  LandmarksPercept landmarksPercept;
  
  /** Detected balls */
  BallPercept ballPercept;
  
  /** Detected players */
  PlayersPercept playersPercept;
  
  /** Detected obstacles */
  ObstaclesPercept obstaclesPercept;
  
  /** The collision percept */
  CollisionPercept collisionPercept;
  
  /** Detected lines */
  LinesPercept linesPercept;
  
  /** Detected switches */
  BodyPercept bodyPercept;
  
  /** A spot relative to the robot that was detected by the PSD sensor */
  PSDPercept psdPercept;
  
  /** A special percept */
//  SpecialPercept specialPercept;
  /** for the open challenge  */
  SpecialPercept& specialPercept;
  
  /** The request for the specialVision module*/
  SpecialVisionRequest specialVisionRequest;
  
  /** The request for calibration*/
  CalibrationRequest calibrationRequest;
  
  /** The sample set proxy linked to the sample set of the active locator */
  SampleSetProxy<PoseSample> selfLocatorSamples;
  
  /** The team message collection */
  TeamMessageCollection teamMessageCollection;
  
  /** The outgoing behavior team message */
  BehaviorTeamMessage outgoingBehaviorTeamMessage;
  
  /** Data produced by an extern sensor called joystick */
  JoystickData joystickData;
  //!@}
};


#endif // __Cognition_H__


/*
* Change log :
* 
* $Log: Cognition.h,v $
* Revision 1.19  2004/06/03 16:10:53  kerdels
* added new headcontrolmode
*
* Revision 1.18  2004/05/14 14:12:08  wachter
* - Added communication support for 5 robots
* - rewrote parts of team-communication to be faster and more stable
*
* Revision 1.17  2004/04/16 14:57:39  dueffert
* cleanup for Martins data flow graphics
*
* Revision 1.16  2004/04/14 06:37:17  loetzsch
* removed the hack "kickInProgress" from the BehaviorControlInterfaces
*
* Revision 1.15  2004/04/08 15:33:07  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.14  2004/04/07 14:42:56  risler
* moved LandsmarksState to Cognition directory, generated by SelfLocator
*
* Revision 1.13  2004/04/07 13:00:47  risler
* ddd checkin after go04 - second part
*
* Revision 1.2  2004/04/01 19:52:44  Charlie
* added LandmarkState
*
* Revision 1.1.1.1  2004/03/29 08:28:46  Administrator
* initial transfer from tamara
*
* Revision 1.12  2004/03/29 15:32:01  dueffert
* evolution mode idea added
*
* Revision 1.11  2004/03/15 12:36:53  dueffert
* measurement of free choosen MotionRequest allowed now
*
* Revision 1.10  2004/02/23 16:48:51  dueffert
* several improvements for measurement of walking
*
* Revision 1.9  2004/02/18 23:30:51  kerdels
* removed warning
*
* Revision 1.8  2004/02/18 14:49:59  dueffert
* behavior control can now change walking parameters
*
* Revision 1.7  2004/01/30 15:44:33  nistico
* Created a Color Table Selector, which permits to switch color tables at run time using the same image processor.
* If your image processor was designed to use only ColorTable64, leave it as the default choice and you should
* (hopefully) have no problems.
* In case of trouble, i put a conditional compilation switch in SolutionRequest.h, simply uncomment the definition of NO_COLOR_TABLE_SELECTOR; in that case, the code should get back to the original solution, but the coltable which will be looked upon first for loading will be ColorTable64 (as it's the default solution), so there's no reason anymore to remove the other color tables from the CVS.
*
* Revision 1.6  2004/01/26 14:14:50  wachter
* activated direct team-communication
*
* Revision 1.5  2004/01/20 12:40:08  nistico
* - Added support for ColorTable32K (65K elements in packed format)
* - RobotControl can now convert GT *.log files into AIBOVision (external ColorTable32K calibration tool) *.amv file format
*
* Revision 1.4  2003/11/29 07:40:19  roefer
* Doxygen comments corrected
*
* Revision 1.3  2003/11/24 15:31:20  dueffert
* SpecialPercept removed from PackageCognition
*
* Revision 1.2  2003/10/12 11:48:14  juengel
* Added CalibrationRequest.
*
* Revision 1.1  2003/10/07 10:07:00  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/26 21:23:20  loetzsch
* renamed class JointState to CollisionPercept
*
* Revision 1.2  2003/09/26 15:30:38  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.29  2003/05/13 11:41:45  goehring
* CollisionDetector added
*
* Revision 1.28  2003/05/12 14:08:40  brunn
* renamed selfLocationSampleSetProxy to selfLocatorSamples
* "Ha, glad am I that no one knew that Rumpelstiltskin I am styled"
*
* Revision 1.27  2003/05/12 12:28:13  brunn
* renamed sampleSetProxy to selfLocationSampleSetProxy
* added selfLocationSampleSetProxy to BehaviourControl-Interfaces
*
* Revision 1.26  2003/05/08 23:52:25  roefer
* SampleSet and SampleSetProxy added
*
* Revision 1.25  2003/05/02 18:26:17  risler
* SensorDataBuffer added
* replaced SensorData with SensorDataBuffer
* full SensorData resolution now accessible
*
* Revision 1.24  2003/05/01 17:09:09  loetzsch
* Redesign of ball modeling:
* - Modularized class BallPosition
* - splitted up module "BallLocator" into "BallLocator" for modeling of percepts
*   and "TeamBallLocator" for modelling communicated positions
* - Removed solution JumpingBallLocator
* - Splitted Solution DefaultBallLocator into DefaultBallLocator and DefaultTeamBallLocator
* - Renamed SensorFusionBallLocator to GaussBellTeamBallLocator
*
* Revision 1.23  2003/03/20 20:33:41  loetzsch
* removed some hacks
*
* Revision 1.22  2003/03/19 15:40:41  jhoffman
* GL simulator support improved
*
* Revision 1.21  2003/03/12 13:46:04  dasmuli
* PlayersPerceptor added to cognition, modules.cfg, DebugKey-table etc.
*
* Revision 1.20  2003/03/10 13:59:05  juengel
* Added ObstaclesLocator
*
* Revision 1.19  2003/03/06 12:03:13  dueffert
* re-order warning and nested comment warnings removed
*
* Revision 1.18  2003/03/05 14:07:10  roefer
* LinesPerceptor2 is now LinesImageProcessor
*
* Revision 1.17  2003/02/27 12:02:30  dueffert
* JoystickData added
*
* Revision 1.16  2003/02/21 18:32:04  roefer
* pColorTable -> colorTable finished
*
* Revision 1.15  2003/02/18 21:29:17  osterhues
* Changed all instances of ColorTable64 to new base class ColorTable
*
* Revision 1.14  2003/02/07 16:43:15  dueffert
* Bombay stuff added after cleanup
*
* Revision 1.13  2002/12/07 12:35:38  loetzsch
* world states and percepts now can be viewed without sending images
*
* Revision 1.12  2002/11/28 14:46:51  jhoffman
* added special percept for motion detection
*
* Revision 1.11  2002/11/26 19:19:24  loetzsch
* JPEG images are put into local processes
*
* Revision 1.10  2002/11/19 17:38:31  dueffert
* doxygen bugs corrected
*
* Revision 1.9  2002/11/07 17:30:10  loetzsch
* renamed Module SensorDataToMotionRequest to SensorBehaviorControl
*
* Revision 1.8  2002/10/10 13:09:50  loetzsch
* First experiments with the PSD Sensor
* - SensorDataProcessor now calculates PSDPercept
* - Added the PerceptBehaviorControl solution PSDTest
* - Added the RadarViewer3D to RobotControl, which can display the Points3D structure
*
* Revision 1.7  2002/10/04 10:20:47  loetzsch
* renamed bool imageIsNew to processImage,
* images are also processed when a new color table arrives
*
* Revision 1.6  2002/09/25 10:25:13  loetzsch
* removed the "executeVisionModules" variable
* from SolutionRequest and ModuleHandler.
*
* Revision 1.5  2002/09/18 19:52:36  loetzsch
* the head state is now sent from Motion to Cognition using the package.
*
* Revision 1.4  2002/09/17 23:55:22  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.3  2002/09/12 14:20:05  juengel
* Created a package for all data sent from Cognition to Motion.
*
* Revision 1.2  2002/09/12 12:40:58  juengel
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:41:25  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
* - Removed all process layouts
* - Added process layout CMD
*
* Revision 1.12  2002/09/07 13:36:58  loetzsch
* unified the vision modules into one module "ImageProcessor"
* - FloodFillRLEImageProcessor, BallPerceptor, LandmarksPerceptor
*   and PlayersPerceptor were are combined to the new solution
*   "BlobImageProcessor"
* - The GridImageProcessor and the SubPixelGradientCalculator became
*   a solution of "ImageProcessor"
*
* Revision 1.11  2002/07/23 13:40:51  loetzsch
* - new streaming classes
* - removed many #include statements
* - new design of debugging architecture
* - exchanged StaticQueue with MessageQueue
* - new debug message handling
* - general clean up
*
* Revision 1.10  2002/07/13 10:54:58  roefer
* New command and sound sender
*
* Revision 1.9  2002/06/20 00:40:21  Thomas Rfer
* WLan crash removed
*
* Revision 1.8  2002/06/12 11:34:29  roefer
* SimpleLinesPerceptor removed, PerceptBehaviorControl added
*
* Revision 1.7  2002/06/09 15:24:52  loetzsch
* Added TeamMessageCollection and BehaviorTeamMessage to the execute of BehaviorControl
*
* Revision 1.6  2002/06/02 23:21:09  roefer
* Single color table and progress in LinesSelfLocator
*
* Revision 1.5  2002/05/27 15:39:13  fischer
* Added SoundState (Sender and Receiver)
*
* Revision 1.4  2002/05/16 22:36:11  roefer
* Team communication and GTMath bugs fixed
*
* Revision 1.3  2002/05/14 21:04:04  hebbel
* processing of SoundDataIn added
*
* Revision 1.2  2002/05/10 17:29:43  juengel
* Added SpecialVision and SpecialPercept.
*
* Revision 1.1.1.1  2002/05/10 12:40:19  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.19  2002/05/05 18:52:03  loetzsch
* added
* - GameControlData,
* - Receivers for GameControlData
* - access by behavior to GameControlData
*
* Revision 1.18  2002/04/29 17:17:51  hebbel
* Put SoundPlay to Motion Process
*
* Revision 1.17  2002/04/28 19:19:46  giese
* SoundPlay added...
*
* Revision 1.16  2002/04/02 13:10:21  dueffert
* big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
*
* Revision 1.15  2002/02/05 20:02:16  risler
* handleDebugMessage now returns bool, added debug message handling to ImageProcessor
*
* Revision 1.14  2002/02/05 04:19:04  loetzsch
* replaced several team color hacks by getPlayer().getTeamColor()
*
* added a few new module selectors
*
* changed distribution of debug messages completely
*
* Revision 1.13  2002/02/03 14:37:58  juengel
* Drawing of the world state removed from Berlin2001BehaviorControl.
* Drawing method for world states added to PaintMethods.
* Drawing of the world state added to the Processes with BehaviorControl.
*
* Revision 1.12  2002/01/30 17:29:55  loetzsch
* handleDebugMessage um Parameter timestamp erweitert
*
* Revision 1.11  2002/01/22 14:56:30  juengel
* ColorTable verschicken angefangen.
*
* Revision 1.10  2002/01/06 13:50:29  juengel
* WalkDemo eingebaut
*
* Revision 1.9  2001/12/21 14:09:39  roefer
* Added several destructors
*
* Revision 1.8  2001/12/20 17:14:35  loetzsch
* Using debug requests from the memory stick.
*
* Revision 1.7  2001/12/19 16:03:55  bach
* SystemDataTypes replaced by Sensors
*
* Revision 1.6  2001/12/15 20:32:09  roefer
* Senders and receivers are now part of the processes
*
* Revision 1.5  2001/12/14 12:34:09  dueffert
* no message
*
* Revision 1.4  2001/12/12 18:08:55  loetzsch
* Streaming- Operatoren fr Bilder eingebaut, DebugKeyTable nicht- statisch gemacht, Debuggin Mechanismen weitergemacht, Bilder aus Logfiles in RobotControl anzeigen, Logfiles in HU1/Debug auf den Stick schreiben
*
* Revision 1.3  2001/12/10 17:47:08  risler
* change log added
*
*/
