/** 
* @file Processes/CMD/Debug.cpp
* 
* Implementation of class Debug.
*
* @author Martin Ltzsch
*/
 
#include "Debug.h"
#include "Tools/RobotConfiguration.h"
#include "Tools/Debugging/Debugging.h"
#include "Platform/GTAssert.h"
#include "Platform/SystemCall.h"

Debug::Debug() : 
  INIT_EXTERNAL_DEBUGGING,
  //INIT_DEBUGGING,

  INIT_DEBUG_RECEIVER(Cognition),
  INIT_DEBUG_RECEIVER(Motion),

  INIT_DEBUG_SENDER(Cognition),
  INIT_DEBUG_SENDER(Motion),

  mode(QueueFillRequest::immediateReadWrite),
  saveOrSendTime(0)
{
  //cout << "FREE Memory " << SystemCall::getFreeMem() << "\n" ;
  //cout << flush;
 
 
  debugIn.setSize(400000);
  debugOut.setSize(1500000); 
  
  theCognitionReceiver.setSize(200000);
  theCognitionSender.setSize(400000);

  theMotionReceiver.setSize(2000);
  theMotionSender.setSize(200000);
}

Debug::~Debug() 
{
}

int Debug::main() 
{
  INFO( printRobotStatus, idText, text,
    "Free mem: " << SystemCall::getFreeMem() << " bytes" << endl <<
    "Remaining capacity: " << SystemCall::getRemainingPower() << "%" << endl <<
    "MAC Address: " << getRobotConfiguration().getMacAddressString()
  );

  // add the messages from the queues from the processes to the outgoing queue
  theCognitionReceiver.moveAllMessages(debugOut);
  theMotionReceiver.moveAllMessages(debugOut);
  
  switch (mode)
  {
  case QueueFillRequest::overwriteOlder:
    debugOut.removeRepetitions();
    break;
  case QueueFillRequest::rejectAll:
    debugOut.clear();
    break;
  case QueueFillRequest::toStickNSeconds:
    if ((SystemCall::getCurrentSystemTime() > saveOrSendTime)&&(saveOrSendTime != 0))
    {
      // save the outgoing queue to the memory stick
      if (!debugOut.isEmpty())
      {
        OutBinaryFile fout("Logfile.log");
        fout << debugOut;
        debugOut.clear();
      }
      saveOrSendTime = 0;
    }
    break;
  }
  
  // send the queues to the processes
  theCognitionSender.send();
  theMotionSender.send();

#if defined(_WIN32) || defined(WLAN)
  // send the outgoing queue to RobotControl 
  // (direct in WIN32, via TcpGateway in WLAN)
  if ((mode==QueueFillRequest::immediateReadWrite)||
      ((mode==QueueFillRequest::collectNSeconds)&&(saveOrSendTime < SystemCall::getCurrentSystemTime()))||
      (mode==QueueFillRequest::overwriteOlder))
  {
#ifndef _WIN32
    if (messageWasReceived) // Hallo Hack
#endif
      theDebugSender.send();
  }
#endif

  return 1; // Wait at least 1 millisecond, then recall main()
}

void Debug::init()
{
  INIT_EXTERNAL_DEBUGGING_HANDLER;
  
  // read requests.dat
  //
  InBinaryFile fin("requests.dat");
  if ((fin.exists())&&(! fin.eof()))
    fin >> debugIn;

  debugIn.handleAllMessages(*this);
  debugIn.clear();
  messageWasReceived = false;
}

bool Debug::handleMessage(InMessage& message)
{
  messageWasReceived = true;

  switch(message.getMessageID())
  {
  case idSensorData:
  case idImage:
  case idJPEGImage:
  case idLowResImage:
  case idPercepts: 
  case idHeadControlMode:   
  case idMotionRequest: 
  case idLEDRequest: 
  case idWorldState: 
  case idColorTable64:
  case idColorTableTSL:
  case idCameraParameters:
  case idOracledWorldState:
  case idGT2003BehaviorConfiguration:
  case idGameControlData:
  case idGlobalGameControlData:
  case idSoundRequest:
  case idSpecialPercept:
  case idParametersForGridImageProcessor2:
  case idGridImageProcessor2DebugParameters:
  case idLinesImageProcessorParameters:
  case idLinesSelfLocatorParameters:
  case idJoystickData:
  case idTacticEntryArray:
  case idFreezeRequest:
  case idKalmanData:
  case idKickSelectionTable:
  case idMotionRatingBehaviorMessage:
    message >> theCognitionSender; 
    return true;
  case idHeadMotionRequest: 
  case idOdometryData: 
  case idJointDataSequence:
  case idFourierCoefficients:
  case idInvKinWalkingParameters:
  case idPIDData:
  case idGameSpeed:
  case idMotionNet:
  case idOdometryScale:
    message >> theMotionSender;
    return true;
  case idDebugKeyTable: 
    Process::handleMessage(message);
  case idSolutionRequest:
  case idGenericDebugData:
  case idUDParameters:
  case idUDEvolutionRequest:
  case idYetAnotherInvKinParams:
  case idYetAnotherInvKinPaws:
  case idAlLxComm:
  case idBodyOffsets:
  case idXabsl2DebugRequest:
  case idXabsl2IntermediateCode:
    message >> theCognitionSender; 
    message >> theMotionSender; 
    return true;
  case idQueueFillRequest:
  	{
    QueueFillRequest qfr;
    message.bin >> qfr;
    mode = qfr.mode;
    saveOrSendTime = 1000*qfr.seconds + SystemCall::getCurrentSystemTime();
    return true;
    }
  default:
    return Process::handleMessage(message);
  }
}

MAKE_PROCESS(Debug);

/*
 * Change log :
 * 
 * $Log: Debug.cpp,v $
 * Revision 1.31  2004/05/18 14:28:21  loetzsch
 * now Xabsl2 intermediate code can be sent to different modules
 *
 * Revision 1.30  2004/05/17 18:38:27  loetzsch
 * continued support for multiple Xabsl engines in different modules
 *
 * Revision 1.29  2004/04/30 08:18:28  goehring
 * DebugQueue is being deleted after being written to stick
 *
 * Revision 1.28  2004/04/21 14:08:16  dueffert
 * Motion can load parameterset itself on request now
 *
 * Revision 1.27  2004/04/07 13:00:47  risler
 * ddd checkin after go04 - second part
 *
 * Revision 1.2  2004/04/07 11:44:06  risler
 * added sending low res images
 * added Image::setCameraInfo
 *
 * Revision 1.1.1.1  2004/03/29 08:28:46  Administrator
 * initial transfer from tamara
 *
 * Revision 1.26  2004/03/22 21:58:13  roefer
 * True odometry
 *
 * Revision 1.25  2004/03/17 19:53:37  juengel
 * Added idKickSelectionTable.
 *
 * Revision 1.24  2004/03/17 16:18:49  thomas
 * added preversion of motion optimisation with behaviour, selflocator, headcontrol and robotcontrol dialog
 *
 * Revision 1.23  2004/03/16 10:24:54  dueffert
 * UDParameters for Motion and Cognition
 *
 * Revision 1.22  2004/03/11 14:27:31  uhrig
 * KalmanBallLocator was added
 *
 * Revision 1.21  2004/02/27 16:44:56  dueffert
 * UDEvolutionRequest introduced; hack removed
 *
 * Revision 1.20  2004/02/24 13:15:41  hamerla
 * *** empty log message ***
 *
 * Revision 1.19  2004/02/24 12:58:25  hamerla
 * no message
 *
 * Revision 1.18  2004/02/18 13:02:42  kindler
 * - requests.dat is now only read once, and is no longer held open. (-> can be updated through ftp now!)
 *
 * Revision 1.17  2004/02/16 19:22:47  jhoffman
 * added debug parameters for yet another inv kin
 *
 * Revision 1.16  2004/01/21 16:02:53  goehring
 * Debug queue size increased by 2 MB
 *
 * Revision 1.15  2004/01/21 15:15:16  loetzsch
 * increased the size of the outgoing queue in Debug
 *
 * Revision 1.14  2004/01/20 13:44:13  wachter
 * Changed sizes of MessageQueue and memory-allocation
 *
 * Revision 1.13  2004/01/19 14:59:47  wachter
 * Reduced MessageQueue size again.
 *
 * Revision 1.12  2004/01/12 17:17:52  kerdels
 * added freeze functionality to tacticDesigner
 *
 * Revision 1.11  2004/01/07 14:56:09  kerdels
 * added send functionality to TacticDesigner
 *
 * Revision 1.10  2004/01/06 17:26:11  kerdels
 * modified some DTT-Options
 *
 * Revision 1.9  2004/01/06 14:41:30  wachter
 * Reduced MessageQueue size because memory was full
 *
 * Revision 1.8  2004/01/03 21:26:46  wachter
 * Removed some debug-stuff
 *
 * Revision 1.7  2004/01/03 19:15:17  wachter
 * Debug-communication without router working now
 *
 * Revision 1.6  2003/12/16 19:02:45  loetzsch
 * The motion net file Config/spec_act.dat can be sent through WLAN to a robot.
 *
 * Revision 1.5  2003/12/02 18:07:43  dueffert
 * first working not yet calibrated version of UDWalkingEngine added
 *
 * Revision 1.4  2003/11/29 07:40:19  roefer
 * Doxygen comments corrected
 *
 * Revision 1.3  2003/11/22 15:59:03  jhoffman
 * parameter sending using generic debug data to
 * walking engine now works if it is explicitely given
 * a messageID
 *
 * Revision 1.2  2003/11/17 14:56:04  urban
 * added SensorBehaviorControl-solution "AlLx RemotePresence" and RobotControl-dialogbar "AlLx Joystick"
 *
 * Revision 1.1  2003/10/07 10:07:01  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.3  2003/09/11 15:20:59  dueffert
 * passing of InvKinParameters added
 *
 * Revision 1.2  2003/07/05 09:49:05  roefer
 * Generic debug message for bodyOffsets improved
 *
 * Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.46  2003/06/20 17:53:30  dueffert
 * silly fix
 *
 * Revision 1.45  2003/06/17 16:57:31  loetzsch
 * made queue smaller
 *
 * Revision 1.44  2003/05/27 12:23:42  risler
 * added GT2003BehaviorConfiguration debug message
 *
 * Revision 1.43  2003/05/08 14:55:28  juengel
 * printRobotStatus contains mac address now
 *
 * Revision 1.42  2003/05/06 16:49:07  juengel
 * increased size of debugIn
 * decreased size of debugOut
 * increased siye of theCognitionSender
 *
 * Revision 1.41  2003/05/05 12:27:02  juengel
 * Removed idGridImageProcessor2Thresholds
 *
 * Revision 1.40  2003/05/03 16:20:43  roefer
 * bodyOffsets generic debug data
 *
 * Revision 1.39  2003/04/16 14:14:38  loetzsch
 * removed Xabsl 1 from GT2003
 *
 * Revision 1.38  2003/04/16 07:00:16  roefer
 * Bremen GO checkin
 *
 * Revision 1.37  2003/03/28 09:17:48  roefer
 * Size of Cognition increased and size of Debug reduced
 *
 * Revision 1.36  2003/03/21 23:38:00  dueffert
 * status output works now
 *
 * Revision 1.35  2003/03/20 20:34:21  loetzsch
 * Game Toolbar now can adjust the game speed
 *
 * Revision 1.34  2003/03/19 09:23:48  dueffert
 * status output added
 *
 * Revision 1.33  2003/03/11 09:10:02  dueffert
 * QueueFillRequest rejectAll added
 *
 * Revision 1.32  2003/03/05 14:07:10  roefer
 * LinesPerceptor2 is now LinesImageProcessor
 *
 * Revision 1.31  2003/03/05 11:10:05  loetzsch
 * added class GlobalGameControlData
 *
 * Revision 1.30  2003/03/04 15:05:11  juengel
 * Added handling for idParametersForGridImageProcessor2
 *
 * Revision 1.29  2003/02/27 12:02:30  dueffert
 * JoystickData added
 *
 * Revision 1.28  2003/02/21 22:20:13  roefer
 * LinesSelfLocator is working
 *
 * Revision 1.27  2003/02/18 21:29:17  osterhues
 * Changed all instances of ColorTable64 to new base class ColorTable
 *
 * Revision 1.26  2003/02/08 18:36:44  juengel
 * added idGridImageProcessor2Thresholds
 *
 * Revision 1.25  2003/02/05 15:51:58  juengel
 * reduced queue size
 *
 * Revision 1.24  2003/01/30 11:28:54  juengel
 * Added idGridImageProcessor2Parameters and id LinesPerceptorParameters.
 *
 * Revision 1.23  2003/01/22 18:09:23  loetzsch
 * continued implementation of sending of intermediate code from xabsl2 dialog
 * to a xabsl2 behavior control
 *
 * Revision 1.22  2003/01/21 15:35:18  loetzsch
 * on the robot messages now are only sent if a message was received from the PC
 *
 * Revision 1.21  2003/01/16 09:54:49  jhoffman
 * Added "Generic Debug Data" data type. This can be used
 * for quick and dirty optimization and debugging, e.g. to send
 * parameters to a module through WLAN to adjust it's settings.
 * The DebugMessageGenerator is used to for parsing and
 * sendig the data
 *
 * Revision 1.20  2003/01/11 22:27:14  roefer
 * Limited 5 sec hack to real Aibo
 *
 * Revision 1.19  2003/01/11 13:29:55  juengel
 * Xabsl2Dialog sends Xabsl2DebugRequest now.
 *
 * Revision 1.18  2003/01/10 15:48:20  loetzsch
 * added a hack that does not send any messages to the WLan or OVirtualRobotComm
 * in the first 5 seconds after the first execution of main()
 *
 * Revision 1.17  2003/01/09 17:27:19  loetzsch
 * added the hack with the 50 seconds
 *
 * Revision 1.16  2002/12/08 07:11:14  roefer
 * Decompress JPEG images in Cognition
 *
 * Revision 1.15  2002/12/07 16:40:45  roefer
 * Blocking for theDebugReceiver changed
 *
 * Revision 1.14  2002/12/05 16:14:31  dueffert
 * started implementing realtime sending
 *
 * Revision 1.13  2002/11/28 14:07:55  dueffert
 * docu improved
 *
 * Revision 1.12  2002/11/27 13:50:27  dueffert
 * QueueFillMode beautified
 *
 * Revision 1.11  2002/11/26 12:23:58  dueffert
 * Debug handles growing requests.dat in Win32 now
 *
 * Revision 1.10  2002/11/25 12:27:15  dueffert
 * first steps towards reading growing files
 *
 * Revision 1.9  2002/11/23 22:42:13  roefer
 * Debug crash fixed
 *
 * Revision 1.8  2002/11/21 16:08:39  dueffert
 * doxygen comments corrected
 *
 * Revision 1.7  2002/11/19 17:11:26  risler
 * added datatype PIDData
 * support for sending new pid values at runtime
 *
 * Revision 1.6  2002/11/19 12:57:54  dueffert
 * queuefillrequest added, debugkey toolbar extended, memorystick toolbar removed
 *
 * Revision 1.5  2002/11/18 10:35:05  juengel
 * Added "block debug sender" mechanism
 *
 * Revision 1.4  2002/10/02 15:52:24  loetzsch
 * increased the size of the queue in the debug process to 11000000
 *
 * Revision 1.3  2002/09/18 16:35:12  loetzsch
 * made GT2003 compilable,
 * rechanged message ids
 *
 * Revision 1.2  2002/09/17 23:55:22  loetzsch
 * - unraveled several datatypes
 * - changed the WATCH macro
 * - completed the process restructuring
 *
 * Revision 1.1  2002/09/10 15:41:25  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 * - Removed all process layouts
 * - Added process layout CMD
 *
 * Revision 1.8  2002/09/07 13:36:58  loetzsch
 * unified the vision modules into one module "ImageProcessor"
 * - FloodFillRLEImageProcessor, BallPerceptor, LandmarksPerceptor
 *   and PlayersPerceptor were are combined to the new solution
 *   "BlobImageProcessor"
 * - The GridImageProcessor and the SubPixelGradientCalculator became
 *   a solution of "ImageProcessor"
 *
 * Revision 1.7  2002/08/21 11:50:44  cesarz
 * GT2001PlayersPerceptor, BerlinSelfLocator, GT2001WalkingEngine,
 * DoWalkingEngine, FuzzyBehaviorControl, BremenGOBehaviorControl,
 * FuzzyBehaviortesterdlgbar and Behaviortesterdlgbar removed.
 *
 * Revision 1.6  2002/07/23 13:40:51  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - new design of debugging architecture
 * - exchanged StaticQueue with MessageQueue
 * - new debug message handling
 * - general clean up
 *
 * Revision 1.5  2002/06/21 16:45:56  dueffert
 * smaller debug queue
 *
 * Revision 1.4  2002/06/08 20:41:15  mkunz
 * specialPercept queue completed
 *
 * Revision 1.3  2002/06/02 23:21:09  roefer
 * Single color table and progress in LinesSelfLocator
 *
 * Revision 1.2  2002/05/15 15:47:29  risler
 * added debug message AcousticMessage, DebugMessageGenerator generates AcousticMessage
 *
 * Revision 1.1.1.1  2002/05/10 12:40:19  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.40  2002/05/05 22:12:42  loetzsch
 * GameControlData can now be sent from the Game toolbar to BehaviorControl
 *
 * Revision 1.39  2002/04/29 10:18:23  cesarz
 * changed idInvKinWalkingParameters
 *
 * Revision 1.38  2002/04/23 15:11:41  jhoffman
 * added some pre GO fourier stuff
 *
 * Revision 1.37  2002/04/09 16:46:01  risler
 * added DarmstadtGOWalkingEngine
 *
 * Revision 1.36  2002/04/09 11:19:06  loetzsch
 * debug message handling for idPlayerConfig
 *
 * Revision 1.35  2002/04/08 18:40:45  dueffert
 * XabslInputData added
 *
 * Revision 1.34  2002/04/06 09:55:53  roefer
 * Image and SensorData path through DebugQueues changed
 *
 * Revision 1.33  2002/04/02 15:03:40  loetzsch
 * distribution of XabslDebugRequests
 *
 * Revision 1.32  2002/03/28 17:43:56  loetzsch
 * mcf tuned
 *
 * Revision 1.31  2002/03/27 12:58:12  piepenstock
 * Added serial communication support
 *
 * Revision 1.30  2002/03/12 14:34:15  jhoffman
 * added fourier coefficient output
 *
 * Revision 1.29  2002/02/28 16:28:25  risler
 * added GT2001WalkingParameters
 *
 * Revision 1.28  2002/02/27 19:53:20  loetzsch
 * increased size of theCognitionReceiver
 *
 * Revision 1.27  2002/02/21 14:22:47  loetzsch
 * added several STOP_WATCH_ON_REQUEST macros
 *
 * Revision 1.26  2002/02/12 16:34:49  risler
 * finished MofTester
 *
 * Revision 1.25  2002/02/06 01:15:40  loetzsch
 * oracled world states are now first send through the local processes
 * before they are painted. (for synchronization with the painting of the
 * calculated world states)
 *
 * Revision 1.24  2002/02/05 20:02:16  risler
 * handleDebugMessage now returns bool, added debug message handling to ImageProcessor
 *
 * Revision 1.23  2002/02/05 04:19:04  loetzsch
 * replaced several team color hacks by getPlayer().getTeamColor()
 *
 * added a few new module selectors
 *
 * changed distribution of debug messages completely
 *
 * Revision 1.22  2002/01/30 17:29:56  loetzsch
 * handleDebugMessage um Parameter timestamp erweitert
 *
 * Revision 1.21  2002/01/23 13:51:55  loetzsch
 * from Debug Queues odometry data is now sent to Motion and not to Perception
 *
 * Revision 1.20  2002/01/23 05:46:42  loetzsch
 * mcf tuned
 *
 * Revision 1.19  2002/01/23 02:47:39  loetzsch
 * mcf tuned
 *
 * Revision 1.18  2002/01/22 14:56:30  juengel
 * ColorTable verschicken angefangen.
 *
 * Revision 1.17  2002/01/22 00:10:08  loetzsch
 * new configuration DebugWLan introduced
 *
 * Revision 1.16  2002/01/19 12:43:16  risler
 * enabled SolutionRequest, changed HandleDebugMessage calls
 *
 * Revision 1.15  2002/01/18 23:30:28  loetzsch
 * Distribution of HeadControlModes, HeadMotionRequests and SolutionRequests added
 *
 * Revision 1.14  2002/01/18 02:17:39  loetzsch
 * MotionTester dialog and distribution of MotionRequests
 * to the processes programmed.
 *
 * Revision 1.13  2002/01/13 17:02:02  roefer
 * SimRobot now has a console window
 *
 * Revision 1.12  2001/12/30 14:48:37  roefer
 * Portable macro PRINT() for direct console output inserted
 *
 * Revision 1.11  2001/12/28 22:40:34  roefer
 * Inserted check for existence of configuration file
 *
 * Revision 1.10  2001/12/20 17:14:35  loetzsch
 * Using debug requests from the memory stick.
 *
 * Revision 1.9  2001/12/19 18:37:32  bach
 * Camera.cfg added
 * Sensors: CameraParameters are set
 *
 * Revision 1.8  2001/12/17 16:42:34  loetzsch
 * Removed the StaticQueue >> StaticQueue operator,
 * replaced by copyAllMessages and moveAllMessages
 *
 * Revision 1.7  2001/12/15 20:32:09  roefer
 * Senders and receivers are now part of the processes
 *
 * Revision 1.6  2001/12/13 12:57:18  loetzsch
 * Aufruf des SensorDataProcesssors entfernt, da dieser abstrzt.
 *
 * Revision 1.5  2001/12/12 18:43:22  loetzsch
 * division by 0 auskommentiert
 *
 * Revision 1.4  2001/12/12 18:08:56  loetzsch
 * Streaming- Operatoren fr Bilder eingebaut, DebugKeyTable nicht- statisch gemacht, Debuggin Mechanismen weitergemacht, Bilder aus Logfiles in RobotControl anzeigen, Logfiles in HU1/Debug auf den Stick schreiben
 *
 * Revision 1.3  2001/12/10 17:47:08  risler
 * change log added
 *
 */
