/** 
* @file Motion.cpp
* Implementation of class Motion
*/
#include "Motion.h"
 
#include "Platform/Sensors.h"
#include "Tools/RobotConfiguration.h"
#include "Tools/Debugging/Stopwatch.h"

/*
* that's very fine - with this blocking sender
* the robot can call for a new set of data and
* decide itself when motion should be started
*/
Motion::Motion () :
INIT_DEBUGGING,
INIT_RECEIVER_SENSORDATA(false),
INIT_RECEIVER(PackageCognitionMotion,false),

INIT_SENDER_MOTORCOMMANDS(true),
INIT_SENDER_SOUNDDATA(false),
INIT_SENDER(PackageMotionCognition,false),
INIT_SENDER(OdometryData,false),

gameSpeed(1.0),
headIsBlockedBySpecialActionOrWalk(false)
{
  frameNumber = 0;
  debugIn.setSize(200000);
  debugOut.setSize(2000);
  
  SensorDataProcessorInterfaces sensorDataProcessorInterfaces(    
    thePackageMotionCognitionSender.headState,
    theSensorDataBufferReceiver,theSensorDataBufferReceiver.frame[0].frameNumber,
    bodyPercept, cameraMatrix, psdPercept);
  pSensorDataProcessor = new SensorDataProcessorSelector(moduleHandler, sensorDataProcessorInterfaces);
  // PRINT("Motion:: SensorDataProcessor created");
  

  HeadControlInterfaces headControlInterfaces(
    thePackageCognitionMotionReceiver.headControlMode,
    thePackageCognitionMotionReceiver.robotPose,
    cameraMatrix,
    theOdometryDataSender,
    thePackageCognitionMotionReceiver.ballModel,
    theSensorDataBufferReceiver, 
    thePackageCognitionMotionReceiver.motionRequest,
    executedMotionRequest,
    thePackageCognitionMotionReceiver.landmarksState,
    thePackageCognitionMotionReceiver.specialPercept,
    headIsBlockedBySpecialActionOrWalk,
    headMotionRequest,
    thePackageMotionCognitionSender.headState, 
    theMotorCommandsSender.pidData,
    robotVerticesBuffer
  );
  pHeadControl = new HeadControlSelector(moduleHandler, headControlInterfaces);
  // PRINT("Motion:: HeadControl created");
  
  LEDControlInterfaces ledControlInterfaces(
    thePackageCognitionMotionReceiver.ledRequest,
    thePackageCognitionMotionReceiver.wLanStatus, 
    theMotorCommandsSender.ledValue);
  pLEDControl = new LEDControlSelector(moduleHandler, ledControlInterfaces);
  // PRINT("Motion:: LEDControl created");
  
  MotionControlInterfaces motionControlInterfaces(
    motionRequest, headMotionRequest, theSensorDataBufferReceiver,
    thePackageCognitionMotionReceiver.invKinWalkingParameters,
    thePackageCognitionMotionReceiver.udParameters,
    thePackageCognitionMotionReceiver.walkParameterTimeStamp,
    robotVerticesBuffer,
    theMotorCommandsSender.jointDataBuffer,
    theMotorCommandsSender.pidData,
    theOdometryDataSender,
    thePackageMotionCognitionSender.headState, executedMotionRequest,
    headIsBlockedBySpecialActionOrWalk);
  pMotionControl = new MotionControlSelector(moduleHandler, motionControlInterfaces);
  // PRINT("Motion:: MotionControl created");
  
  SoundControlInterfaces soundControlInterfaces(
    thePackageCognitionMotionReceiver.soundRequest, theSoundDataSender);
  pSoundControl = new SoundControlSelector(moduleHandler, soundControlInterfaces);
  // PRINT("Motion:: SoundControl created");
  
}

Motion::~Motion()
{
  delete pSensorDataProcessor;
  delete pHeadControl;
  delete pLEDControl;
  delete pMotionControl;
  delete pSoundControl;
}

int Motion::main() 
{
  if(theSensorDataBufferReceiver.receivedNew())
  {
    frameNumber = theSensorDataBufferReceiver.frame[0].frameNumber;
    //OUTPUT(idText, text, "SDR: " << frameNumber); 
  }
  else
  {
    frameNumber++;
  }
  INFO(sendJointData,idJointData,bin,theMotorCommandsSender.jointDataBuffer.frame[0]);
  
#ifndef _WIN32
  if (SystemCall::getTimeSince(thePackageCognitionMotionReceiver.timeStamp) > 1000UL)
  {
    thePackageCognitionMotionReceiver.motionRequest.motionType = MotionRequest::specialAction;
    thePackageCognitionMotionReceiver.motionRequest.specialActionType = MotionRequest::swing;
  }
#endif

  // without STOP_TIME
  pSensorDataProcessor->execute();

#ifndef _WIN32
  if (motionRequest.updateRP)
  {
    if((lastRobotPoseUpdateFrame != thePackageCognitionMotionReceiver.robotPose.frameNumber) &&
    (frameNumber != thePackageCognitionMotionReceiver.robotPose.frameNumber))
    {
      int frameDifference = (int)(frameNumber - 
        thePackageCognitionMotionReceiver.robotPose.frameNumber);

       // OUTPUT(idText, text, "+ new rp! " << frameDifference << ", " << thePackageCognitionMotionReceiver.robotPose.frameNumber);

      thePackageCognitionMotionReceiver.robotPose += 
        (theOdometryDataSender - odometryHistory.getEntry(frameDifference + 1));
    } 
    else
    {
      thePackageCognitionMotionReceiver.robotPose += (theOdometryDataSender - lastOdometry);  
    }  
  lastOdometry = theOdometryDataSender;
  odometryHistory.add(theOdometryDataSender);
  lastRobotPoseUpdateFrame = thePackageCognitionMotionReceiver.robotPose.frameNumber;
  }

  // OUTPUT(idText, text, "[" << frameNumber << "] od: " << theOdometryDataSender.rotation << " rp: " << thePackageCognitionMotionReceiver.robotPose.rotation);
#endif

  STOP_TIME_ON_REQUEST(headControl,pHeadControl->execute(););
  getPlayer().setTeamColor(thePackageCognitionMotionReceiver.teamColor);
  pLEDControl->execute();
  
  // slow down the motion request if requested
  motionRequest = thePackageCognitionMotionReceiver.motionRequest;
  motionRequest.walkParams.translation.x *= gameSpeed;
  motionRequest.walkParams.translation.y *= gameSpeed;
  motionRequest.walkParams.rotation *= gameSpeed;
  
  STOP_TIME_ON_REQUEST(motionControl,pMotionControl->execute(););
  thePackageMotionCognitionSender.executedMotionRequest.setFrameNumber(frameNumber);
  
  theMotorCommandsSender.send();
  
  INFO(sendExecutedMotionRequest, idExecutedMotionRequest, bin, thePackageMotionCognitionSender.executedMotionRequest);
  
  if(theSoundDataSender.requestedNew())
  {
    STOP_TIME_ON_REQUEST(soundControl,pSoundControl->execute(););
    if (theSoundDataSender.isInUse) theSoundDataSender.send();            
  }
  
  // speed up the executed motion request again
  thePackageMotionCognitionSender.executedMotionRequest = executedMotionRequest;
  thePackageMotionCognitionSender.executedMotionRequest.walkParams.translation.x /= gameSpeed;
  thePackageMotionCognitionSender.executedMotionRequest.walkParams.translation.x /= gameSpeed;
  thePackageMotionCognitionSender.executedMotionRequest.walkParams.rotation /= gameSpeed;
  
  theOdometryDataSender.send();
  thePackageMotionCognitionSender.send();
  theDebugSender.send();
  
  return 0;
}


bool Motion::handleMessage(InMessage& message)
{
  switch (message.getMessageID())
  {
  case idOdometryData:
    message.bin >> theOdometryDataSender;
    return true;
  case idHeadMotionRequest:
    message.bin >> headMotionRequest;
    return true;
  case idPIDData:
    message.bin >> theMotorCommandsSender.pidData;
    return true;
  case idGameSpeed:
    message.bin >> gameSpeed;
    return true;
  case idBodyOffsets:
    getRobotConfiguration().handleMessage(message);
    return true;
  case idXabsl2DebugRequest:
    return pHeadControl->handleMessage(message);
  case idOdometryScale:
    pMotionControl->handleMessage(message);
    return true;
  default:
    return Process::handleMessage(message);
  }
}

MAKE_PROCESS(Motion);

/*
* Change log :
* 
* $Log: Motion.cpp,v $
* Revision 1.25  2004/06/03 16:10:53  kerdels
* added new headcontrolmode
*
* Revision 1.24  2004/05/24 19:40:06  goehring
* ballPosition to ballModel renamed
*
* Revision 1.23  2004/05/18 11:38:16  loetzsch
* the Motion process now also executes a SensorDataProcessor for the calculation of the cameraMatrix
*
* Revision 1.22  2004/05/17 19:21:51  loetzsch
* renamed all Variables "cameraMatrix" to "cameraMatrix2"
*
* Revision 1.21  2004/05/17 18:38:27  loetzsch
* continued support for multiple Xabsl engines in different modules
*
* Revision 1.20  2004/05/14 14:12:08  wachter
* - Added communication support for 5 robots
* - rewrote parts of team-communication to be faster and more stable
*
* Revision 1.19  2004/05/10 10:30:15  juengel
* Added executedMotionRequest to HeadControlInterfaces.
*
* Revision 1.18  2004/04/11 18:49:45  roefer
* Team color is sent again from Cognition to Motion.
* Otherwise, messages sent by Motion may contain the wrong team color.
*
* Revision 1.17  2004/04/07 14:42:56  risler
* moved LandsmarksState to Cognition directory, generated by SelfLocator
*
* Revision 1.16  2004/04/07 13:00:47  risler
* ddd checkin after go04 - second part
*
* Revision 1.2  2004/04/01 19:52:44  Charlie
* added LandmarkState
*
* Revision 1.1.1.1  2004/03/29 08:28:46  Administrator
* initial transfer from tamara
*
* Revision 1.15  2004/03/28 14:06:41  jhoffman
* renamed headcontrolmode ATH2004 in ATH2004ERS7
*
* Revision 1.15  2004/03/28 14:06:41  jhoffman
* renamed headcontrolmode ATH2004 in ATH2004ERS7
*
* Revision 1.14  2004/03/27 14:45:08  loetzsch
* removed team color from PackageCognitionMotion
*
* Revision 1.13  2004/03/22 21:58:14  roefer
* True odometry
*
* Revision 1.12  2004/03/20 17:20:01  juengel
* Added selector for LEDControl
*
* Revision 1.11  2004/03/20 09:55:26  roefer
* Preparation for improved odometry
*
* Revision 1.10  2004/03/16 14:00:21  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.9  2004/03/10 12:17:50  roefer
* Player bug fixed
*
* Revision 1.8  2004/03/08 15:29:11  tim
* Sending the Player object to the Motion process
*
* Revision 1.2  2004/03/15 17:11:40  hoffmann
* - added ATH2004HeadControl
* - added ATH2004LEDControl
* - headmotiontester shows "tilt2"
* - motion process updates odometry while no new robotPose is received, added to motion request
* - some ui adjustments
* - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
*
* Revision 1.7  2004/03/04 16:00:21  fritsche
* added pidData to HeadControlInterfaces
*
* Revision 1.6  2004/03/04 10:05:26  jhoffman
* - motion process now uses odometry to propagate the robot pose while no new robot pose is being sent (this makes headcontrol a little better)
* - removed headcontrol member variable "propagatedPose" from headcontrol and cognition->motion-sender
*
* Revision 1.5  2004/02/16 17:57:43  dueffert
* packageCognitionMotion extended with invkin parameters
*
* Revision 1.4  2003/12/16 19:02:45  loetzsch
* The motion net file Config/spec_act.dat can be sent through WLAN to a robot.
*
* Revision 1.3  2003/11/24 15:31:20  dueffert
* SpecialPercept removed from PackageCognition
*
* Revision 1.2  2003/11/14 19:02:26  goehring
* frameNumber added
*
* Revision 1.1  2003/10/07 10:07:01  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.7  2003/09/26 11:41:17  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.6  2003/09/05 16:47:12  juengel
* idOdometryData is handled again.
*
* Revision 1.5  2003/09/05 14:38:44  juengel
* idOdometryData is not handled any longer.
*
* Revision 1.4  2003/07/05 09:49:05  roefer
* Generic debug message for bodyOffsets improved
*
* Revision 1.3  2003/07/03 22:50:41  dueffert
* no spastic startup movements anymore: sending after calculating
*
* Revision 1.2  2003/07/02 10:48:17  loetzsch
* removed a PRINT() statement
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.27  2003/06/15 22:42:22  loetzsch
* no message
*
* Revision 1.26  2003/05/27 16:16:23  juengel
* Added headIsBlockedBySpecialActionOrWalk.
*
* Revision 1.25  2003/05/13 11:42:05  goehring
* CollisionDetector removed
*
* Revision 1.24  2003/05/03 16:20:43  roefer
* bodyOffsets generic debug data
*
* Revision 1.23  2003/05/02 18:26:17  risler
* SensorDataBuffer added
* replaced SensorData with SensorDataBuffer
* full SensorData resolution now accessible
*
* Revision 1.22  2003/04/26 11:36:21  juengel
* Added delete pCollisionDetector.
*
* Revision 1.21  2003/04/25 19:49:44  goehring
* Added new module CollisionDetector
*
* Revision 1.20  2003/04/16 07:00:16  roefer
* Bremen GO checkin
*
* Revision 1.21  2003/04/08 18:28:28  roefer
* bodyTiltOffset added
*
* Revision 1.20  2003/04/08 16:26:36  timrie
* added current MotionRequest
*
* Revision 1.19  2003/04/07 16:51:31  jhoffman
* send odometry to headcontrol
*
* Revision 1.18  2003/03/20 20:34:21  loetzsch
* Game Toolbar now can adjust the game speed
*
* Revision 1.17  2003/03/07 19:04:59  juengel
* Added INFO for the executedMotionRequest
*
* Revision 1.16  2002/12/07 16:40:45  roefer
* Blocking for theDebugReceiver changed
*
* Revision 1.15  2002/11/28 23:06:12  hebbel
* sending wavedata only if necessary
*
* Revision 1.14  2002/11/28 14:49:50  jhoffman
* no message
*
* Revision 1.13  2002/11/19 17:12:47  risler
* added datatype PIDData
* support for sending new pid values at runtime
*
* Revision 1.12  2002/10/10 13:09:50  loetzsch
* First experiments with the PSD Sensor
* - SensorDataProcessor now calculates PSDPercept
* - Added the PerceptBehaviorControl solution PSDTest
* - Added the RadarViewer3D to RobotControl, which can display the Points3D structure
*
* Revision 1.11  2002/10/10 11:48:43  juengel
* WATCH(sendJointData ...  and  WATCH(sendSensorData ...
* replaced by
* INFO(sendJointData ...  and  INFO(sendSensorData ...
*
* Revision 1.10  2002/09/19 12:15:13  loetzsch
* made compilable for the gcc compiler.
*
* Revision 1.9  2002/09/18 19:52:36  loetzsch
* the head state is now sent from Motion to Cognition using the package.
*
* Revision 1.8  2002/09/17 23:55:22  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.7  2002/09/16 17:34:23  dueffert
* anonymous contructors returns &CLASS with VS, but CLASS with gcc.
*
* Revision 1.6  2002/09/12 14:20:05  juengel
* Created a package for all data sent from Cognition to Motion.
*
* Revision 1.5  2002/09/11 20:06:19  loetzsch
* continued experiments with modules/solutions
*
* Revision 1.4  2002/09/11 17:26:31  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.3  2002/09/11 13:43:19  loetzsch
* Created a package for all data that are sent from Motion to Cognition.
* Removed all previous Senders
*
* Revision 1.2  2002/09/11 00:06:59  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:41:25  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
* - Removed all process layouts
* - Added process layout CMD
*
* Revision 1.10  2002/07/23 13:40:52  loetzsch
* - new streaming classes
* - removed many #include statements
* - new design of debugging architecture
* - exchanged StaticQueue with MessageQueue
* - new debug message handling
* - general clean up
*
* Revision 1.9  2002/06/28 10:30:51  roefer
* OUTPUT is possible in constructors
*
* Revision 1.8  2002/05/29 15:54:29  cesarz
* Removed deadlock in acoustic communication
*
* Revision 1.7  2002/05/27 15:39:13  fischer
* Added SoundState (Sender and Receiver)
*
* Revision 1.6  2002/05/25 22:52:19  roefer
* WLan, first working approach
*
* Revision 1.5  2002/05/23 12:15:19  hebbel
* sends acoustic messages only if there is no other dog sending
*
* Revision 1.4  2002/05/16 22:36:11  roefer
* Team communication and GTMath bugs fixed
*
* Revision 1.3  2002/05/15 07:28:55  hebbel
* Uses Soundprotocol
*
* Revision 1.2  2002/05/14 18:54:30  hebbel
* Sending audiodata only if it is not mute
*
* Revision 1.1.1.1  2002/05/10 12:40:19  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.33  2002/05/06 16:03:48  hebbel
* Added buffered Sender for SoundDataOut
*
* Revision 1.32  2002/05/04 17:47:44  loetzsch
* LEDControl now has access to an instance of WLanStatus
* (which shall be displayed then with the green leds. follows next)
*
* Revision 1.31  2002/05/03 17:15:57  giese
* StopTimeOnRequest for SoundOutControl added
*
* Revision 1.30  2002/04/29 17:17:28  hebbel
* Put SoundPlay to Motion Process
*
* Revision 1.29  2002/04/23 10:38:30  risler
* renamed headOdometry to headState
*
* Revision 1.28  2002/04/02 13:10:21  dueffert
* big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
*
* Revision 1.27  2002/02/21 14:22:47  loetzsch
* added several STOP_WATCH_ON_REQUEST macros
*
* Revision 1.26  2002/02/08 17:48:57  risler
* SensorData to MotionControl
*
* Revision 1.25  2002/02/05 20:02:16  risler
* handleDebugMessage now returns bool, added debug message handling to ImageProcessor
*
* Revision 1.24  2002/02/05 04:19:04  loetzsch
* replaced several team color hacks by getPlayer().getTeamColor()
*
* added a few new module selectors
*
* changed distribution of debug messages completely
*
* Revision 1.23  2002/01/30 17:29:56  loetzsch
* handleDebugMessage um Parameter timestamp erweitert
*
* Revision 1.22  2002/01/28 14:01:07  loetzsch
* INFO macros for JointData and SensorData added
*
* Revision 1.21  2002/01/26 20:26:27  juengel
* Nichts gendert
*
* Revision 1.20  2002/01/26 18:10:16  juengel
* DebugDrawingManager umstrukturiert.
*
* Revision 1.19  2002/01/23 13:51:55  loetzsch
* from Debug Queues odometry data is now sent to Motion and not to Perception
*
* Revision 1.18  2002/01/19 21:36:26  risler
* added HeadMotionTester, HeadControlSelector
*
* Revision 1.17  2002/01/19 12:43:16  risler
* enabled SolutionRequest, changed HandleDebugMessage calls
*
* Revision 1.16  2002/01/18 23:30:45  loetzsch
* Distribution of HeadControlModes, HeadMotionRequests and SolutionRequests added
*
* Revision 1.15  2002/01/18 02:17:39  loetzsch
* MotionTester dialog and distribution of MotionRequests
* to the processes programmed.
*
* Revision 1.14  2001/12/21 14:09:39  roefer
* Added several destructors
*
* Revision 1.13  2001/12/19 16:03:55  bach
* SystemDataTypes replaced by Sensors
*
* Revision 1.12  2001/12/15 20:32:08  roefer
* Senders and receivers are now part of the processes
*
* Revision 1.11  2001/12/14 12:34:09  dueffert
* no message
*
* Revision 1.10  2001/12/12 18:08:56  loetzsch
* Streaming- Operatoren fr Bilder eingebaut, DebugKeyTable nicht- statisch gemacht, Debuggin Mechanismen weitergemacht, Bilder aus Logfiles in RobotControl anzeigen, Logfiles in HU1/Debug auf den Stick schreiben
*
* Revision 1.9  2001/12/11 18:14:45  kosen
* SensorData added.
*
* problems with Receiver (multiple definition error) solved.
*
* Revision 1.8  2001/12/11 16:15:10  kosen
* reihenfolge angepasst :)
*
* Revision 1.7  2001/12/11 16:06:49  kosen
* aufruf von HeadControl::execute() angepasst.
*
* Revision 1.6  2001/12/10 17:47:08  risler
* change log added
*
*/
