/**
* @file BallModel.h
*
* Declaration of class BallPosition
*
* @author Martin Ltzsch
* @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
*/

#ifndef __BallModel_h_
#define __BallModel_h_

#include "Tools/Streams/InOut.h"
#include "Tools/Math/Pose2D.h"
#include "Tools/Math/Geometry.h"

/**
* @class SeenBallPosition
*
* The knowledge about the ball that was obtained by own observations.
*/
class SeenBallPosition : public Vector2<double>
{
public:
  /** Constructor */
  SeenBallPosition();
  
  /** The speed of the ball estimated from own observations. */
  Vector2<double> speed;
  
  /**
  * The time when the ball was last seen by the own camera.
  * "position" and speed were set at that point in time.
  */
  unsigned long timeWhenLastSeen;
  
  /**
  * The time when the ball was last seen consecutively, i.e.
  * it has been seen "a lot" but not necessarily in all frames;
  * gaps of n frames or more are okay
  */
  unsigned long timeWhenFirstSeenConsecutively;
  
  /**
  * The time until the ball has been seen consecutively, i.e.
  * it has been seen "a lot" but not necessarily in all frames
  * until the time stored here;
  */
  unsigned long timeUntilSeenConsecutively;
  
  bool ballInFrontOfOpponentGoal;

  /**
  * If this is positive, we see the ball for that time consecutively.
  * If this is negative, we have not seen the ball consecutively for -that time.
  */
  long getConsecutivelySeenTime() const;

  double distanceError;

  double angleError;
  
  /** Assignment operator for Vector2<double>
	 * @param other The Vector2<double> to be assigned to (x,y)
   * @return A reference to this object after the assignment.
	 */
  SeenBallPosition& operator=(const Vector2<double>& other){
    x=other.x;
    y=other.y;
    return *this;
  }
  
};

/**
* The position of the ball estimated from own observations and the observed
* ball speed; it is propagated even if the ball is not seen.
*/
class PropagatedBallPosition : public Vector2<double>
{
public:
  /** Constructor */
  PropagatedBallPosition();

  /** The speed of the ball estimated from propagations and last speed. */
  Vector2<double> speed;

  Vector2<double> getPropagatedPosition(long time) const;
  Vector2<double> getPropagatedSpeed(long time) const;
  
  long timeOfObservation;
  Vector2<double> positionWhenLastObserved;
  Vector2<double>	observedSpeed;
  
private:
};

/** The position of the ball estimated from own observations and from
* observation of other players. */
class CommunicatedBallPosition : public Vector2<double>
{
public:
  /** Constructor */
  CommunicatedBallPosition();
  
  /** The time when an observation made by an other robot. */
  unsigned long timeWhenLastObserved;
  
  /** used for the Ruhrpott-Hellhounds-collective-scoccer behavior to access
  the gaussian distribution of the GaussBellTeamBallLocator
  May be removed in the futute. */
  double distanceInMajDirection;
  double distanceInMinDirection;
  double orientationAngleOfGaussBell;
  
};

/**
* @class BallModel
*
* Contains the modeled knowledge about the ball.
*/
class BallModel
{
public:
  /** Constructor. */
  BallModel();
  
  unsigned long frameNumber;
  void setFrameNumber(unsigned long frameNumber) {this->frameNumber = frameNumber;}
  
  /** The position of the ball estimated from own observations */
  SeenBallPosition seen;
  
  /**
  * The position of the ball estimated from own observations and the observed
  * ball speed; it is propagated even if the ball is not seen
  */
  PropagatedBallPosition propagated;
  
  /**
  * The position of the ball estimated from own observations and from
  * observation of other players.
  */
  CommunicatedBallPosition communicated;
  
  /**
  * Short-Range-Ball-Movement Information
  */
  enum BallState{none, ballRollsByLeft, ballRollsByRight, ballRollsFast, waitForFeedback, ballRollsMiddle} ballState;
  
  /**
  * Returns a "known" ball position. If the ball was seen, the seen ball position
  * is used. If the ball was not seen for a timeAfterThatCommunicatedBallsAreAccepted,
  * the communicatedBallPosition is returned, if there was received a ball position.
  * @param timeAfterWhichCommunicatedBallsAreAccepted The time after that the communicated position is returned
  */
  const Vector2<double>& getKnownPosition(unsigned long timeAfterWhichCommunicatedBallsAreAccepted) const;
  
  /**
  * Returns the time since the ball was seen or communicated.
  * @param timeAfterWhichCommunicatedBallsAreAccepted as in getKnownPosition()
  */
  unsigned long getTimeSinceLastKnown(unsigned long timeAfterWhichCommunicatedBallsAreAccepted) const;
  
  /** The time for the behaviorControl after that communicated ball positions are accepted */
  enum {behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted=6000};
  
  // ????
  double validity;
  double motionValidity;
  
};

/**
* Streaming operator that reads a BallModel from a stream.
* @param stream The stream from which is read.
* @param ballPosition The BallModel object.
* @return The stream.
*/
In& operator>>(In& stream,BallModel& ballPosition);

/**
* Streaming operator that writes a BallModel to a stream.
* @param stream The stream to write on.
* @param ballPosition The BallModel object.
* @return The stream.
*/
Out& operator<<(Out& stream, const BallModel& ballPosition);


#endif //__BallModel_h_

/*
* Change log :
*
* $Log: BallModel.h,v $
* Revision 1.11  2004/05/18 21:14:33  goehring
* vector2 Propagated speed added
*
* Revision 1.10  2004/05/04 10:48:59  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.9  2004/05/03 15:32:25  uhrig
* Added additional ball state and disabled sending of Kalman filter process model states.
*
* Revision 1.8  2004/05/02 13:33:21  juengel
* Added GT2004BehaviorControl.
* (gt2004BehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted)
*
* Revision 1.7  2004/04/05 17:56:49  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.3  2004/04/02 10:01:58  jumped
* added ballIFrontOfOpponentGoal member
*
*
* Revision 1.6  2004/03/28 17:54:11  juengel
* Added ballState "ballRollsFast".
*
* Revision 1.5  2004/03/25 13:33:50  schmitt
* Added distance- and angleError again
*
* Revision 1.4  2004/03/21 12:44:48  juengel
* Added "wait-for-feedback" to ballState
*
* Revision 1.3  2004/02/29 13:37:15  dueffert
* doxygen bugs fixed and beautified
*
* Revision 1.2  2004/02/05 14:21:24  spranger
* added type definition fpr ball-state enum
*
* Revision 1.1  2004/02/03 12:56:41  spranger
* renamed BallPosition class to BallModel (including the files)
*
* Revision 1.9  2004/01/22 20:08:00  schumann
* added validity for movement
* added elemination of jumping ball
* improved calculation of ball movement
*
* Revision 1.8  2004/01/19 14:55:23  dueffert
* all frameNumbers (and not only some of them) are unsigned long now
*
* Revision 1.7  2004/01/13 18:07:41  spranger
* changed name from ballDirection to ballState
*
* Revision 1.6  2004/01/13 17:55:01  spranger
* added ballDirectionn(shortRange)
*
* Revision 1.5  2003/11/14 19:02:25  goehring
* frameNumber added
*
* Revision 1.4  2003/10/26 22:49:41  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
* Revision 1.3  2003/10/22 13:06:21  goehring
* = Operator for Vector<double> added
*
* Revision 1.2  2003/10/21 13:11:15  goehring
* review
*
* Revision 1.1  2003/10/07 10:07:01  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.4  2003/09/25 11:23:52  juengel
* Removed BlobCollection.
*
* Revision 1.3  2003/07/08 03:27:53  wachter
* - added challengeBallBetweenFeetd
* - removed propagetedBallDistance imput symbol
* - fixed cvs ;)
*
* Revision 1.2  2003/07/04 16:40:34  loetzsch
* gt2003BehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
*
* Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
* created new repository for the competitions in Padova from the
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.18  2003/06/17 20:01:25  dueffert
* getConsecutivelySeenTime() added
*
* Revision 1.17  2003/06/15 16:43:56  jhoffman
* propagated position is calculated by a function rather then in iteratively in cognition
*
* Revision 1.16  2003/06/15 14:26:44  jhoffman
* + moved "relative2FieldCoord" to Geometry
* + added member function to ballposition to calculate the propagated position and speed for a given time
* + propagated speed and time calculation using exponential decay instead of using an iterative calculation
* + in motion you can now use propageted ball pos at 125 Hz rather then the framerate determined by cognition
*
* Revision 1.15  2003/05/19 15:35:20  wachter
* Added some comments
*
* Revision 1.14  2003/05/09 15:28:51  loetzsch
* added time-after-which-communicated-ball-are-accepted
*
* Revision 1.13  2003/05/02 14:06:36  wachter
* Moved distanceInMayDirection and distanceInMinDirection from BallPosition to CommunicatedBallPosition
*
* Revision 1.12  2003/05/01 19:32:04  roefer
* Ball position assignment issues solved
*
* Revision 1.11  2003/05/01 18:10:03  loetzsch
* renamed CommunicatedBallPosition::timeWhenLastReceived to CommunicatedBallPosition::timeWhenLastObserved
*
* Revision 1.10  2003/05/01 17:09:06  loetzsch
* Redesign of ball modeling:
* - Modularized class BallPosition
* - splitted up module "BallLocator" into "BallLocator" for modeling of percepts
*   and "TeamBallLocator" for modelling communicated positions
* - Removed solution JumpingBallLocator
* - Splitted Solution DefaultBallLocator into DefaultBallLocator and DefaultTeamBallLocator
* - Renamed SensorFusionBallLocator to GaussBellTeamBallLocator
*
* Revision 1.9  2003/04/14 16:00:42  loetzsch
* ATH after GermanOpen CVS merge:
* added timeUntilSeenConsecutively
*
* Revision 1.2  2003/04/11 21:45:35  Jan Hoffmann
* Added timeUntilSeenConsecutively
*
* Revision 1.1.1.1  2003/04/09 14:22:16  loetzsch
* started Aibo Team Humboldt's GermanOpen CVS
*
* Revision 1.8  2003/04/03 21:07:32  jhoffman
* added "consecutive time ball seen"
*
* Revision 1.7  2003/03/31 13:22:13  jhoffman
* added propagated speed
*
* Revision 1.6  2003/02/14 14:34:02  wachter
* Added SensorFusionBallLocator
*
* Revision 1.5  2003/01/30 13:16:48  loetzsch
* Redesign of class BallPosition
*
* Revision 1.4  2002/11/26 12:25:48  dueffert
* doxygen docu corrected
*
* Revision 1.3  2002/11/21 16:08:39  dueffert
* doxygen comments corrected
*
* Revision 1.2  2002/09/22 18:40:50  risler
* added new math functions, removed GTMath library
*
* Revision 1.1  2002/09/10 15:26:39  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed Challenge Code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.2  2002/07/23 13:32:56  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.12  2002/04/25 20:29:57  roefer
* New BallPercept and BallPosition, GTMath errors in SimGT2002 fixed
*
* Revision 1.11  2002/04/06 02:28:45  loetzsch
* added time when ball was seen last
*
* Revision 1.10  2002/02/11 11:19:57  roefer
* no message
*
* Revision 1.9  2002/02/11 11:13:06  roefer
* BallPerceptor and BallLocator inserted
*
* Revision 1.8  2002/02/05 03:30:52  loetzsch
* replaced direct member access by
* inline const VALUE& get...() const   and
* inline void set...(const Value&) methods.
*
* Revision 1.7  2002/01/27 22:14:54  roefer
* BallPosition is now a pair of two vectors
*
* Revision 1.6  2002/01/25 15:40:13  roefer
* The Oracle
*
* Revision 1.5  2002/01/22 00:06:55  loetzsch
* In den Get...() Funktionen wurden die Parameter nicht als Referenz bergeben,
* gendert
*
* Revision 1.4  2002/01/11 23:50:09  xiang
* BallPosition von Hong & Lang
*
* Revision 1.3  2001/12/10 17:47:05  risler
* change log added
*
*/
