/**
* @file BallPercept.h
*
* Declaration of class BallPercept
*
* @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
*/ 

#ifndef __BallPercept_h_
#define __BallPercept_h_


#include "Tools/Streams/InOut.h"
#include "Tools/Math/Vector2.h"
#include "Tools/Math/Vector3.h"
#include "Tools/Math/Common.h"
#include "Tools/FieldDimensions.h"
#include "Tools/RobotConfiguration.h"

/**
* Contains a percept for a ball
*/ 
class BallPercept
{
public:
/**
* Constructor.
  */
  BallPercept() {reset(0);}
  BallPercept(const BallPercept& other) {*this = other;}
  BallPercept& operator=(const BallPercept& other);
  
  bool ballWasSeen; /**< Indicates if the ball was seen. */
  unsigned long frameNumber; /**< The frame number of the image that was used to create the percept. */
  
  /** The functions resets the percept. */
  void reset(unsigned long frameNumber) {this->frameNumber = frameNumber; ballWasSeen = false;}
  
  /** The function adds a seen ball to the percept */
  void add(
    Vector2<double> anglesToCenter,
    double ballRadiusAsAngle,
    Vector3<double> translationOfCamera,
    bool isCameraMatrixValid
    );
  
  /** Adds a seen ball to the percept, specifing its center and radius in image coords */
  void add(
    CameraInfo cameraInfo,
    Vector2<int> ballCenter,
    double ballRadiusInPixel,
    Vector2<double> anglesToCenter,
    double ballRadiusAsAngle,
    Vector3<double> translationOfCamera,
    bool isCameraMatrixValid
    );

  /** 
  * The horizontal and the vertical angle from the camera to the ball.
  * The reference is not the optical axis but the axes of the coordinate system of the robot.
  */
  Vector2<double> anglesToCenter; 
  
  /**
  * The translation of the camera relative to the robot.
  */
  Vector3<double> translationOfCamera;
  
  /**
  * The radius of the ball in the image (measure: angle)
  */
  double ballRadiusAsAngle;
  
  
  /** The center of the ball in pixel coords */
  Vector2<int> ballCenter;
  
  /** The radius of the ball in pixels */
  double ballRadiusInPixel;
  
  /** Indicates, if the camera matrix was reliable when the percept was created */
  bool isCameraMatrixValid; 
  
  CameraInfo cameraInfo;
  
private:
  
  void checkOffset(Vector2<double>& offset) const
  {
    if(offset.x < ballRadius / 2 && fabs(offset.y) < getRobotConfiguration().getRobotDimensions().lowerBodyWidth / 2 + ballRadius)
      offset.x = ballRadius / 2;
  }
  
public:
  double getAngle() const;
  double getDistance() const;
  double getAngleBearingBased() const;
  double getDistanceBearingBased() const;
  double getAngleSizeBased() const;
  double getDistanceSizeBased() const;

  /** The following methods are calculating the ball position based on size, position in image plane 
  *     coords and camera intrinsic parameters
  */
  double getAngleIntrinsicBased() const;
  double getDistanceIntrinsicBased() const;


  void getOffset(Vector2<double>& offset) const;
  void getOffsetSizeBased(Vector2<double>& offset) const;
  void getOffsetBearingBased(Vector2<double>& offset) const;

  /** Calculates ball position based on intrinsic camera parameters, instead of opening angle width*/
  void getOffsetIntrinsic(Vector2<double>& offset) const;
};

/**
* Streaming operator that reads a BallPercept from a stream.
* @param stream The stream from which is read.
* @param ballPercept The BallPercept object.
* @return The stream.
*/ 
In& operator>>(In& stream,BallPercept& ballPercept);

/**
* Streaming operator that writes a BallPercept to a stream.
* @param stream The stream to write on.
* @param ballPercept The BallPercept object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const BallPercept& ballPercept);

#endif //__BallPercept_h_

/*
* Change log :
* 
* $Log: BallPercept.h,v $
* Revision 1.11  2004/02/10 10:48:05  nistico
* Introduced Intrinsic camera parameters to perform geometric calculations (distance, angle, size...) without opening angle
* Implemented radial distortion correction function
* Implemented ball distance calculation based on size and intrinsic params (potentially more stable)
* To Be Done: calculate intrinsic params for ERS7, as soon as we get our puppies back
*
* Revision 1.10  2004/01/19 14:53:46  dueffert
* all frameNumbers (and not only some of them) are unsigned long now
*
* Revision 1.9  2004/01/01 10:58:51  roefer
* RobotDimensions are in a class now
*
* Revision 1.8  2003/12/15 11:46:13  juengel
* Introduced CameraInfo
*
* Revision 1.7  2003/12/11 22:52:47  loetzsch
* fixed doxygen bugs
*
* Revision 1.6  2003/11/16 14:43:58  goehring
* getBallDistance and Angle methods for size- and bearing  added
*
* Revision 1.5  2003/11/15 17:58:45  juengel
* implemented getOffset
*
* Revision 1.4  2003/11/15 17:09:02  juengel
* changed BallPercept
*
* Revision 1.3  2003/11/12 16:19:35  goehring
* frameNumber added to percepts
*
* Revision 1.2  2003/11/05 16:34:28  goehring
* FrameNumber added
*
* Revision 1.1  2003/10/07 10:09:36  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/25 11:23:52  juengel
* Removed BlobCollection.
*
* Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.5  2003/02/27 13:29:17  engel
* Validity for the BallPercept
*
* Revision 1.4  2002/11/21 16:08:39  dueffert
* doxygen comments corrected
*
* Revision 1.3  2002/09/22 18:40:49  risler
* added new math functions, removed GTMath library
*
* Revision 1.2  2002/09/17 23:55:20  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.1  2002/09/10 15:26:40  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed Challenge Code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.2  2002/07/23 13:32:56  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.10  2002/04/25 20:29:57  roefer
* New BallPercept and BallPosition, GTMath errors in SimGT2002 fixed
*
* Revision 1.9  2002/04/02 13:10:18  dueffert
* big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
*
* Revision 1.8  2002/02/11 11:19:56  roefer
* no message
*
* Revision 1.7  2002/02/11 11:13:06  roefer
* BallPerceptor and BallLocator inserted
*
* Revision 1.6  2002/02/05 03:30:52  loetzsch
* replaced direct member access by
* inline const VALUE& get...() const   and
* inline void set...(const Value&) methods.
*
* Revision 1.5  2002/01/22 00:06:55  loetzsch
* In den Get...() Funktionen wurden die Parameter nicht als Referenz bergeben,
* gendert
*
* Revision 1.4  2002/01/11 23:50:44  xiang
* BallPercept von Hong & Lang
*
* Revision 1.3  2001/12/10 17:47:05  risler
* change log added
*
*/
