/**
 * @file BodyPercept.h
 * 
 * Declaration of class BodyPercept
 */ 

#ifndef __BodyPercept_h_
#define __BodyPercept_h_


#include "Tools/Streams/InOut.h"
#include "Tools/Math/Vector3.h"

/**
 * The class represents the robots body percept
 *
 * The percept consists of current switches and a variable containing
 * whether the robot stands or is crashed. */
class BodyPercept
{
public:

  enum States { undefined, standing, crashed, rollLeft, rollRight, pickedUp };

  enum TailPositions {
    tailLeftTop,    tailCenterTop,    tailRightTop,
    tailLeftCenter, tailCenterCenter, tailRightCenter,
    tailLeftBottom, tailCenterBottom, tailRightBottom
  };

  /** bit position of switch bits in switches value */
  enum Switches { 
                  back = 0,
                  backFront, // ERS7 only
                  backBack,  // ERS7 only
                  head,
                  headFront, // ERS 210 only
                  headBack,  // ERS 210 only
                  mouth,
                  chin,
                  tailUp,
                  tailDown,
                  tailLeft,
                  tailRight,
                    numOfSwitches};
  
  enum MouthStates { mouthOpen, mouthClosed };
  
  Vector3<double> acceleration;

  BodyPercept();
  ~BodyPercept();

  /**< The frame number when perceived. */
  unsigned long frameNumber; 

  /**
  * The functions sets the frame number
  */
  void setFrameNumber(unsigned long frameNumber) {this->frameNumber = frameNumber;}

  void setState(const States s) {state = s;}
  States getState() const {return state;}

  void setSwitches(const int s) {switches = s;}
  int getSwitches() const {return switches;}

  void setTailPosition(const TailPositions t) {tailPosition = t;}
  TailPositions getTailPosition() const {return tailPosition;}
  
  void setMouthState(const MouthStates t) {mouthState = t;}
  MouthStates getMouthState() const {return mouthState;}

  void setForelegOpeningAngle(const double angle) {forelegOpeningAngle = angle;}
  double getForelegOpeningAngle() const {return forelegOpeningAngle;}

  /** copies another BodyPercept to this one */
  void operator = (const BodyPercept& other);

  static const char* getStateName(States s)
  {
    switch (s)
    {
      case undefined: return "undefined";
      case standing: return "standing";
      case crashed: return "crashed";
      case rollLeft: return "rollLeft";
      case rollRight: return "rollRight";
      case pickedUp: return "pickedUp";
      default: return "please edit BodyPercept::getStateName";
    }
  }

private:

  /**
   * current state of robot position
   * if crashed or standing
   */
  States state;

  /** value representing pressed switches bits according to switches enum */
  int switches;

  TailPositions tailPosition;

  MouthStates mouthState;

  double forelegOpeningAngle;

};

/**
 * Streaming operator that reads a BodyPercept from a stream.
 * @param stream The stream from which is read.
 * @param bodyPercept The BodyPercept object.
 * @return The stream.
 */ 
In& operator>>(In& stream,BodyPercept& bodyPercept);
 
/**
 * Streaming operator that writes a BodyPercept to a stream.
 * @param stream The stream to write on.
 * @param bodyPercept The BodyPercept object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const BodyPercept& bodyPercept);


#endif //__BodyPercept_h_

/*
 * Change log :
 * 
 * $Log: BodyPercept.h,v $
 * Revision 1.4  2004/01/28 08:32:28  dueffert
 * frameNumber is unsigned long everywhere, missing cases added
 *
 * Revision 1.3  2004/01/24 11:27:39  juengel
 * Added ERS7 switches (head, backFront, backBack).
 *
 * Revision 1.2  2003/11/14 19:02:25  goehring
 * frameNumber added
 *
 * Revision 1.1  2003/10/07 10:09:36  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.3  2003/09/25 11:23:52  juengel
 * Removed BlobCollection.
 *
 * Revision 1.2  2003/07/06 12:05:31  schumann
 * added foreleg opening angle for ball challenge
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2003/06/20 15:32:32  dueffert
 * getting from down from wall by rolling added
 *
 * Revision 1.5  2003/06/20 14:50:07  risler
 * numOfSwitches added
 *
 * Revision 1.4  2003/06/20 13:27:18  risler
 * added tailLeft tailRight messages
 *
 * Revision 1.3  2003/04/01 22:40:45  cesarz
 * added mouth states
 *
 * Revision 1.2  2003/01/30 11:26:47  juengel
 * Added tailPosition to bodyPercept
 *
 * Revision 1.1  2002/09/10 15:26:40  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.3  2002/08/22 14:41:03  risler
 * added some doxygen comments
 *
 * Revision 1.2  2002/07/23 13:32:57  loetzsch
 * new streaming classes
 *
 * removed many #include statements
 *
 * Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.9  2002/04/08 17:48:53  risler
 * added pickedUp
 *
 * Revision 1.8  2002/03/28 15:19:20  risler
 * implemented switch messages in RobotStateDetector
 *
 * Revision 1.7  2002/02/23 16:37:15  risler
 * State in RobotState like BodyPercept state
 *
 * Revision 1.6  2002/02/05 03:30:52  loetzsch
 * replaced direct member access by
 * inline const VALUE& get...() const   and
 * inline void set...(const Value&) methods.
 *
 * Revision 1.5  2002/02/01 15:03:23  risler
 * Removed pickup detection, as not working with paw sensors
 *
 * Revision 1.4  2002/01/18 11:19:41  petters
 * data fields and access methods added
 *
 * Revision 1.3  2001/12/10 17:47:05  risler
 * change log added
 *
 */
