/**
* @file CameraInfo.cpp
*
* Implementation of class CameraInfo.
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
* @author <a href="mailto:walter.nistico@uni-dortmund.de">Walter Nistico</a>
*/

#include "CameraInfo.h"

CameraInfo::CameraInfo() : 
resolutionWidth(cameraResolutionWidth_ERS210),
resolutionHeight(cameraResolutionHeight_ERS210),
openingAngleWidth(openingAngleWidth_ERS210),
openingAngleHeight(openingAngleHeight_ERS210),
focalLength(focalLength_ERS210),
opticalCenter(opticalCenterX_ERS210, opticalCenterY_ERS210),
secondOrderRadialDistortion(secondOrderRadialDistortion_ERS210),
fourthOrderRadialDistortion(fourthOrderRadialDistortion_ERS210),
focalLenPow2(focalLength*focalLength),
focalLenPow4(focalLength*focalLength*focalLength*focalLength),
focalLengthInv(1.0/focalLength)
{
} 

CameraInfo::CameraInfo(RobotDesign::Design robotDesign) : 
resolutionWidth(robotDesign == RobotDesign::ERS210 ? cameraResolutionWidth_ERS210 : cameraResolutionWidth_ERS7),
resolutionHeight(robotDesign == RobotDesign::ERS210 ? cameraResolutionHeight_ERS210 : cameraResolutionHeight_ERS7),
openingAngleWidth(robotDesign == RobotDesign::ERS210 ? openingAngleWidth_ERS210 : openingAngleWidth_ERS7),
openingAngleHeight(robotDesign == RobotDesign::ERS210 ? openingAngleHeight_ERS210 : openingAngleHeight_ERS7),
focalLength(robotDesign == RobotDesign::ERS210 ? focalLength_ERS210 : focalLength_ERS7),
opticalCenter(robotDesign == RobotDesign::ERS210 ? opticalCenterX_ERS210 : opticalCenterX_ERS7, 
                  robotDesign == RobotDesign::ERS210 ? opticalCenterY_ERS210 : opticalCenterY_ERS7),
secondOrderRadialDistortion(robotDesign == RobotDesign::ERS210 ? secondOrderRadialDistortion_ERS210 : secondOrderRadialDistortion_ERS7),
fourthOrderRadialDistortion(robotDesign == RobotDesign::ERS210 ? fourthOrderRadialDistortion_ERS210 : fourthOrderRadialDistortion_ERS7),
focalLenPow2(focalLength*focalLength),
focalLenPow4(focalLength*focalLength*focalLength*focalLength),
focalLengthInv(1.0/focalLength)
{
} 

In& operator>>(In& stream,CameraInfo& cameraInfo)
{
  stream >> cameraInfo.resolutionWidth;
  stream >> cameraInfo.resolutionHeight;
  stream >> cameraInfo.openingAngleWidth;
  stream >> cameraInfo.openingAngleHeight;
  stream >> cameraInfo.focalLength;
  stream >> cameraInfo.opticalCenter;
  stream >> cameraInfo.secondOrderRadialDistortion;
  stream >> cameraInfo.fourthOrderRadialDistortion;
  stream >> cameraInfo.focalLenPow2;
  stream >> cameraInfo.focalLenPow4;
  cameraInfo.focalLengthInv = 1.0/cameraInfo.focalLength;
  return stream;
}

Out& operator<<(Out& stream, const CameraInfo& cameraInfo)
{
  stream << cameraInfo.resolutionWidth;
  stream << cameraInfo.resolutionHeight;
  stream << cameraInfo.openingAngleWidth;
  stream << cameraInfo.openingAngleHeight;
  stream << cameraInfo.focalLength;
  stream << cameraInfo.opticalCenter;
  stream << cameraInfo.secondOrderRadialDistortion;
  stream << cameraInfo.fourthOrderRadialDistortion;
  stream << cameraInfo.focalLenPow2;
  stream << cameraInfo.focalLenPow4;
  return stream;
}

/*
* Change log :
* 
* $Log: CameraInfo.cpp,v $
* Revision 1.4  2004/03/09 11:32:08  nistico
* - Intrinsic parameters based measurements can now be triggered through a single conditional compilation
* switch located in CameraInfo.h
* - Implemented fast (look-up table based) radial distortion correction
*
* Revision 1.3  2004/02/10 10:48:05  nistico
* Introduced Intrinsic camera parameters to perform geometric calculations (distance, angle, size...) without opening angle
* Implemented radial distortion correction function
* Implemented ball distance calculation based on size and intrinsic params (potentially more stable)
* To Be Done: calculate intrinsic params for ERS7, as soon as we get our puppies back
*
* Revision 1.2  2004/01/23 00:10:02  roefer
* New constructor added, opening angles from Sony
*
* Revision 1.1  2003/12/15 11:13:15  juengel
* Introduced CameraInfo
*
*/
