/** 
* @file ColorTable32K.h
* Declaration of class ColorTable32K.
*
* @author <A href=mailto:walter.nistico@uni-dortmund.de>Walter Nistico</A>
*/

#ifndef _ColorTable32K_h_
#define _ColorTable32K_h_

#include "ColorTable.h"
#include "Tools/Streams/InOut.h"
#include "Tools/Math/Vector3.h"
#include "ColorClassImage.h"
#include "Image.h"

/**
* @class ColorTable32K
*
* Contains a ColorTable32K which can decode the color for
* every 16 * 4 * 4 cube in the 255 * 255 * 255 YUV color space.
* The 64K elements color cube is packed into a 32KBytes color table (each element is just 4bits)
* accessed through fast shift and masking operations
*
* @author <A href=mailto:walter.nistico@uni-dortmund.de>Walter Nistico</A>
*/
class ColorTable32K : public ColorTable
{
public:
  /** Constructor */
  ColorTable32K();

  /** Destructor */
  ~ColorTable32K();

  /** 
  * Calculates the color class of a pixel, it's fast because is non-virtual and inline,
  * obviously can't be used through the common ColorTable interface
  * @param y the y value of the pixel
  * @param u the u value of the pixel
  * @param v the v value of the pixel
  * @return the color class
  */
  inline colorClass getColorClassFast(const unsigned char y, 
    const unsigned char u, 
    const unsigned char v) const
  {
    unsigned char subY = (y&0x10)>>2;
    unsigned char internal_y = y>>5;
    unsigned char internal_u = u>>2;
    unsigned char internal_v = v>>2;
    int colorMapPos = (internal_y<<12)|(internal_v<<6)|internal_u;
    return (colorClass) (((colorClasses[colorMapPos])>>subY)&0x0f);
  }


  /** 
  * Calculates the color class of a pixel.
  * @param y the y value of the pixel
  * @param u the u value of the pixel
  * @param v the v value of the pixel
  * @return the color class
  */
  virtual colorClass getColorClass (const unsigned char y, 
    const unsigned char u, 
    const unsigned char v) const;

  /**
  * Computes an axis aligned box around all positions of a specified color
  * in YUV space
  * @param color The color
  * @param pNear The corner of the box nearest to the origin
  * @param pFar The corner of the box farthest to the origin
  */
  void getBoxAroundColorClass(colorClass color, 
                              Vector3<int>& pNear, Vector3<int>& pFar);

  /**
  * Segments an image to a color class image.
  * 
  * This doesn't need to be used in the image processor, but is needed for visualisation
  * of color tables.
  * @param image A reference to the image to be segmented
  * @param colorClassImage A reference to the color class image to be created
  */
  virtual void generateColorClassImage(const Image& image, 
    ColorClassImage& colorClassImage) const;

  /**
  * Generates an image that contains all pixels that have the specified color class.
  * 
  * @param image A reference to the image to be segmented
  * @param colorClassImage A reference to the color class image to be created
  * @param colorClass The color class. 
  */
  virtual void generateColorClassImage(const Image& image, 
    ColorClassImage& colorClassImage,
    colorClass colorClass
    ) const;

  /**
  * Segments an image to an color class image.
  * 
  * This doesn't need to used in the image processor, but is needed for visualisation
  * of color tables.
  * @param image A reference to the image to be segmented
  * @param colorClassImage A reference to the color class image to be created
  */
  virtual void generateHighResColorClassImage(const Image& image, 
    ColorClassImage& colorClassImage) const;

  /**
  * Generates an image that contains all pixels that have the specified color class.
  * 
  * @param image A reference to the image to be segmented
  * @param colorClassImage A reference to the color class image to be created
  * @param colorClass The color class. 
  */
  virtual void generateHighResColorClassImage(const Image& image, 
    ColorClassImage& colorClassImage,
    colorClass colorClass
    ) const;

  /**
  * Unpacks the color table into a 64K array, used for performance testing 
  * (higher cache hit rate Vs less instructions per pixel)
  * 
  * @param unpackedCube Pointer to the unpacked buffer
  * @param packedCube Pointer to the packed buffer
  */
  void unpack(unsigned char* unpackedCube, const unsigned char* packedCube);

  /** 
  * The color table (array of 32K elements).
  * Each element in the array contains 2 points in the color space having same U,V and adjacent Y.
  * Quantization is 16x4x4 (YUV)
  */
  unsigned char colorClasses[8*64*64];

  /** 
  * The unpacked color table (array of 64K elements), used for performance testing
  */
  unsigned char colorClassesUnpacked[16*64*64];
  
  //!@name Members to be used by the ColorTable32K dialog
  //!@{

  /** Resets the color table to noColor */
  void clear();
  
  /** Sets all cubes that have the given color class to noColor */
  void clearChannel(colorClass colorClass);
  
  /** 
  * Sets the color class for a pixel in the color space given by y, u, v
  * to the given color class.
  */
  void addColorClass(
    colorClass colorClass,
    unsigned char y,
    unsigned char u,
    unsigned char v
    );

  /** 
  * Sets the color class for a cube with the size "range" around a pixel
  * given by y,u,v to the given color class.
  */
  void addColorClass(
    colorClass colorClass,
    unsigned char y, 
    unsigned char u, 
    unsigned char v, 
    unsigned char range
    );

  /**
  * Sets the color class for all sub cubes within the specified cuboid to the given color class.
  */
  void addCuboidToColorClass(
    colorClass colorClass,
    unsigned char yMin, 
    unsigned char uMin, 
    unsigned char vMin, 
    unsigned char yMax, 
    unsigned char uMax, 
    unsigned char vMax
    );
  
  /**
  * Sets the color class for a cube with the size "range" around a pixel
  * given by y,u,v to noColor.
  */
  void removeColorClass(
    unsigned char y, 
    unsigned char u, 
    unsigned char v, 
    unsigned char range
    );

};

/**
 * Streaming operator that reads a ColorTable32K from a stream.
 * @param stream The stream from which is read.
 * @param colorTable32K The ColorTable32K object.
 * @return The stream.
 */ 
In& operator>>(In& stream,ColorTable32K& colorTable32K);
 
/**
 * Streaming operator that writes a ColorTable32K to a stream.
 * @param stream The stream to write on.
 * @param colorTable32K The ColorTable32K object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const ColorTable32K& colorTable32K);

#endif   //  _ColorTable32K_h_

/*
* Change log :
* 
* $Log: ColorTable32K.h,v $
* Revision 1.7  2004/05/09 15:48:14  nistico
* Speed optimized
*
* Revision 1.6  2004/05/06 16:01:41  nistico
* Integrated functions required to run with GT2004ImageProcessor
*
* Revision 1.5  2004/02/16 12:26:40  nistico
* Added noise reduction functionality for jpeg images in log file player
*
* Revision 1.4  2004/02/03 11:35:35  nistico
* ColorTable32KImageProcessor is again bound to ColorTable32K, for performance testings
* ColorTable32K now supports unpacked access format, for performance testings
* Under evaluation the performance impact of higher cache hit rate Vs fewer instructions per pixel
*
* Revision 1.3  2004/01/23 17:07:33  nistico
* Added FastSUSANNoiseReduction.*, a non-linear image noise reduction filter, to ImageProcessorTools
* Added debug visualization of segmented image in ColorTable32KImageProcessor, and experimental support of FastSUSANNoiseReduction
*
* Revision 1.2  2004/01/21 14:11:07  nistico
* Added change log string to ColorTable32KImageProcessor.*
*
*
*/
