/**
 * @file Image.h
 * 
 * Declaration of class Image
 */ 

#ifndef __Image_h_
#define __Image_h_

#include "CameraInfo.h"

// forward declaration
class Image;

#include "Tools/Streams/InOut.h"
#include "Representations/Perception/ColorTable.h"
#include "Tools/ColorClasses.h"
#include "Tools/Math/Common.h"
#include <string.h>

/**
* Platform independend definition of an image class
*/
class Image
{
public:
  /** constructs an image */
  Image();

  /** deconstructs an image */
  ~Image();

  bool          hasColorTable(void);
  void          setColorTable(const ColorTable* ct);
  char          getClassifiedColor(int x, int y);
  char          getClassifiedColor(unsigned long index);
  

  /** Converts an YUV image into an RGB image.
   *  @param yuvImage The given YUV image
   */
  void convertFromYUVToRGB(const Image& yuvImage);

  /** Converts an YCbCr pixel into an RGB pixel.
   *  @param Y The Y channel of the source pixel.
   *  @param Cb The Cb channel of the source pixel.
   *  @param Cr The Cr channel of the source pixel.
   *  @param R The R channel of the target pixel.
   *  @param G The G channel of the target pixel.
   *  @param B The B channel of the target pixel.
   */
  static void convertFromYCbCrToRGB(unsigned char Y,
                                    unsigned char Cb,
                                    unsigned char Cr,
                                    unsigned char& R,
                                    unsigned char& G,
                                    unsigned char& B)
  {
	  int r = (int)(Y + 1.4021 * (Cb - 128)),
	      g = (int)(Y - 0.3456 * (Cr - 128) - 0.71448 * (Cb - 128)),
	      b = (int)(Y + 1.7710 * (Cr - 128));
    if(r < 0) r = 0; else if(r > 255) r = 255;
    if(g < 0) g = 0; else if(g > 255) g = 255;
    if(b < 0) b = 0; else if(b > 255) b = 255;
    R = (unsigned char) r;
    G = (unsigned char) g;
    B = (unsigned char) b;
  }

  /** Converts an YCbCr image into an RGB image.
   *  @param ycbcrImage The given YCbCr image
   */
  void convertFromYCbCrToRGB(const Image& ycbcrImage);

  /** Converts an RGB image into an YUV image.
   *  @param rgbImage The given RGB image
   */
  void convertFromRGBToYUV(const Image& rgbImage);

  /** Converts an YCbCr pixel into an HSI pixel.
   *  @param Y The Y channel of the source pixel.
   *  @param Cb The Cb channel of the source pixel.
   *  @param Cr The Cr channel of the source pixel.
   *  @param H The H channel of the target pixel.
   *  @param S The S channel of the target pixel.
   *  @param I The I channel of the target pixel.
   */
  static void convertFromYCbCrToHSI(unsigned char Y,
                                    unsigned char Cb,
                                    unsigned char Cr,
                                    unsigned char& H,
                                    unsigned char& S,
                                    unsigned char& I)
  {
    const double sqrt3 = 1.732050808;
    unsigned char R,
                  G,
                  B;
    convertFromYCbCrToRGB(Y, Cb, Cr, R, G, B);
    I = R;
    if(G > I) I = G;
    if(B > I) I = B;
    if(I)
    {
      S = R;
      if(G < S) S = G;
      if(B < S) S = B;
      S = (unsigned char) (255 - 255 * S / I);
      if(S)
      {
        int h = int(atan2(sqrt3 * (G - B), 2 * R - G - B) / pi2 * 256);
        if(h > 256) h -= 256;
        else if(h < 0) h += 256;
        H = (unsigned char) h;
      }
      else
        H = 0;
    }
    else
      S = H = 0;
  }

  /** Converts an YCbCr image into an HSI image.
   *  @param yuvImage The given YUV image
   */
  void convertFromYCbCrToHSI(const Image& ycrcbImage);

  /** Converts an YUV image into an TSL image.
   *  @param yuvImage The given YUV image
   */
  void convertFromYUVToTSL(const Image& yuvImage);

	/** Returns the high resolution y value of a pixel
   *  @param x The x coord. of the pixel in high resolution
   *  @param y The y coord. of the pixel in high resolution
	 *  @return The y value of the pixel
	 */
	unsigned char getHighResY(int x, int y) const;

  /** Sets the high resolution y values for one pixel in low resolution 
   * @params x The x coord. of the pixel
   * @params y The y coord. of the pixel
   * @params tl The y value of the top left subpixel
   * @params bl The y value of the bottom left subpixel
   * @params tr The y value of the top right subpixel
   * @params br The y value of the bottom right subpixel
   */
  void setHighResY(int x, int y, unsigned char tl, unsigned char bl, unsigned char tr, unsigned char br);

  /** Assignment operator
  *\param other The other Image that is assigned to this one
  *\return A reference to this object after the assignment.
  */
  Image& operator=(const Image& other) 
  {
    memcpy((void *)this, (const void *)&other, sizeof(Image));
    return *this;
  }

  /** representation for an image
  * height color width
  * point of origin is the upper left corner, height is positive downwards
  * and width positive to the right
  * the color values are in the order Y,U,V
  */
  unsigned char image[cameraResolutionHeight_ERS7][6][cameraResolutionWidth_ERS7];

  CameraInfo cameraInfo;

  /** the height of the image */
//  int height;

  /**  the width of the image  */
//  int width;

  /** the frame number of that image */
  unsigned long frameNumber;

  /** associated color table */
  const ColorTable* colorTable;

  /**
  * set values in CameraInfo according to image size
  */
  void setCameraInfo();
};

/**
 * Streaming operator that writes an Image to a stream.
 * @param stream The stream to write on.
 * @param image The Image object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const Image& image);

/**
 * Streaming operator that reads a Image from a stream.
 * @param stream The stream to read from.
 * @param image The Image object.
 * @return The stream.
 */ 
In& operator>>(In& stream,Image& image);

#endif //__Image_h_

/*
 * Change log :
 * 
 * $Log: Image.h,v $
 * Revision 1.6  2004/05/01 17:09:33  roefer
 * Streamlined HSI stuff
 *
 * Revision 1.5  2004/04/27 17:22:06  thomas
 * added convertFromYCbCr2RGB for images
 *
 * Revision 1.4  2004/04/07 13:00:44  risler
 * ddd checkin after go04 - second part
 *
 * Revision 1.5  2004/04/07 11:44:05  risler
 * added sending low res images
 * added Image::setCameraInfo
 *
 * Revision 1.4  2004/04/06 13:19:35  risler
 * cleaned up and improved high resolution image support
 *
 * Revision 1.3  2004/03/29 20:45:16  risler
 * bugfix getHighResY
 *
 * Revision 1.2  2004/03/29 15:19:03  Marc
 * Intruduced the Black and White Image
 * Normal Images (not Jpeg) images were now send as Color Image with BW
 *
 * Revision 1.3  2003/12/30 20:12:03  roefer
 * Image size is now 208 x 160. Smaller images are placed in the upper left corner
 *
 * Revision 1.2  2003/12/15 11:47:07  juengel
 * Introduced CameraInfo
 *
 * Revision 1.1  2003/10/07 10:09:36  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/26 15:27:27  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.9  2003/03/10 18:20:44  dueffert
 * const cast warning removed
 *
 * Revision 1.8  2003/02/24 22:30:54  juengel
 * Added convertion methods for TSL and HSI
 *
 * Revision 1.7  2003/02/19 14:59:54  roefer
 * pColorTable -> colorTable
 *
 * Revision 1.6  2003/02/18 21:29:17  osterhues
 * Changed all instances of ColorTable64 to new base class ColorTable
 *
 * Revision 1.5  2003/01/09 13:07:16  jhoffman
 * no message
 *
 * Revision 1.4  2002/12/15 23:33:02  dueffert
 * rgb-yuv-convertion moved to Image
 *
 * Revision 1.3  2002/11/25 14:48:31  jhoffman
 * added "=" operator
 *
 * Revision 1.2  2002/09/17 23:55:20  loetzsch
 * - unraveled several datatypes
 * - changed the WATCH macro
 * - completed the process restructuring
 *
 * Revision 1.1  2002/09/10 15:26:40  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.3  2002/07/23 13:32:57  loetzsch
 * new streaming classes
 *
 * removed many #include statements
 *
 * Revision 1.2  2002/05/17 09:58:48  brunn
 * added comments discribing ordinate and coordinate of image
 *
 * Revision 1.1.1.1  2002/05/10 12:40:13  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.8  2002/04/02 13:10:18  dueffert
 * big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
 *
 * Revision 1.7  2001/12/12 20:21:12  petters
 * Streaming for SensorData / Image implemented; Conflict solved
 *
 * Revision 1.6  2001/12/12 18:08:55  loetzsch
 * Streaming- Operatoren fr Bilder eingebaut, DebugKeyTable nicht- statisch gemacht, Debuggin Mechanismen weitergemacht, Bilder aus Logfiles in RobotControl anzeigen, Logfiles in HU1/Debug auf den Stick schreiben
 *
 * Revision 1.5  2001/12/10 17:47:05  risler
 * change log added
 *
 */
