/**
 * @file JPEGImage.cpp
 * 
 * Implementation of class JPEGImage
 */ 

#include "Platform/GTAssert.h"
#include "JPEGImage.h"

JPEGImage::JPEGImage(const Image& image)
{
  *this = image;
}

JPEGImage& JPEGImage::operator=(const Image& src)
{
  cameraInfo = src.cameraInfo;
  frameNumber = src.frameNumber;

  jpeg_compress_struct cInfo;
  jpeg_error_mgr jem;
  cInfo.err = jpeg_std_error(&jem);
  jpeg_create_compress(&cInfo);

  if(!cInfo.dest)
    cInfo.dest = (jpeg_destination_mgr*)
      (*cInfo.mem->alloc_small) ((j_common_ptr) &cInfo,JPOOL_PERMANENT,
                                 sizeof(DestDescriptor));
  cInfo.dest->init_destination = onDestInit;
  cInfo.dest->empty_output_buffer = onDestEmpty;
  cInfo.dest->term_destination = onDestTerm;
  ((DestDescriptor*) cInfo.dest)->theObject = this;

  cInfo.image_width = cameraInfo.resolutionWidth;
  cInfo.image_height = cameraInfo.resolutionHeight * 3;
  cInfo.input_components = 1;
  cInfo.in_color_space = JCS_GRAYSCALE;
  jpeg_set_defaults(&cInfo);
  cInfo.dct_method = JDCT_FASTEST;
  jpeg_set_quality(&cInfo,75,true);

  jpeg_start_compress(&cInfo,true);

  while(cInfo.next_scanline < cInfo.image_height) 
  {
    JSAMPROW rowPointer = const_cast<JSAMPROW>(&src.image[cInfo.next_scanline % src.cameraInfo.resolutionHeight]
                                                         [cInfo.next_scanline / src.cameraInfo.resolutionHeight][0]);
    jpeg_write_scanlines(&cInfo,&rowPointer,1);
  }

  jpeg_finish_compress(&cInfo);
  jpeg_destroy_compress(&cInfo);
  return *this;
}

void JPEGImage::toImage(Image& dest) const
{
  dest.cameraInfo = cameraInfo;
  dest.frameNumber = frameNumber;

  jpeg_decompress_struct cInfo;
  jpeg_error_mgr jem;
  cInfo.err = jpeg_std_error(&jem);

  jpeg_create_decompress(&cInfo);

  if(!cInfo.src)
    cInfo.src = (jpeg_source_mgr *)
      (*cInfo.mem->alloc_small)((j_common_ptr) &cInfo,JPOOL_PERMANENT,
                                 sizeof(jpeg_source_mgr));
  cInfo.src->init_source       = onSrcIgnore;
  cInfo.src->fill_input_buffer = onSrcEmpty;
  cInfo.src->skip_input_data   = onSrcSkip;
  cInfo.src->resync_to_restart = jpeg_resync_to_restart;
  cInfo.src->term_source       = onSrcIgnore;
  cInfo.src->bytes_in_buffer = size;
  cInfo.src->next_input_byte = (const JOCTET*) image;

  jpeg_read_header(&cInfo,true);
  jpeg_start_decompress(&cInfo);

  // setup rows
  while (cInfo.output_scanline < cInfo.output_height) 
  {
    JSAMPROW rowPointer = (unsigned char*) (cInfo.output_height == (unsigned) cameraResolutionHeight_ERS210
                          ? &dest.image[cInfo.output_scanline][0][0]
                          : &dest.image[cInfo.output_scanline % dest.cameraInfo.resolutionHeight]
                                       [cInfo.output_scanline / dest.cameraInfo.resolutionHeight][0]);
    (void) jpeg_read_scanlines(&cInfo,&rowPointer,1);
  }

  // decompress
  jpeg_finish_decompress(&cInfo);
  jpeg_destroy_decompress(&cInfo);

  // compatibility with old format: move to correct positions
  if(cInfo.output_height == (unsigned) cameraResolutionHeight_ERS210)
    for(unsigned i = 0; i < cInfo.output_height; ++i)
      for(unsigned c = 2; c > 0; --c)
        memmove(&dest.image[i][c][0], 
                &dest.image[i][0][c * cameraResolutionWidth_ERS210], 
                cameraResolutionWidth_ERS210);

  // clear high resolution y channels
  for(int y = 0; y < dest.cameraInfo.resolutionHeight; ++y)
    for(int c = 3; c < 6; ++c)
      memset(&dest.image[y][c][0], 128, dest.cameraInfo.resolutionWidth);
}

void JPEGImage::onDestInit(j_compress_ptr cInfo)
{
  JPEGImage& image = *((DestDescriptor*) cInfo->dest)->theObject;
  cInfo->dest->next_output_byte = (JOCTET*) image.image;
  cInfo->dest->free_in_buffer = image.cameraInfo.resolutionWidth * image.cameraInfo.resolutionHeight * 3;
}

int JPEGImage::onDestEmpty(j_compress_ptr)
{
  ASSERT(false);
  return false;
}

void JPEGImage::onDestTerm(j_compress_ptr cInfo)
{
  JPEGImage& image = *((DestDescriptor*) cInfo->dest)->theObject;
  image.size = (char*) cInfo->dest->next_output_byte - (char*) image.image;
}

void JPEGImage::onSrcSkip(j_decompress_ptr,long)
{
}

int JPEGImage::onSrcEmpty(j_decompress_ptr)
{
  ASSERT(false);
  return false;
}

void JPEGImage::onSrcIgnore(j_decompress_ptr)
{
}

Out& operator<<(Out& stream,const JPEGImage& image)
{
  ASSERT(image.size);
  stream << image.cameraInfo.resolutionWidth << image.cameraInfo.resolutionHeight << image.frameNumber << image.size;
  stream.write(&image.image,image.size);
  return stream;
}

In& operator>>(In& stream,JPEGImage& image)
{
  stream >> image.cameraInfo.resolutionWidth >> image.cameraInfo.resolutionHeight >> image.frameNumber >> image.size;
  image.setCameraInfo();
  stream.read(&image.image,image.size);
  return stream;
}

/*
 * Change log :
 * 
 * $Log: JPEGImage.cpp,v $
 * Revision 1.11  2004/04/17 12:08:35  risler
 * toImage now correctly clears hires y channels
 *
 * Revision 1.10  2004/04/07 13:00:44  risler
 * ddd checkin after go04 - second part
 *
 * Revision 1.2  2004/04/07 11:44:05  risler
 * added sending low res images
 * added Image::setCameraInfo
 *
 * Revision 1.1.1.1  2004/03/29 08:28:45  Administrator
 * initial transfer from tamara
 *
 * Revision 1.9  2004/02/29 14:48:32  roefer
 * Minor inconsistancy removed
 *
 * Revision 1.8  2004/02/12 14:21:49  nistico
 * Calculated refined (4 robots used) intrinsic camera parameters for ERS-7
 * Fixed bug with ERS-7 logfiles and intrinsic camera parameters (formerly, they weren't correctly associated
 * with the higher resolution image from the new robot)
 *
 * Revision 1.7  2004/02/02 17:50:01  roefer
 * Replaced hack by const_cast
 *
 * Revision 1.6  2004/01/10 17:26:26  roefer
 * Warnings removed
 *
 * Revision 1.5  2004/01/09 13:44:38  roefer
 * Improving image quality
 *
 * Revision 1.4  2003/12/30 20:12:03  roefer
 * Image size is now 208 x 160. Smaller images are placed in the upper left corner
 *
 * Revision 1.3  2003/12/16 19:43:50  roefer
 * Compatibility to proposed new image format
 *
 * Revision 1.2  2003/12/15 11:47:07  juengel
 * Introduced CameraInfo
 *
 * Revision 1.1  2003/10/07 10:09:36  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.4  2003/05/26 12:51:51  dueffert
 * does not work without const for some reason
 *
 * Revision 1.3  2003/05/23 11:24:25  dueffert
 * cast warnings removed, hopefully nobody wants to change (now non-const) src in JPEGImage's =operator...
 *
 * Revision 1.2  2003/03/22 18:11:23  roefer
 * Warnings removed
 *
 * Revision 1.1  2002/10/11 13:54:44  roefer
 * JPEGImage added
 *
 *
 */
