/**
 * @file Representations/WLan/TeamMessage.h
 *
 * Declaration of class TeamMessage
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 * @author <A href=mailto:sebastian.schmidt@udo.edu>Sebastian Schmidt</A>
 * @author <A href=mailto:m_wachter@gmx.de>Michael Wachter</A>
 */ 

#ifndef __TeamMessage_h_
#define __TeamMessage_h_

#include "Tools/Streams/InOut.h"
#include "Tools/Player.h"
#include "Representations/Cognition/RobotPose.h"
#include "Representations/Cognition/BallModel.h"
#include "Representations/WLan/BehaviorTeamMessage.h"

class TeamMessage;

/**
 * Streaming operator that reads a team message from a stream.
 * @param stream The stream from which is read.
 * @param teamMessage The object to be read.
 * @return The stream.
 */ 
In& operator>>(In& stream,TeamMessage& teamMessage);

/**
 * Streaming operator that writes a team message to a stream.
 * @param stream The stream to write on.
 * @param teamMessage The object to be written.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const TeamMessage& teamMessage);


/** 
 * @class TeamMessage
 *
 * Represents a collection of all data that is exchanged between the different robots.
 */
class TeamMessage
{
  public:
	  
    unsigned timeStamp, /**< The time stamp of the sending robot. */
             lastReceivedTimeStamp, /**< The timeStamp of the last package received from the receiving robot. */
             incomingTimeStamp; /**< The local receiving time of the last package received from the receiving robot. */

    RobotPose robotPose; /**< The robot pose of the robot. */
    bool sendRobotPose; /**< Send robotPose if true. */
    
    SeenBallPosition seenBallPosition; /**< The ball seen by the robot. */
    bool sendSeenBallPosition; /**< Send seenBallPosition if true. */
    
    BehaviorTeamMessage behaviorTeamMessage; /**< Messages by the behavior */
    bool sendBehaviorTeamMessage; /**< Send behaviorTeamMessage if true */
    
    signed long timeOffset; /**< timeOffset  Remote time - local time */

    Player::playerNumber playerNumberOfSender; /**< The player number of the Sender */
  
    /**
     * Constructor.
     */
    TeamMessage() {
      timeStamp = lastReceivedTimeStamp = incomingTimeStamp = 0;
      sendRobotPose = sendSeenBallPosition = sendBehaviorTeamMessage = false;
    }

    /**
     * The function determines, whether the team message is actual.
     * @return Did the communication take less than 1 seconds?
     */
    bool isActual() const;

    /**
     * This function returns the time of the remote player in the time 
     * system of the local player
     */
    long getTimeStampInOwnTime() const ;

    /**
     * This function converts a remote timeStamp to the local time system
     * @param time the timeStamp to be converted
     */
    long getTimeInOwnTime(long time) const;

    /**
     * This function converts a local timeStamp to the remote time system
     * @param time the timeStamp to be converted
     */
    long getTimeInRemoteTime(long time) const;

  private:
    static void read(In& stream,Vector2<double>& v);
    static void read(In& stream,Pose2D& p);
    static void write(Out& stream,const Vector2<double>& v);
    static void write(Out& stream,const Pose2D& p);
    friend In& operator>>(In& stream,TeamMessage& teamMessage);
    friend Out& operator<<(Out& stream, const TeamMessage& teamMessage);
};

/** A TeamMessage from team mate 1 */
class TeamMessage1 : public TeamMessage {};

/** A TeamMessage from team mate 2 */
class TeamMessage2 : public TeamMessage {};

/** A TeamMessage from team mate 3 */
class TeamMessage3 : public TeamMessage {};

class TeamMessage4 : public TeamMessage {};

#endif //__TeamMessage_h_

/*
 * Change log :
 *
 * $Log: TeamMessage.h,v $
 * Revision 1.5  2004/05/14 14:12:08  wachter
 * - Added communication support for 5 robots
 * - rewrote parts of team-communication to be faster and more stable
 *
 * Revision 1.4  2004/02/03 13:19:50  spranger
 * renamed all references to  class BallPosition to BallModel
 *
 * Revision 1.3  2003/12/06 17:45:33  loetzsch
 * replaced Player::playerRole (goalie, defender, striker1, striker2)
 * by Player::playerNumber (one, two, three, four)
 *
 * Revision 1.2  2003/11/30 01:53:18  loetzsch
 * prepared RobotControl port to Visual C++ .Net
 *
 * Revision 1.1  2003/10/07 10:09:36  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.4  2003/09/26 15:27:27  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.3  2003/08/09 14:53:10  dueffert
 * files and docu beautified
 *
 * Revision 1.2  2003/07/02 19:14:23  loetzsch
 * bug fixes, removed unused functions
 *
 * Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.8  2003/05/06 22:19:11  dueffert
 * Greenhills compilability restored
 *
 * Revision 1.7  2003/05/02 12:57:13  loetzsch
 * TeamMessage now contains a SeenBallPosition instead of a BallPercept
 *
 * Revision 1.6  2003/03/30 11:57:27  loetzsch
 * removed backward warnings
 *
 * Revision 1.5  2003/02/28 17:02:55  wachter
 * Reenabled trainer-syncronisation for RuhrpottHellhound-Behavior
 *
 * Revision 1.4  2003/02/18 13:24:36  wachter
 * added new TeamMessageCollection and TeamMessage
 *
 * Revision 1.3  2002/11/27 13:47:21  dueffert
 * doxygen docu added
 *
 * Revision 1.2  2002/09/22 18:40:51  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.1  2002/09/10 15:26:41  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed Challenge Code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.6  2002/07/23 13:32:57  loetzsch
 * new streaming classes
 *
 * removed many #include statements
 *
 * Revision 1.5  2002/06/09 15:24:51  loetzsch
 * Added TeamMessageCollection and BehaviorTeamMessage to the execute of BehaviorControl
 *
 * Revision 1.4  2002/06/06 16:10:46  roefer
 * Robot pose and ball percept added
 *
 * Revision 1.3  2002/06/03 15:27:30  roefer
 * BallLocator gets TeamMessageCollection
 *
 * Revision 1.2  2002/05/26 14:55:25  roefer
 * Team communication is working
 *
 * Revision 1.1  2002/05/16 22:36:11  roefer
 * Team communication and GTMath bugs fixed
 *
 *
 */
