/**
* @file CameraParametersToolBar.cpp
* 
* Implementation of the camera parameters toolbar.
*
* @author Joscha Bach
* @author Jan Hoffmann
*/

#include "StdAfx.h"
#include "resource.h"
#include "CameraParametersToolBar.h"
#include "RobotControlQueues.h"
#include "Tools/Location.h"

IMPLEMENT_DYNAMIC( CCameraParametersToolBar, CToolBarEx )

void CCameraParametersToolBar::Init()
{
  InBinaryFile fin(getLocation().getFilename("camera.cfg"));
  if (fin.exists()) fin >> cameraParameters;

  static TBBUTTONEX tbButtons[] =
  { 
    { {  0, IDC_CAMERA_LABEL_MODE, TBSTATE_ENABLED, TBSTYLE_BUTTON , 0, 0 }, true },
    { {  1, IDC_CAMERA_WHITE_BALANCE, TBSTATE_ENABLED, TBSTYLE_BUTTON , 0, 0 }, true },
    { {  2, IDC_CAMERA_LABEL_SHUTTER, TBSTATE_ENABLED, TBSTYLE_BUTTON , 0, 0 }, true },
    { {  3, IDC_CAMERA_SHUTTER_SPEED, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0 }, true },
    { {  4, IDC_CAMERA_LABEL_GAIN, TBSTATE_ENABLED, TBSTYLE_BUTTON , 0, 0 }, true },
    { {  5, IDC_CAMERA_GAIN, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0 }, true },
    { {  6, IDC_CAMERA_SEND_TO_ROBOT, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0 }, true },
    { {  7, IDC_CAMERA_SAVE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0 }, true }
  };

  SetBitmaps( IDB_CAMERAPARAMETERS, IDB_CAMERAPARAMETERS_COLD);
  SetButtons( sizeof( tbButtons ) / sizeof( tbButtons[ 0 ] ), tbButtons);

  AddString(&label1, IDC_CAMERA_LABEL_MODE, "Mode ", 34);
  AddString(&label2, IDC_CAMERA_LABEL_SHUTTER, "Shutter speed ", 80);
  AddString(&label3, IDC_CAMERA_LABEL_GAIN, "Gain ", 29);

  AddCombo(&m_wndWhiteBalanceCombo, IDC_CAMERA_WHITE_BALANCE, 70);
  AddCombo(&m_wndShutterSpeedCombo, IDC_CAMERA_SHUTTER_SPEED, 42);
  AddCombo(&m_wndGainCombo, IDC_CAMERA_GAIN, 42);

  for (int i=0; i<3; i++) 
  {
    m_wndWhiteBalanceCombo.AddString(CameraParameters::getShortWhiteBalanceName(CameraParameters::whiteBalance(i)));
    m_wndShutterSpeedCombo.AddString(CameraParameters::getShortShutterSpeedName(CameraParameters::shutterSpeed(i)));
    m_wndGainCombo.AddString(CameraParameters::getShortGainName(CameraParameters::gain(i)));
  }
  m_wndWhiteBalanceCombo.SetCurSel((int)cameraParameters.theWhiteBalance);
  m_wndShutterSpeedCombo.SetCurSel((int)cameraParameters.theShutterSpeed);
  m_wndGainCombo.SetCurSel((int)cameraParameters.theGain);

  send();
}

bool CCameraParametersToolBar::HasButtonText(UINT nID)
{
  switch ( nID )
  {
  case IDC_CAMERA_LABEL_MODE:
  case IDC_CAMERA_LABEL_SHUTTER:
  case IDC_CAMERA_LABEL_GAIN:
  case IDC_CAMERA_WHITE_BALANCE:
  case IDC_CAMERA_SHUTTER_SPEED:
  case IDC_CAMERA_GAIN:
  case IDC_CAMERA_SEND_TO_ROBOT:
    return true;
  case IDC_CAMERA_SAVE:
  default:
    return false;
  }
}

bool CCameraParametersToolBar::handleSelChange(UINT nID)
{
  switch (nID)
  {
  case IDC_CAMERA_WHITE_BALANCE:
  case IDC_CAMERA_SHUTTER_SPEED:
  case IDC_CAMERA_GAIN:
    send();
    return true;
  default: 
    return false;
  }

}

void CCameraParametersToolBar::send()
{
  cameraParameters.theWhiteBalance=(CameraParameters::whiteBalance)m_wndWhiteBalanceCombo.GetCurSel();
  cameraParameters.theShutterSpeed=(CameraParameters::shutterSpeed)m_wndShutterSpeedCombo.GetCurSel();
  cameraParameters.theGain=(CameraParameters::gain)m_wndGainCombo.GetCurSel();

  getQueues().toPhysical.selectedRobot.out.bin << cameraParameters;
  getQueues().toPhysical.selectedRobot.out.finishMessage(idCameraParameters);
}

bool CCameraParametersToolBar::handleCommand(UINT command)
{
  switch(command)
  {
  case IDC_CAMERA_SEND_TO_ROBOT:
    getQueues().toPhysical.selectedRobot.out.bin << cameraParameters;
    getQueues().toPhysical.selectedRobot.out.finishMessage(idCameraParameters);
    return true;
  case IDC_CAMERA_SAVE:
    {
      OutBinaryFile fout(getLocation().getFilename("camera.cfg"));
      if (fout.exists()) fout << cameraParameters; else AfxMessageBox("Can't open camera.cfg!");
      return true;
    }
  default:
    return false;
  }
}

/*
 * Change log :
 * 
 * $Log: CameraParametersToolBar.cpp,v $
 * Revision 1.7  2004/02/24 09:45:27  jhoffman
 * - message handling bug removed
 * - added camera toolbar functionality to menu
 *
 * Revision 1.6  2004/02/23 21:35:23  jhoffman
 * - message handling bug removed
 * - added camera toolbar functionality to menu
 *
 * Revision 1.5  2004/02/23 15:35:31  jhoffman
 * warnings removed
 *
 * Revision 1.4  2004/02/23 13:27:25  jhoffman
 * - changes to Main-menu are shown ... this currently results in one additional menu (!) somebody please fix this!
 * - added GenericModalBinary-dialog: can be used to create a modal dialog where you can specify the button labels and the descriptive text (like APX...ModalDialog but you can specify what is written on the buttons)
 * - CameraToolBar: mode, shutter, gain extracted from dropdown
 * - ToolBarEx: added AddString method to write const. strings on a toolbar
 *
 * Revision 1.3  2003/12/09 19:49:24  loetzsch
 * Renamed some of the main queues of RobotControl.
 *
 * Added possibility to send messages to specific simulated or physical robots.
 *
 * Revision 1.2  2003/12/06 06:31:19  loetzsch
 * no message
 *
 * Revision 1.1  2003/10/07 10:09:37  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.4  2003/05/11 23:46:32  dueffert
 * Depend now works with RobotControl too
 *
 * Revision 1.3  2003/05/11 17:03:00  risler
 * added location.cfg
 *
 * Revision 1.2  2003/03/05 17:09:27  loetzsch
 * redesign of the queues and debug key tables in RobotControl
 *
 * Revision 1.1  2002/09/10 15:49:04  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 *
 * Revision 1.4  2002/08/13 12:55:10  loetzsch
 * - Added base class CRobotControlToolBar for tool bars
 * - Moved command handling from CRobotControlMainFrame to the tool bars
 *
 * Revision 1.3  2002/07/23 13:43:36  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - 5 instead of 3 debug queues in RobotControl
 * - exchanged StaticQueue with MessageQueue
 * - new debug message handling
 * - empty constructors in bars / dialogs
 * - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
 * - general clean up
 *
 * Revision 1.2  2002/06/10 12:00:41  dueffert
 * commentation bugs fixed
 *
 * Revision 1.1.1.1  2002/05/10 12:40:20  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.6  2002/04/23 17:45:17  loetzsch
 * - splitted debugKeyTable into debugKeyTableForRobot and debugKeyTableForLocalProcesses
 * - removed Modules Toolbar
 * - removed access for dialogs and toolbars to solutionRequest.
 * - changed access for dialogs and toolbars to debug queues
 * - removed the instance of SolutionRequest in CRobotControlApp
 * - only the log player, local processes and the robot put messages into queueToRobotControl
 *
 * Revision 1.5  2002/02/07 16:26:36  loetzsch
 * made smaller
 *
 * Revision 1.4  2001/12/14 17:05:15  loetzsch
 * CameraParameters Toolbar fertig
 *
 * Revision 1.3  2001/12/14 12:26:11  loetzsch
 * CameraParameters Toolbar weitergemacht
 *
 * Revision 1.2  2001/12/13 17:20:39  loetzsch
 * CameraParameters Toolbar weitergemacht
 *
 * Revision 1.1  2001/12/13 15:36:44  loetzsch
 * CameraParameters Toolbar
 *
 * 
 *
 */
