/**
* @file HistogramDlgBar.cpp
*
* Implementation of class CHistogramDlgBar
*
* @author Matthias Jngel
*/

#include "StdAfx.h"
#include "HistogramDlgBar.h"

#include "RobotControlDebugKeyTables.h"
#include "RobotControlQueues.h"

#include "Tools/Math/Common.h"

CHistogramDlgBar::CHistogramDlgBar()
	: CRobotControlDialogBar(IDD)
{
  //{{AFX_DATA_INIT(CHistogramDlgBar)
  //}}AFX_DATA_INIT
  bmpOffScreen = NULL;
  oldBitmap = NULL;

  numberOfHistograms = 1;
  numberOfRows = 1; numberOfColumns = 1;

  mouseOverRow = 0;
  mouseOverColumn = 0;

  currentSelection = 0;

  memset(histogramID, 0, sizeof(histogramID));
  histogramID[0] = Histogram::imageIntensityY;
  histogramID[1] = Histogram::imageIntensityU;
  histogramID[2] = Histogram::imageIntensityV;

  for(int i = 0; i < maxNumberOfHistograms; i++) scale[i] = 1;
}

CHistogramDlgBar::~CHistogramDlgBar()
{
  if(bmpOffScreen)
  {
    dcOffScreen.SelectObject(oldBitmap);
    bmpOffScreen->DeleteObject();
    delete(bmpOffScreen);
  }
  deleteBrushes();
  saveCurrentSelection();
}


void CHistogramDlgBar::DoDataExchange(CDataExchange* pDX)
{
  CDialog::DoDataExchange(pDX);
  //{{AFX_DATA_MAP(CHistogramDlgBar)
	DDX_Control(pDX, IDC_HISTOGRAM_SCALE_SLIDER, m_scaleSlider);
	DDX_Control(pDX, IDC_HISTOGRAM_SCALE_EDIT, m_scaleEdit);
	DDX_Control(pDX, IDC_HISTOGRAM_NUMBER_SLIDER, m_numberSlider);
	DDX_Control(pDX, IDC_HISTOGRAM_NUMBER_EDIT, m_numberEdit);
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CHistogramDlgBar, CDynamicBarDlg)
//{{AFX_MSG_MAP(CHistogramDlgBar)
ON_WM_PAINT()
ON_WM_SIZE()
ON_WM_CONTEXTMENU()
	ON_WM_HSCROLL()
	ON_WM_MOUSEMOVE()
	ON_WM_LBUTTONDOWN()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


BOOL CHistogramDlgBar::OnInitDialog() 
{
  CDynamicBarDlg::OnInitDialog();

  calculateRects();

  m_numberSlider.SetRange(1,maxNumberOfHistograms);
  m_scaleSlider.SetRange(0,100);

  unselectAll();
  loadSelection();

  setSliders();

  createBrushes();
  return TRUE;
}

void CHistogramDlgBar::OnPaint() 
{
  CPaintDC dc(this); // device context for painting
  
  //  CBrush backGroundBrush(GetSysColor(COLOR_WINDOWFRAME));
  currentRect.top = 0;
  currentRect.left = 0;
  currentRect.bottom = rect.bottom;
  currentRect.right = rect.right;
  
  CBrush backGroundBrush(RGB(255,255,255));
  dcOffScreen.FillRect(&currentRect, &backGroundBrush);
  
  
  int fontSize = 14;
  CFont font;
  CFont* oldFont;
  font.CreateFont(
    fontSize,                  // nHeight
    0,                         // nWidth
    0,                         // nEscapement
    0,                         // nOrientation
    FW_NORMAL,                 // nWeight
    FALSE,                     // bItalic
    FALSE,                     // bUnderline
    0,                         // cStrikeOut
    ANSI_CHARSET,              // nCharSet
    OUT_DEFAULT_PRECIS,        // nOutPrecision
    CLIP_DEFAULT_PRECIS,       // nClipPrecision
    DEFAULT_QUALITY,           // nQuality
    DEFAULT_PITCH | FF_SWISS,  // nPitchAndFamily
    "Arial");                 // lpszFacename
  oldFont = dcOffScreen.SelectObject(&font);
  
  CPen greyPen, blackPen;
  greyPen.CreatePen(PS_SOLID, 1, RGB(127,127,127));
  blackPen.CreatePen(PS_SOLID, 1, RGB(0,0,0));
  CPen* oldPen;
  
  oldPen = dcOffScreen.SelectObject(&blackPen);

  paintHistograms();
  
  dcOffScreen.SelectObject(oldFont);
  dcOffScreen.SelectObject(oldPen);
  
  greyPen.DeleteObject();
  blackPen.DeleteObject();
  font.DeleteObject();
  
  dc.BitBlt(rect.left, rect.top, rect.right, rect.bottom,
    &dcOffScreen, 0, 0, SRCCOPY);
}

void CHistogramDlgBar::OnSize(UINT nType, int cx, int cy) 
{
  calculateRects();

  CDynamicBarDlg::OnSize(nType,cx,cy);
  
  CPaintDC dc(this); // device context for painting
  
  rect.left = 0;
  rect.right = cx;
  rect.top = 25;
  rect.bottom = cy - 25;
  
  if(!dcOffScreen)
    dcOffScreen.CreateCompatibleDC(&dc);
  
  if(bmpOffScreen)
  {
    dcOffScreen.SelectObject(oldBitmap);
    bmpOffScreen->DeleteObject();
    delete(bmpOffScreen);
  }
  bmpOffScreen = new CBitmap;
  bmpOffScreen->CreateCompatibleBitmap(
    &dc, cx, cy);
  oldBitmap = dcOffScreen.SelectObject(bmpOffScreen);

  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

bool CHistogramDlgBar::handleMessage(InMessage& message)
{
  if(message.getMessageID() == idHistogram)
  {
    Histogram histogram;
    message.bin >> histogram;
    histograms[histogram.getHistogramID()] = histogram;
 
    RedrawWindow(NULL, NULL, RDW_INVALIDATE);

    return true;
  }
  return false;
}

void CHistogramDlgBar::OnContextMenu(CWnd* pWnd, CPoint point) 
{
  CMenu menu;
  VERIFY( menu.CreatePopupMenu() );

  int selectedIndex = mouseOverRow * numberOfColumns + mouseOverColumn;
  currentSelection = selectedIndex;
  if(currentSelection >= numberOfHistograms) currentSelection = numberOfHistograms - 1;

  scaleSliderPosition = scaleToScaleSliderPosition(scale[currentSelection]);
  setSliders();
  
  UINT nFlags = 0;
  for(int i = 0; i < Histogram::numberOfHistogramIDs; i++)
  {
    Histogram::HistogramID currentHistogramID = (Histogram::HistogramID)i;
    if(histogramID[selectedIndex] == currentHistogramID) nFlags = MF_CHECKED;
    else nFlags = 0;
    menu.AppendMenu(nFlags, i + 3000, Histogram::getName(currentHistogramID) );
  }
  menu.AppendMenu(MF_SEPARATOR);
  menu.AppendMenu(0,3100,"Export values to csv file");
  menu.AppendMenu(MF_SEPARATOR);
  menu.AppendMenu(showAverage[selectedIndex]?MF_CHECKED:0,3101,"show average");
  menu.AppendMenu(showAverageOfFrequencies[selectedIndex]?MF_CHECKED:0,3102,"show average of frequencies");
  menu.AppendMenu(showClusters[selectedIndex]?MF_CHECKED:0,3103,"show clusters");
  
  // Track menu
  int nID = menu.TrackPopupMenu( TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_RETURNCMD | TPM_NONOTIFY,
    point.x, point.y, this);
  
  if (nID == 3100)
  {
    exportToFileAsCSV();
  }
  else if(nID == 3101) showAverage[selectedIndex] = !showAverage[selectedIndex];
  else if(nID == 3102) showAverageOfFrequencies[selectedIndex] = !showAverageOfFrequencies[selectedIndex];
  else if(nID == 3103) showClusters[selectedIndex] = !showClusters[selectedIndex];
  else if(nID >= 3000 && nID < 3000 + Histogram::numberOfHistogramIDs)
  {
    histogramID[selectedIndex] = (Histogram::HistogramID)(nID - 3000);

    bool histogramIsShown[Histogram::numberOfHistogramIDs];
    memset(histogramIsShown, 0, sizeof(histogramIsShown));
    for(int i = 0; i < maxNumberOfHistograms; i++)
    {
      histogramIsShown[histogramID[i]] = true;
    }
    for(int id = 1; id < Histogram::numberOfHistogramIDs; id++)
    {
      if(histogramIsShown[id])
      {
        getDebugKeyTables().forPhysicalRobots.set(Histogram::getDebugKeyID(
          (Histogram::HistogramID)id),DebugKey::always);
      
        getDebugKeyTables().forSimulatedRobots.set(Histogram::getDebugKeyID(
          (Histogram::HistogramID)id),DebugKey::always);
      }
      else
      {
        getDebugKeyTables().forPhysicalRobots.set(Histogram::getDebugKeyID(
          (Histogram::HistogramID)id),DebugKey::disabled);
      
        getDebugKeyTables().forSimulatedRobots.set(Histogram::getDebugKeyID(
          (Histogram::HistogramID)id),DebugKey::disabled);
      }
    }   
    getDebugKeyTables().sendForSimulatedRobots();
    getDebugKeyTables().sendForPhysicalRobots();
    
    RedrawWindow(NULL, NULL, RDW_INVALIDATE);
  }
}

void CHistogramDlgBar::createBrushes()
{
  crColor[Histogram::imageIntensityY] = RGB(127,255,127);
  crColor[Histogram::imageIntensityU] = RGB(255,63,63);
  crColor[Histogram::imageIntensityV] = RGB(255,127,0);
  //crColor[Histogram::] = RGB(,,);
  //crColor[Histogram::] = RGB(,,);
  
  for(int i = 0; i < Histogram::numberOfHistogramIDs; i++)
  {
    brush[i].CreateSolidBrush(crColor[i]);
  }
}

void CHistogramDlgBar::deleteBrushes()
{
  for(int i = 0; i < Histogram::numberOfHistogramIDs; i++)
  {
    brush[i].DeleteObject();
  }
}

void CHistogramDlgBar::unselectAll()
{
/*  for(int i = 0; i < Stopwatch::numberOfStopwatchEventIDs; i++)
  {
    showTime[i] = false;
    getDebugKeyTables().forPhysicalRobots.set(Stopwatch::getDebugKeyID(
      (Stopwatch::StopwatchEventID)i),
      DebugKey::disabled
      );
    getDebugKeyTables().forSimulatedRobots.set(Stopwatch::getDebugKeyID(
      (Stopwatch::StopwatchEventID)i),
      DebugKey::disabled
      );
  }
  numberOfSelectedTimes = 0;
  */
}

void CHistogramDlgBar::loadSelection()
{
  CString string;
  string = "Histogram\\selections";
  for(int i = 0; i < maxNumberOfHistograms; i++)
  {
    CString key;
    if(i < 10) key.Format("id 0%d", i);
    else key.Format("id %d", i);
    int currentID = AfxGetApp()->GetProfileInt(string, key, 0);
    histogramID[i] = (Histogram::HistogramID)currentID; 

    if(i < 10) key.Format("scale 0%d", i);
    else key.Format("scale %d", i);
    scale[i] = scaleSliderPositionToScale( AfxGetApp()->GetProfileInt(string, key, 51) );
    
    if(i < 10) key.Format("showAverage 0%d", i);
    else key.Format("showAverage %d", i);
    int showAverageInt = AfxGetApp()->GetProfileInt(string, key, 0);
    showAverage[i] = (showAverageInt == 0)?0:1;

    if(i < 10) key.Format("showAverageOfFrequencies 0%d", i);
    else key.Format("showAverageOfFrequencies %d", i);
    int showAverageOfFrequenciesInt = AfxGetApp()->GetProfileInt(string, key, 0);
    showAverageOfFrequencies[i] = (showAverageOfFrequenciesInt == 0)?0:1;

    if(i < 10) key.Format("showClusters 0%d", i);
    else key.Format("showClusters %d", i);
    int showClustersInt = AfxGetApp()->GetProfileInt(string, key, 0);
    showClusters[i] = (showClustersInt == 0)?0:1;
  }
  numberOfHistograms = AfxGetApp()->GetProfileInt("Histogram", "numberOfHistograms", 2);
}

void CHistogramDlgBar::saveCurrentSelection()
{
  CString string;
  string = "Histogram\\selections";
  for(int i = 0; i < maxNumberOfHistograms; i++)
  {
    CString key;
    if(i < 10) key.Format("id 0%d", i);
    else key.Format("id %d", i);
    AfxGetApp()->WriteProfileInt(string, key, histogramID[i]);

    if(i < 10) key.Format("scale 0%d", i);
    else key.Format("scale %d", i);
    AfxGetApp()->WriteProfileInt(
      string, key, 
      scaleToScaleSliderPosition(scale[i]) );

    if(i < 10) key.Format("showAverage 0%d", i);
    else key.Format("showAverage %d", i);
    AfxGetApp()->WriteProfileInt(string, key, showAverage[i]);

    if(i < 10) key.Format("showAverageOfFrequencies 0%d", i);
    else key.Format("showAverageOfFrequencies %d", i);
    AfxGetApp()->WriteProfileInt(string, key, showAverageOfFrequencies[i]);

    if(i < 10) key.Format("showClusters 0%d", i);
    else key.Format("showClusters %d", i);
    AfxGetApp()->WriteProfileInt(string, key, showClusters[i]);
  }
  AfxGetApp()->WriteProfileInt("Histogram", "numberOfHistograms", numberOfHistograms);
}

void CHistogramDlgBar::exportToFileAsCSV()
{
/*
  CString defaultPath = File::getGTDir();
  defaultPath += "/Config/";
  defaultPath.Replace('/','\\');
  CString pathName = 
    AfxGetApp()->GetProfileString("Histogram", "csvWritePath", defaultPath);
  pathName += "*.csv";
  
  CFileDialog fileDialog(false, ".csv",pathName,
    OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
    , "csv files (*.csv)|*.csv||", this);
  
  if (fileDialog.DoModal()==IDOK)
  {
    CString fileName = fileDialog.GetFileName();
    CString pathAndFileName = fileDialog.GetPathName();
    CString pathName = 
      pathAndFileName.Left(
      pathAndFileName.GetLength() -
      fileName.GetLength()
      );
    AfxGetApp()->WriteProfileString("Histogram", "csvWritePath", pathName);
    
    OutTextFile fout(pathAndFileName);
    
    fout << "id,averageTime,minTime,maxTime,frequency" << endl;
    for(int i = 0; i < Stopwatch::numberOfStopwatchEventIDs; i++)
    {
      Stopwatch::StopwatchEventID stopwatchEventID = Stopwatch::StopwatchEventID(i);
      
      if(showTime[stopwatchEventID])
      {
        CString frequency;
        frequency.Format("%.1f",averageFrequency[stopwatchEventID]);
        
        fout << Stopwatch::getStopwatchEventIDName(stopwatchEventID) << "," 
          << averageMeasuredTime[stopwatchEventID] << ","
          << minMeasuredTime[stopwatchEventID] << ","
          << maxMeasuredTime[stopwatchEventID] << ","
          << frequency.GetBuffer(frequency.GetLength())
          << endl;
      }
    }
  }
  */
}

void CHistogramDlgBar::paintHistogram
(
 Histogram& histogram, 
 CRect& rectForPainting, 
 double scale,
 bool showAverage,
 bool showAverageOfFrequencies,
 bool showClusters
 )
{
  int x1, x2, y;
  int x, y1, y2;
  CPen redPen, bluePen, lightBluePen, greenPen, lightGreenPen;
  redPen.CreatePen(PS_SOLID, 2, RGB(255,127,127));
  bluePen.CreatePen(PS_SOLID, 2, RGB(0,0,255));
  lightBluePen.CreatePen(PS_SOLID, 2, RGB(127,127,255));
  greenPen.CreatePen(PS_SOLID, 1, RGB(0,255,0));
  lightGreenPen.CreatePen(PS_SOLID, 1, RGB(127,255,127));
  CPen* oldPen;
  oldPen = dcOffScreen.SelectObject(&redPen);

  //average 
  if(showAverage)
  {
    x = rectForPainting.left + 2 + (int)(histogram.getAverage() * (rectForPainting.right - rectForPainting.left - 4) / (histogram.getNumberOfEntries()));
    y1 = rectForPainting.bottom - 4;
    y2 = rectForPainting.top + 2;
    dcOffScreen.MoveTo(x,y1);
    dcOffScreen.LineTo(x,y2);
  }

  if(showAverageOfFrequencies)
  {
    //average frequency over all
    dcOffScreen.SelectObject(&bluePen);
    x1 = rectForPainting.left + 2;
    x2 = rectForPainting.right - 2;
    y = rectForPainting.bottom - (int)(histogram.getAverageFrequencyOverAllEntries() * scale + 4);
    dcOffScreen.MoveTo(x1,y);
    dcOffScreen.LineTo(x2,y);

    //average frequency over used
    dcOffScreen.SelectObject(&lightBluePen);
    x1 = rectForPainting.left + 2;
    x2 = rectForPainting.right - 2;
    y = rectForPainting.bottom - (int)(histogram.getAverageFrequencyOverUsedEntries() * scale + 4);
    dcOffScreen.MoveTo(x1,y);
    dcOffScreen.LineTo(x2,y);
  }

  //clusters
  if(showClusters)
  {
    histogram.analyseClusters();
    dcOffScreen.SelectObject(&greenPen);
    for(int clusterIndex = 0; clusterIndex < histogram.getNumberOfClusters(); clusterIndex++)
    {
      x1 = rectForPainting.left + 2 + (int)(histogram.getBeginOfCluster(clusterIndex) * (rectForPainting.right - rectForPainting.left - 4) / (histogram.getNumberOfEntries()));
      x2 = rectForPainting.left + 2 + (int)(histogram.getEndOfCluster(clusterIndex) * (rectForPainting.right - rectForPainting.left - 4) / (histogram.getNumberOfEntries()));
      y1= rectForPainting.bottom - 4;
      y2 = rectForPainting.top + 2;
      dcOffScreen.MoveTo(x1,y1);
      dcOffScreen.LineTo(x1,y2);
      dcOffScreen.MoveTo(x2,y1);
      dcOffScreen.LineTo(x2,y2);
      dcOffScreen.MoveTo(x1,y1);
      dcOffScreen.LineTo(x2,y1);
      dcOffScreen.MoveTo(x1,y2);
      dcOffScreen.LineTo(x2,y2);
    }
  }
  dcOffScreen.SelectObject(oldPen);
  redPen.DeleteObject();
  bluePen.DeleteObject();
  lightBluePen.DeleteObject();
  greenPen.DeleteObject();
  lightGreenPen.DeleteObject();

  for(int i = 0; i < histogram.getNumberOfEntries(); i++)
  {
    x = rectForPainting.left + 2 + i * (rectForPainting.right - rectForPainting.left - 4) / (histogram.getNumberOfEntries());
    y1 = rectForPainting.bottom - 4;
    y2 = rectForPainting.bottom - (int)(histogram.getValue(i) * scale + 4);
    if(y2 < rectForPainting.top + 2) y2 = rectForPainting.top + 2;
    dcOffScreen.MoveTo(x,y1);
    dcOffScreen.LineTo(x,y2);
  }
}

void CHistogramDlgBar::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
  scaleSliderPosition = m_scaleSlider.GetPos(); 
  scale[currentSelection] = scaleSliderPositionToScale(scaleSliderPosition);

  numberOfHistograms = m_numberSlider.GetPos();
  if(currentSelection >= numberOfHistograms)
  {
    currentSelection = numberOfHistograms - 1;
    scaleSliderPosition = scaleToScaleSliderPosition(scale[currentSelection]);
  }
  calculateRects();
  setSliders();

  RedrawWindow(NULL, NULL, RDW_INVALIDATE);

  CDynamicBarDlg::OnHScroll(nSBCode, nPos, pScrollBar);
}

void CHistogramDlgBar::setSliders()
{
  m_scaleSlider.SetPos(scaleSliderPosition);
  m_numberSlider.SetPos(numberOfHistograms);

  CString string;
  string.Format("%f", scale[currentSelection]); m_scaleEdit.SetWindowText(string);
  string.Format("%d", numberOfHistograms); m_numberEdit.SetWindowText(string);
}

void CHistogramDlgBar::calculateRects()
{
  double ratio = double(rect.right - rect.left) / double(rect.bottom - rect.top);

  switch(numberOfHistograms)
  {
  case 1:
    numberOfRows = 1; numberOfColumns = 1;
    break;
  case 2:
    if(ratio > 1) {numberOfRows = 1; numberOfColumns = 2;}
    else {numberOfRows = 2; numberOfColumns = 1;}
    break;
  case 3:
    if(ratio > 1.5) {numberOfRows = 1; numberOfColumns = 3;}
    else {numberOfRows = 3; numberOfColumns = 1;}
    break;
  case 4:
    if(ratio > 1.5)        {numberOfRows = 1; numberOfColumns = 4;}
    else if(ratio > 1) {numberOfRows = 2; numberOfColumns = 2;}
    else                 {numberOfRows = 4; numberOfColumns = 1;}
    break;
/*  case 5:
    numberOfRows = 2; numberOfColumns = 3;
    break;
    */
  default:
    numberOfRows = (int)ceil(sqrt((double)numberOfHistograms)); numberOfColumns = (int)ceil(sqrt((double)numberOfHistograms));
    while((numberOfRows - 1) * numberOfColumns >= numberOfHistograms) numberOfRows--;
    break;
  }
}

void CHistogramDlgBar::paintHistograms()
{
//  dcOffScreen.Rectangle(currentRect);
  int selectedIndex = mouseOverRow * numberOfColumns + mouseOverColumn;
  if(selectedIndex >= numberOfHistograms) selectedIndex = numberOfHistograms - 1;

  CPen thickPen, thinPen, blackPen;
  CPen* oldPen;
  thickPen.CreatePen(PS_SOLID, 3, RGB(127,127,127));
  thinPen.CreatePen(PS_SOLID, 1, RGB(200,200,200));

  CBrush whiteBrush;
  CBrush* oldBrush;
  whiteBrush.CreateSolidBrush(RGB(255,255,255));

  oldBrush = dcOffScreen.SelectObject(&whiteBrush);
  oldPen = dcOffScreen.SelectObject(&thinPen);
  
  dcOffScreen.Rectangle(
    histogramRect[selectedIndex].left - 5,
    histogramRect[selectedIndex].top - 5,
    histogramRect[selectedIndex].right + 5,
    histogramRect[selectedIndex].bottom + 5
    );

  dcOffScreen.SelectObject(&thickPen);

  dcOffScreen.Rectangle(
    histogramRect[currentSelection].left - 4,
    histogramRect[currentSelection].top - 4,
    histogramRect[currentSelection].right + 4,
    histogramRect[currentSelection].bottom + 4
    );

  dcOffScreen.SelectObject(oldPen);
  dcOffScreen.SelectObject(oldBrush);
  whiteBrush.DeleteObject();
  thickPen.DeleteObject();
  thinPen.DeleteObject();
  blackPen.DeleteObject();

  for(int i = 0; i < numberOfHistograms; i++)
  {
    dcOffScreen.Rectangle(histogramRect[i]);
  }

  int width = (currentRect.right-currentRect.left) / numberOfColumns;
  int height = (currentRect.bottom-currentRect.top) / numberOfRows;

  int currentRectIndex = 0;
  for(int row = 0; row < numberOfRows; row++)
  {
    for(int column = 0; column < numberOfColumns; column++)
    {
      if(currentRectIndex < numberOfHistograms)
      {
        histogramRect[currentRectIndex].left   = width * column + 5;
        histogramRect[currentRectIndex].right  = width * (column + 1) - 5;
        histogramRect[currentRectIndex].top    = height * row + 5;
        histogramRect[currentRectIndex].bottom = height * (row + 1) - 5;

        paintHistogram(
          histograms[histogramID[currentRectIndex]], histogramRect[currentRectIndex], 
          scale[currentRectIndex],
          showAverage[currentRectIndex],
          showAverageOfFrequencies[currentRectIndex],
          showClusters[currentRectIndex]
          );
      }
      currentRectIndex++;
    }
  }
}

void CHistogramDlgBar::OnMouseMove(UINT nFlags, CPoint point) 
{
  if ((currentRect.bottom - currentRect.top>0)&&(currentRect.right - currentRect.left > 0))
  {
    mouseOverRow = (point.y - 24) * numberOfRows / (currentRect.bottom - currentRect.top); 
    mouseOverColumn = (point.x) * numberOfColumns / (currentRect.right - currentRect.left);

    RedrawWindow(NULL, NULL, RDW_INVALIDATE);
  }

  CDynamicBarDlg::OnMouseMove(nFlags, point);
}

void CHistogramDlgBar::OnLButtonDown(UINT nFlags, CPoint point) 
{
  currentSelection = mouseOverRow * numberOfColumns + mouseOverColumn;
  if(currentSelection >= numberOfHistograms) currentSelection = numberOfHistograms - 1;
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);

  scaleSliderPosition = scaleToScaleSliderPosition(scale[currentSelection]);
  setSliders();

	CDynamicBarDlg::OnLButtonDown(nFlags, point);
}

double CHistogramDlgBar::scaleSliderPositionToScale(int scaleSliderPosition)
{
  if(scaleSliderPosition > 50) return scaleSliderPosition - 50;
  else return 1.0 / double(52 - scaleSliderPosition);
}

int CHistogramDlgBar::scaleToScaleSliderPosition(double scale)
{
  if(scale < 1) return (int)(-(1.0 / scale) + 52);
  else return (int)scale + 50;
}


/*
* Change log :
* 
* $Log: HistogramDlgBar.cpp,v $
* Revision 1.3  2003/11/30 01:53:19  loetzsch
* prepared RobotControl port to Visual C++ .Net
*
* Revision 1.2  2003/11/18 15:45:37  dueffert
* possible crash fixed
*
* Revision 1.1  2003/10/07 10:09:37  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/26 11:40:12  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.2  2003/08/29 13:10:04  juengel
* Pens and brushes are deleted now.
*
* Revision 1.1  2003/08/25 17:19:54  juengel
* Added Histogram
*
*/
