/**
* @file JointViewerDlgBar.h
*
* Declaration of class CJointViewerDlgBar.
*
* @author Martin Ltzsch
*/
#ifndef __JointViewerDlgBar_h_
#define __JointViewerDlgBar_h_

#include "StdAfx.h"
#include "RobotControlDialogBar.h"

#include "Representations/Perception/SensorData.h"
#include "Representations/Motion/JointData.h"


/**
* A docking window for the joint viewer
*
* @author Martin Ltzsch
*/
class CJointViewerDlgBar : public CRobotControlDialogBar
{
private:
  /** groups of sensor data ids */
  enum sensorDataGroup
  {
    sdHead,sdLegFL,sdLegHL,sdLegFR,sdLegHR,sdAcceleration,sdSwitches,sdPaws,
      sdTail,sdPsd,sdThermo,sdMouth,sdChin,
      numOfSensorDataGroups
  };

  /** returns the name of a sensor data group */
  char* getSensorDataGroupName(sensorDataGroup group);

  /** returns the group to which a sensor data id belongs to */
  sensorDataGroup getSensorDataGroup(SensorData::sensors sensor);

  /** determines if sensor data of that group are displayed */
  bool viewSensorDataGroup[numOfSensorDataGroups];

  /** returns a vertical scale factor for every sensor */
  double getSensorDataScaleFactor(SensorData::sensors sensor);

  /** groups of joint data ids */
  enum jointDataGroup
  {    
    jdHead, jdLegFR, jdLegFL, jdLegHR, jdLegHL, jdTail, jdMouth, jdEars,
    numOfJointDataGroups
  };

  /** returns the name of a joint data group */
  char* getJointDataGroupName(jointDataGroup group);

  /** returns the group to which a joint id belongs to */
  jointDataGroup getJointDataGroup(JointData::JointID joint);

  /** determines if a joint data group is displayed */
  bool viewJointDataGroup[numOfJointDataGroups];

  /** returns a vertical scale factor for every joint */
  double getJointDataScaleFactor(JointData::JointID joint);
public:
  /**
  * Default constructor.
  */
  CJointViewerDlgBar();

  ~CJointViewerDlgBar();

  /** 
  * Called from a MessageQueue to distribute messages 
  * @param message The message that can be read.
  * @return true if the message was read (handled).
  */
  virtual bool handleMessage(InMessage& message);

  /** 
  * Enables the controls in the toolbar. 
  * This function is called from the main window for all controls. 
  * @param pCmdUI An interface to the control that allows enabling/disabling, checking etc.
  */
  virtual void updateUI(CCmdUI* pCmdUI);

  /** determines if the horizontal scroll bar should be enabled */
  bool bHScrollEnabled;

private:
  /** deletes all stored data */
  void clear();

  /** saves the data to a CSV (comma separated values) file (merged time line)*/
  void saveAsCSV();

  /** saves the frequency spectrum (Fourier Transform) of
  the last (FourierCoefficient::lengthOfPeriod) joint datas*/
  void saveFTSpectrum();

    /** saves all data to a CSV (comma separated values) file (seperate time lines for 
  * joints and sensors */
  void saveAllCSV();

  /** the array, where incoming sensor data are stored */
  CArray<SensorData,SensorData> sensorDataArray;

  /** the time for every sensor datum */
  CArray<unsigned long,unsigned long> sensorDataTimeStampsArray;

  /** the array, where incoming joint data are stored */
  CArray<JointData,JointData> jointDataArray;

  /** the time for every joint datum */
  CArray<unsigned long,unsigned long> jointDataTimeStampsArray;

  /** the time of the last stored element */
  unsigned long lastTime;

  /** the time of the first stored element */
  unsigned long firstTime;

  /** the time of the first datum that is displayed in the window */
  unsigned long firstDisplayedTime;

  /** the time of the last datum that is displayed in the window */
  unsigned long lastDisplayedTime;

  /** the time for that the values are shown */
  unsigned long viewedTime;

  /** offset for calculating timestamps from sensor date frame numbers */
  long sensorTimeOffset;

  /** the number of stored sensor data */
  int numOfSensorData;

  /** the number of stored joint data */
  int numOfJointData;

  /** determines the horizontal resolution of the graph */
  int pixelsPerSecond;

  //{{AFX_DATA(CMessageViewerDlgBar)
  enum { IDD = IDD_DIALOG_BAR_JOINT_VIEWER};
  CScrollBar m_vScrollBar;
  CScrollBar m_hScrollBar;
  //}}AFX_DATA

  /** updates the scroll range and position of the horizontal scroll bar */
  void updateHScroll();

  //{{AFX_VIRTUAL(CMotionTesterDlgBar)
protected:
  virtual void DoDataExchange(CDataExchange* pDX);  
  virtual BOOL OnInitDialog();
  //}}AFX_VIRTUAL

public:
  //{{AFX_MSG(CJointViewerDlgBar)
	afx_msg void OnPaint();
  afx_msg void OnSize(UINT nType, int cx, int cy);
  afx_msg void OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar);
  afx_msg void OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar);
	afx_msg void OnContextMenu(CWnd* pWnd, CPoint point);
	//}}AFX_MSG
  DECLARE_MESSAGE_MAP();
    
private:
  /** the rect, where the scope is painted */
  CRect scopeRect;

  /** the rect, where the inscription is painted */
  CRect inscriptionRect;

  /** */
  void calculateJointDataPeriodTime(JointData jointData, unsigned long time);

  /** */
  unsigned long jointDataPeriodTime;
  
  /** */
  long jointValue[3];
  
  /** */
  unsigned long lastOccurrenceOfJointValue0Time;
};


#endif //__JointViewerDlgBar_h_

/*
 * Change log :
 * $Log: JointViewerDlgBar.h,v $
 * Revision 1.3  2003/12/06 06:31:20  loetzsch
 * no message
 *
 * Revision 1.2  2003/11/30 01:53:19  loetzsch
 * prepared RobotControl port to Visual C++ .Net
 *
 * Revision 1.1  2003/10/07 10:09:38  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.3  2003/09/30 10:51:12  dueffert
 * typos fixed
 *
 * Revision 1.2  2003/09/26 15:28:23  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2003/05/11 23:46:33  dueffert
 * Depend now works with RobotControl too
 *
 * Revision 1.5  2003/05/05 14:47:57  risler
 * idJointData debug message sends JointDataBuffer
 * JointViewerDlg shows reference values
 *
 * Revision 1.4  2002/11/26 14:18:08  juengel
 * Period detection for joint viewer added.
 *
 * Revision 1.3  2002/11/19 17:14:14  risler
 * coding conventions: renamed JointData::joint to JointID, GetName to getName
 *
 * Revision 1.2  2002/09/22 18:40:51  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.1  2002/09/10 15:49:05  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 *
 * Revision 1.4  2002/08/30 17:13:56  dueffert
 * doxygen parser bugs fixed, docu added
 *
 * Revision 1.3  2002/08/29 14:39:57  dueffert
 * includes in correct case, system includes in <>
 *
 * Revision 1.2  2002/07/23 13:43:36  loetzsch
 * - new streaming classes
 * - removed many #include statements
 * - 5 instead of 3 debug queues in RobotControl
 * - exchanged StaticQueue with MessageQueue
 * - new debug message handling
 * - empty constructors in bars / dialogs
 * - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
 * - general clean up
 *
 * Revision 1.1.1.1  2002/05/10 12:40:21  cvsadm
 * Moved GT2002 Project from ute to tamara.
 *
 * Revision 1.14  2002/05/02 12:12:33  kallnik
 * GTMath
 *
 * Revision 1.13  2002/04/25 14:50:34  kallnik
 * changed double/float to double
 * added several #include GTMath
 *
 * PLEASE use double
 *
 * Revision 1.12  2002/04/23 17:45:16  loetzsch
 * - splitted debugKeyTable into debugKeyTableForRobot and debugKeyTableForLocalProcesses
 * - removed Modules Toolbar
 * - removed access for dialogs and toolbars to solutionRequest.
 * - changed access for dialogs and toolbars to debug queues
 * - removed the instance of SolutionRequest in CRobotControlApp
 * - only the log player, local processes and the robot put messages into queueToRobotControl
 *
 * Revision 1.11  2002/03/19 16:45:12  jhoffman
 * added fourier spectrum output
 *
 * Revision 1.10  2002/02/21 12:35:11  jhoffman
 * _ added different CSV export that is more suitable for use in Excel (based on Martin's code)
 * _ in doing so found that existing CSV output gets mixed up with time stamps and therefore needs to be debugged
 *
 * Revision 1.9  2002/02/12 16:34:49  risler
 * finished MofTester
 *
 * Revision 1.8  2002/02/09 15:12:29  juengel
 * change log added.
 *
 *
 */
