/**
* @file KickEditorDlgBar.cpp
*
* Implementation of class CKickEditorDlgBar
*
* @author Matthias Jngel
*/

#include "StdAfx.h"
#include "KickEditorDlgBar.h"

#include "Visualization/PaintMethodsWin32.h"
#include "RobotControlQueues.h"

CKickEditorDlgBar::CKickEditorDlgBar()
  : CRobotControlDialogBar(IDD)
{
  //{{AFX_DATA_INIT(CKickEditorDlgBar)
  //}}AFX_DATA_INIT
  m_LeftButtonDown = false;
  highlightedSector = 0;
  selectedSector = 0;
  selectedSectorWhenCopied = 0;
}

CKickEditorDlgBar::~CKickEditorDlgBar()
{
}

BEGIN_MESSAGE_MAP(CKickEditorDlgBar, CDynamicBarDlg)
  //{{AFX_MSG_MAP(CKickEditorDlgBar)
	ON_WM_PAINT()
	ON_WM_SIZE()
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_WM_CONTEXTMENU()
	ON_BN_CLICKED(IDC_KICK_EDITOR_SAVE, OnSave)
	ON_CBN_SELCHANGE(IDC_KICK_EDITOR_ACTION_COMBO, OnSelchangeActionCombo)
	ON_CBN_SELCHANGE(IDC_KICK_EDITOR_TABLE_COMBO, OnSelchangeTableCombo)
	ON_BN_CLICKED(IDC_KICK_EDITOR_OPEN, OnOpen)
	ON_WM_LBUTTONDBLCLK()
	ON_BN_CLICKED(IDC_KICK_EDITOR_SEND, OnSend)
	ON_WM_HSCROLL()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

void CKickEditorDlgBar::DoDataExchange(CDataExchange* pDX)
{
  CDialog::DoDataExchange(pDX);
  //{{AFX_DATA_MAP(CKickEditorDlgBar)
	DDX_Control(pDX, IDC_KICK_EDITOR_SIZE_SLIDER, m_sizeSlider);
	DDX_Control(pDX, IDC_KICK_EDITOR_TABLE_COMBO, m_tableCombo);
	DDX_Control(pDX, IDC_KICK_EDITOR_ACTION_COMBO, m_actionCombo);
	//}}AFX_DATA_MAP
}

BOOL CKickEditorDlgBar::OnInitDialog() 
{
  CDynamicBarDlg::OnInitDialog();

  loadCaseBaseFromFile("kick-selection-case-base.cfg");
  showCoordinateSystem = true;
  showRobot = true;
  showKickCases = false;

//  horizontalOffset = 88;
  horizontalOffset = 0;
  verticalOffset = 22;

  frameSize = 30;

  int i;
  for (i = 0; i < KickSelectionTable::numberOfKickSelectionTableIDs; i++)
    m_tableCombo.AddString(KickSelectionTable::getKickSelectionTableIDName(
    (KickSelectionTable::KickSelectionTableID)i));
  
  currentTable = KickSelectionTable::inCenterOfField;
  m_tableCombo.SetCurSel(0);

  for (i = 0; i < KickSelectionTable::numberOfActions; i++)
    m_actionCombo.AddString(KickSelectionTable::getActionName(
    (KickSelectionTable::ActionID)i));

  currentAction = KickSelectionTable::nothing;
  m_actionCombo.SetCurSel(0);
  

  
  paintStyles[KickSelectionTable::nothing].penColor = RGB(0,0,0);
  paintStyles[KickSelectionTable::nothing].brushColor = RGB(0,0,0);
  paintStyles[KickSelectionTable::nothing].shape = PaintMethodsWin32::PaintStyle::horizontalLine;

  paintStyles[KickSelectionTable::armKickLeft].penColor = RGB(255,128,0);
  paintStyles[KickSelectionTable::armKickLeft].brushColor = RGB(255,128,0);
  paintStyles[KickSelectionTable::armKickLeft].shape = PaintMethodsWin32::PaintStyle::filledSquare;

  paintStyles[KickSelectionTable::armKickRight].penColor = RGB(255,255,0);
  paintStyles[KickSelectionTable::armKickRight].brushColor = RGB(255,255,0);
  paintStyles[KickSelectionTable::armKickRight].shape = PaintMethodsWin32::PaintStyle::filledSquare;

  paintStyles[KickSelectionTable::athErs7ChestKick2].penColor = RGB(255,0,0);
  paintStyles[KickSelectionTable::athErs7ChestKick2].brushColor = RGB(255,0,0);
  paintStyles[KickSelectionTable::athErs7ChestKick2].shape = PaintMethodsWin32::PaintStyle::filledSquare;
 
  paintStyles[KickSelectionTable::athErs7fastDiveKick].penColor = RGB(128,255,0);
  paintStyles[KickSelectionTable::athErs7fastDiveKick].brushColor = RGB(128,255,0);
  paintStyles[KickSelectionTable::athErs7fastDiveKick].shape = PaintMethodsWin32::PaintStyle::filledSquare;

  paintStyles[KickSelectionTable::athErs7KickForwardWithLeftPaw].penColor = RGB(0,128,255);
  paintStyles[KickSelectionTable::athErs7KickForwardWithLeftPaw].brushColor = RGB(0,128,255);
  paintStyles[KickSelectionTable::athErs7KickForwardWithLeftPaw].shape = PaintMethodsWin32::PaintStyle::filledSquare;

  paintStyles[KickSelectionTable::athErs7KickForwardWithRightPaw].penColor = RGB(128,128,255);
  paintStyles[KickSelectionTable::athErs7KickForwardWithRightPaw].brushColor = RGB(128,128,255);
  paintStyles[KickSelectionTable::athErs7KickForwardWithRightPaw].shape = PaintMethodsWin32::PaintStyle::filledSquare;

  paintStyles[KickSelectionTable::athErs7ForwardLeapingKick].penColor = RGB(255,0,128);
  paintStyles[KickSelectionTable::athErs7ForwardLeapingKick].brushColor = RGB(255,0,128);
  paintStyles[KickSelectionTable::athErs7ForwardLeapingKick].shape = PaintMethodsWin32::PaintStyle::filledSquare;

  paintStyles[KickSelectionTable::athErs7HeadKickLeft].penColor = RGB(128,128,0);
  paintStyles[KickSelectionTable::athErs7HeadKickLeft].brushColor = RGB(128,128,0);
  paintStyles[KickSelectionTable::athErs7HeadKickLeft].shape = PaintMethodsWin32::PaintStyle::filledSquare;
 
  paintStyles[KickSelectionTable::athErs7HeadKickRight].penColor = RGB(128,0,255);
  paintStyles[KickSelectionTable::athErs7HeadKickRight].brushColor = RGB(128,0,255);
  paintStyles[KickSelectionTable::athErs7HeadKickRight].shape = PaintMethodsWin32::PaintStyle::filledSquare;

  paintStyles[KickSelectionTable::athErs7KickForwardWithRightPaw].penColor = RGB(0,0,0);
  paintStyles[KickSelectionTable::athErs7KickForwardWithRightPaw].brushColor = RGB(0,0,0);
  paintStyles[KickSelectionTable::athErs7KickForwardWithRightPaw].shape = PaintMethodsWin32::PaintStyle::filledCircle;

  paintStyles[KickSelectionTable::athErs7SideHeadKickL].penColor = RGB(255,0,0);
  paintStyles[KickSelectionTable::athErs7SideHeadKickL].brushColor = RGB(255,0,0);
  paintStyles[KickSelectionTable::athErs7SideHeadKickL].shape = PaintMethodsWin32::PaintStyle::filledCircle;
 
  paintStyles[KickSelectionTable::athErs7SideHeadKickR].penColor = RGB(128,255,0);
  paintStyles[KickSelectionTable::athErs7SideHeadKickR].brushColor = RGB(128,255,0);
  paintStyles[KickSelectionTable::athErs7SideHeadKickR].shape = PaintMethodsWin32::PaintStyle::filledCircle;

  paintStyles[KickSelectionTable::kickWithHeadRight].penColor = RGB(128,128,0);
  paintStyles[KickSelectionTable::kickWithHeadRight].brushColor = RGB(128,128,0);
  paintStyles[KickSelectionTable::kickWithHeadRight].shape = PaintMethodsWin32::PaintStyle::filledCircle;
 
  paintStyles[KickSelectionTable::kickWithHeadLeft].penColor = RGB(128,0,255);
  paintStyles[KickSelectionTable::kickWithHeadLeft].brushColor = RGB(128,0,255);
  paintStyles[KickSelectionTable::kickWithHeadLeft].shape = PaintMethodsWin32::PaintStyle::filledCircle;


  
  
  paintStyles[KickSelectionTable::msh7AnyLeft].penColor = RGB(255,0,0);
  paintStyles[KickSelectionTable::msh7AnyLeft].penWidth = 2;
  paintStyles[KickSelectionTable::msh7AnyLeft].shape = PaintMethodsWin32::PaintStyle::circle;

  paintStyles[KickSelectionTable::msh7AnyRight].penColor = RGB(0,255,0);
  paintStyles[KickSelectionTable::msh7AnyRight].penWidth = 2;
  paintStyles[KickSelectionTable::msh7AnyRight].shape = PaintMethodsWin32::PaintStyle::circle;

  paintStyles[KickSelectionTable::msh7LeftHook].penColor = RGB(0,0,255);
  paintStyles[KickSelectionTable::msh7LeftHook].penWidth = 2;
  paintStyles[KickSelectionTable::msh7LeftHook].shape = PaintMethodsWin32::PaintStyle::circle;

  paintStyles[KickSelectionTable::msh7RightHook].penColor = RGB(255,255,0);
  paintStyles[KickSelectionTable::msh7RightHook].penWidth = 2;
  paintStyles[KickSelectionTable::msh7RightHook].shape = PaintMethodsWin32::PaintStyle::circle;

  paintStyles[KickSelectionTable::msh7ForwardLeft].penColor = RGB(0,255,255);
  paintStyles[KickSelectionTable::msh7ForwardLeft].penWidth = 2;
  paintStyles[KickSelectionTable::msh7ForwardLeft].shape = PaintMethodsWin32::PaintStyle::circle;

  paintStyles[KickSelectionTable::msh7ForwardRight].penColor = RGB(255,0,255);
  paintStyles[KickSelectionTable::msh7ForwardRight].penWidth = 2;
  paintStyles[KickSelectionTable::msh7ForwardRight].shape = PaintMethodsWin32::PaintStyle::circle;

  // Legend
  columnsCount = 3;
  columnWidth = 200;
  actionsPerColumn = (KickSelectionTable::numberOfActions + 1) / columnsCount;

  setSliders();

  return TRUE;
}

void CKickEditorDlgBar::loadCaseBaseFromFile(const char* fileName)
{
  InTextFile f(fileName);
  if (f.exists())
  {
    int numberOfRecords;
    f >> numberOfRecords;

    for (int i=0; i<numberOfRecords; i++)
    {
      char kick[200];
      KickCase c;
      f >> kick >> c.ballXR >> c.ballYR >> c.ballTR >> c.ballXS >> c.ballYS >> c.ballTS >> c.ballXA >> c.ballYA >> c.tA;
      KickSelectionTable::ActionID id=getActionIDFromSpecialActionID(MotionRequest::getSpecialActionIDFromName(kick));
      caseBase[id].Add(c);
    }
  }
}

void CKickEditorDlgBar::OnPaint() 
{
	CPaintDC dc(this); // device context for painting

  currentWidth = (rect.right - rect.left - horizontalOffset);

  currentHeight = (rect.bottom - rect.top - verticalOffset);
  CDC dcOffScreen; //an in memory device context
  CBitmap bmpOffScreen;
  CBitmap* oldBitmap;
  
  dcOffScreen.CreateCompatibleDC(&dc);
  bmpOffScreen.CreateCompatibleBitmap(&dc, currentWidth, currentHeight);
  oldBitmap = dcOffScreen.SelectObject(&bmpOffScreen);

  //Generate white background
  CPen noPen;
  CPen* oldPen;
  noPen.CreatePen(PS_NULL, 0 , RGB(0,0,0));
  oldPen = dcOffScreen.SelectObject(&noPen);
  dcOffScreen.Rectangle(0, 0, currentWidth + 1, currentHeight + 1);
  dcOffScreen.SelectObject(oldPen);

  dcOffScreen.SetWindowOrg(
    -currentWidth / 2,
    -currentHeight / 4 * 3);

  double rotationAngle = -pi_2;
  //transformation matrix for turning
  XFORM xform;
  xform.eM11 = (FLOAT)cos(rotationAngle);
  xform.eM12 = (FLOAT)sin(rotationAngle);
  xform.eM21 = (FLOAT)sin(rotationAngle);
  xform.eM22 = (FLOAT)-cos(rotationAngle);
  xform.eDx = 0.0;
  xform.eDy = 0.0;
  SetGraphicsMode(dcOffScreen.m_hDC, GM_ADVANCED); //needed for SetWorldTransform
  SetWorldTransform(dcOffScreen.m_hDC, &xform); 


  paintCurrentSelectionToCDC(dcOffScreen);
//  dcOffScreen.MoveTo(0,0);
//  dcOffScreen.LineTo(highlightedX, highlightedY);

  xform.eM11 = 1.0;xform.eM12 = 0.0;xform.eM21 = 0.0;xform.eM22 = 1.0;
  SetWorldTransform(dcOffScreen.m_hDC, &xform);

  dcOffScreen.SetWindowOrg(0,0);

  paintLegend(dcOffScreen);

  //copy the off screen dc to the dc for painting
  dc.BitBlt(rect.left + horizontalOffset, rect.top + verticalOffset, currentWidth, currentHeight,
    &dcOffScreen, 0, 0, SRCCOPY);

  dcOffScreen.SelectObject(oldBitmap);
  dcOffScreen.DeleteDC();
}

void CKickEditorDlgBar::paintCurrentSelectionToCDC(CDC& dc, bool paintFrame)
{
  if(paintFrame) PaintMethodsWin32::paintFrameForScalingToCDC(dc, scale);
  paintKickSelectionTableToCDC(dc, scale, kickSelectionTable, currentTable, selectedSector);
  if(showKickCases) paintCaseBase(dc, scale, currentAction);
  if(showCoordinateSystem) PaintMethodsWin32::paintCoordinateSystemForKickSelectionToCDC(dc, scale);
  if(showRobot) PaintMethodsWin32::paintRobotToCDC(dc, scale);
  PaintMethodsWin32::paintKickSegmentsToCDC(
    dc,
    currentWidth,
    currentHeight, 
    KickSelectionTable::numberOfSectors, 
    highlightedSector, 
    selectedSector,
    frameSize);
}


void CKickEditorDlgBar::OnConfigurationLoad(CString sectionName)
{
  showCoordinateSystem = AfxGetApp()->GetProfileInt(sectionName + "\\KickEditor", "showCoordinateSystem",1)==1;
  showRobot            = AfxGetApp()->GetProfileInt(sectionName + "\\KickEditor", "showRobot",1)==1;
  showKickCases        = AfxGetApp()->GetProfileInt(sectionName + "\\KickEditor", "showKickCases",1)==1;

  scale = AfxGetApp()->GetProfileInt(sectionName + "\\KickEditor", "scale",1000000 + 200);
  scale -= 1000000; scale /= 1000.0;

  InvalidateRect(NULL, false);
}


void CKickEditorDlgBar::OnConfigurationSave(CString sectionName)
{
  AfxGetApp()->WriteProfileInt(sectionName + "\\KickEditor", "showCoordinateSystem",showCoordinateSystem?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\KickEditor", "showRobot",showRobot?1:0);
  AfxGetApp()->WriteProfileInt(sectionName + "\\KickEditor", "showKickCases",showKickCases?1:0);
  
  AfxGetApp()->WriteProfileInt(sectionName + "\\KickEditor", "scale", (int)(scale * 1000 + 1000000));
}

void CKickEditorDlgBar::copyToClipBoard()
{
  HENHMETAFILE metaFileHandle;
  CMetaFileDC* metaFileDC;
  metaFileDC = new CMetaFileDC();

  metaFileDC->CreateEnhanced(NULL, NULL, NULL, NULL);

  metaFileDC->SetWindowOrg(0,0);

  double rotationAngle = -pi_2;
  //transformation matrix for turning
  XFORM xform;
  xform.eM11 = (FLOAT)cos(rotationAngle);
  xform.eM12 = (FLOAT)sin(rotationAngle);
  xform.eM21 = (FLOAT)sin(rotationAngle);
  xform.eM22 = (FLOAT)-cos(rotationAngle);
  xform.eDx = 0.0;
  xform.eDy = 0.0;
  SetGraphicsMode(metaFileDC->m_hDC, GM_ADVANCED); //needed for SetWorldTransform
  SetWorldTransform(metaFileDC->m_hDC, &xform); 


  double oldScale = scale;
  scale = 0.2;
  paintCurrentSelectionToCDC(*metaFileDC, true);
  scale = oldScale;

  xform.eM11 = 1.0;xform.eM12 = 0.0;xform.eM21 = 0.0;xform.eM22 = 1.0;
  SetWorldTransform(metaFileDC->m_hDC, &xform);

  metaFileHandle = metaFileDC->CloseEnhanced();
  
  OpenClipboard();
	EmptyClipboard();
	::SetClipboardData(CF_ENHMETAFILE,metaFileHandle);
	CloseClipboard();

  DeleteEnhMetaFile(metaFileHandle);
  metaFileDC->DeleteDC();
  delete metaFileDC;
}

void CKickEditorDlgBar::OnSize(UINT nType, int cx, int cy) 
{
  rect.left = 0;
  rect.right = cx;
  rect.top = 0;
  rect.bottom = cy;
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
}

void CKickEditorDlgBar::OnLButtonDown(UINT nFlags, CPoint point) 
{
  m_LeftButtonDown = true;
  m_LeftDownPos = point;
  selectedSector = highlightedSector;

  highlightedX = - point.y + verticalOffset + currentHeight / 4 * 3;
  highlightedY = - point.x + horizontalOffset + currentWidth / 2;
  // is the mouse over the grid for the ball positions?
  int x,y;
  x = (int)(1 + highlightedX / 10.0 / scale) - 1;
  y = (int)(1 + highlightedY / 10.0 / scale + KickSelectionTable::yRange / 2) -1;
  if(x >= 0 && x < KickSelectionTable::xRange &&
     y >= 0 && y < KickSelectionTable::yRange )
  {
    oldAction = kickSelectionTable.action[x][y][selectedSector][currentTable];
    kickSelectionTable.action[x][y][selectedSector][currentTable] = currentAction;
  }
  
  //Aldi
  changeKickPerLegend(point);


  RedrawWindow(NULL, NULL, RDW_INVALIDATE);
  CRobotControlDialogBar::OnLButtonDown(nFlags, point);
}

void CKickEditorDlgBar::OnLButtonUp(UINT nFlags, CPoint point) 
{
  m_LeftButtonDown = false;
  CRobotControlDialogBar::OnLButtonUp(nFlags, point);
}

void CKickEditorDlgBar::OnMouseMove(UINT nFlags, CPoint point) 
{
  highlightedX = - point.y + verticalOffset + currentHeight / 4 * 3;
  highlightedY = - point.x + horizontalOffset + currentWidth / 2;
  if(m_LeftButtonDown)
  {
    // is the mouse over the grid for the ball positions?
    int x,y;
    x = (int)(1 + highlightedX / 10.0 / scale) - 1;
    y = (int)(1 + highlightedY / 10.0 / scale + KickSelectionTable::yRange / 2) -1;
    if(x >= 0 && x < KickSelectionTable::xRange &&
       y >= 0 && y < KickSelectionTable::yRange )
    {
      kickSelectionTable.action[x][y][selectedSector][currentTable] = currentAction;
    }
    // else resize
    else
    {
      /*
      CSize diff = m_LeftDownPos - point;
      m_LeftDownPos = point;
      for(int i = 0; i < abs(diff.cx); i++)
      {
        if(diff.cx >= 0) scale /= 1.1;
        else scale *= 1.1;
      }
      if(scale < 0.05) scale = 0.05;
      if(scale > 25) scale = 25;
      */
    }
  }
  else
  {
    double angle = atan2((double)highlightedY, (double)highlightedX);
    int distance = (int)sqrt(sqr((double)highlightedY) + sqr((double)highlightedX));
    if(
      point.x - horizontalOffset < frameSize || point.x > rect.right - frameSize ||
      point.y - verticalOffset < frameSize || point.y > rect.bottom - frameSize)
      highlightedSector = (int)((normalize(angle + pi + pi2 / KickSelectionTable::numberOfSectors / 2) + pi) / pi2 * KickSelectionTable::numberOfSectors);
    else highlightedSector = selectedSector;
  }
  InvalidateRect(NULL,FALSE);
  CRobotControlDialogBar::OnMouseMove(nFlags, point);
}

void CKickEditorDlgBar::OnContextMenu(CWnd* pWnd, CPoint point) 
{
  CMenu menu;
  VERIFY( menu.CreatePopupMenu() );

  UINT flags = 0;

  if(showCoordinateSystem) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,1900,"coordinate system");
  
  if(showRobot) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,1901,"robot");

  if(showKickCases) flags = MF_CHECKED; else flags = 0;
  menu.AppendMenu(flags,1902,"showKickCases");

  menu.AppendMenu(MF_SEPARATOR );

  flags = 0;
  menu.AppendMenu(flags,9101,"copy");
  menu.AppendMenu(flags,9102,"pasteForCurrentSector");
  menu.AppendMenu(flags,9103,"pasteForAllSectors");
  menu.AppendMenu(flags,9104,"mirrorForCurrentSector");
  menu.AppendMenu(flags,9105,"mirrorForAllSectors");

  menu.AppendMenu(MF_SEPARATOR );

  flags = 0;
  menu.AppendMenu(flags,9100,"copy as wmf to clipboard");

  UINT nID = menu.TrackPopupMenu( TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_RETURNCMD | TPM_NONOTIFY,
    point.x, point.y, this );
  
  switch(nID)
  {
  case 1900:
    showCoordinateSystem = !showCoordinateSystem;
    break;
  case 1901:
    showRobot = !showRobot;
    break;
  case 1902:
    showKickCases = !showKickCases;
    break;
  case 9100:
    copyToClipBoard();
    break;
  case 9101:
    copy();
    break;
  case 9102:
    pasteForCurrentSector();
    break;
  case 9103:
    pasteForAllSectors();
    break;
  case 9104:
    mirrorForCurrentSector();
    break;
  case 9105:
    mirrorForAllSectors();
    break;
  }

  InvalidateRect(NULL, false);
}

void CKickEditorDlgBar::paintKickSelectionTableToCDC
(
 CDC& dc, 
 double scale,
 const KickSelectionTable& kickSelectionTable,
 KickSelectionTable::KickSelectionTableID kickSelectionTableID,
 int sector
 )
{
  for(int x = 0; x < KickSelectionTable::xRange; x++)
  {
    for(int y = 0; y < KickSelectionTable::yRange; y++)
    {
      KickSelectionTable::ActionID actionID = kickSelectionTable.action[x][y][sector][kickSelectionTableID];
      if(actionID != KickSelectionTable::nothing)
        paintDot(dc, scale, (int)((x+0.5)*10*scale),(int)((y-KickSelectionTable::yRange/2+0.5)*10*scale), paintStyles[actionID]);
    }
  }
}

void CKickEditorDlgBar::paintDot
(
  CDC& dc, 
  double scale,
  int x, int y, 
  PaintMethodsWin32::PaintStyle style)
{
  CPen pen;
  pen.CreatePen(PS_SOLID, style.penWidth, style.penColor);
  CPen* oldPen;
  oldPen = dc.SelectObject(&pen);

  CBrush brush;
  brush.CreateSolidBrush(style.brushColor);
  CBrush* oldBrush;

  int size = (int)(4 * scale);

  switch(style.shape)
  {
  case PaintMethodsWin32::PaintStyle::horizontalLine:
    dc.MoveTo(x-size,y);
    dc.LineTo(x+size,y);
    break;
  case PaintMethodsWin32::PaintStyle::circle: 
    dc.Ellipse(x - size,y - size, x+size, y+size);
    break;
  case PaintMethodsWin32::PaintStyle::filledCircle:
    oldBrush = dc.SelectObject(&brush);
    dc.Ellipse(x - size,y - size, x+size, y+size);
    dc.SelectObject(oldBrush);
    break;
  case PaintMethodsWin32::PaintStyle::square: 
    dc.MoveTo(x-size,y-size);
    dc.LineTo(x-size,y+size);
    dc.LineTo(x+size,y+size);
    dc.LineTo(x+size,y-size);
    dc.LineTo(x-size,y-size);
    break;
  case PaintMethodsWin32::PaintStyle::filledSquare:
    oldBrush = dc.SelectObject(&brush);
    dc.Rectangle(x - size,y - size, x+size, y+size);
    dc.SelectObject(oldBrush);
    break;
  case PaintMethodsWin32::PaintStyle::vcross:
    dc.MoveTo(x-size,y-size);
    dc.LineTo(x+size,y+size);
    dc.MoveTo(x-size,y+size);
    dc.LineTo(x+size,y-size);
    break;
  case PaintMethodsWin32::PaintStyle::cross:
  default:
    dc.MoveTo(x-size,y);
    dc.LineTo(x+size,y);
    dc.MoveTo(x,y-size);
    dc.LineTo(x,y+size);
    break;
  }
  
  dc.SelectObject(oldPen);
  pen.DeleteObject();

  brush.DeleteObject();
}

void CKickEditorDlgBar::paintLegend(CDC& dc)
{
  int fontSize = 14;

  CFont font;
  CFont* oldFont;
  font.CreateFont(
    fontSize,                  // nHeight
    0,                         // nWidth
    0,                         // nEscapement
    0,                         // nOrientation
    FW_NORMAL,                 // nWeight
    FALSE,                     // bItalic
    FALSE,                     // bUnderline
    0,                         // cStrikeOut
    ANSI_CHARSET,              // nCharSet
    OUT_DEFAULT_PRECIS,        // nOutPrecision
    CLIP_DEFAULT_PRECIS,       // nClipPrecision
    DEFAULT_QUALITY,           // nQuality
    DEFAULT_PITCH | FF_SWISS,  // nPitchAndFamily
    "Arial");                 // lpszFacename
  
  xOffset = 45, yOffset = currentHeight / 4 * 3 + 15, ySpacing = 13;

  //Rectangle
  CPen blackPen;
  CPen* oldPen;
  blackPen.CreatePen(PS_SOLID, 2 , RGB(0,0,0));
  oldPen = dc.SelectObject(&blackPen);
  dc.Rectangle(xOffset - 7, yOffset - 8, currentWidth - xOffset + 7, currentHeight - xOffset + 7);
  dc.SelectObject(oldPen);

  
	oldFont = dc.SelectObject(&font);
	for (int column = 0; column < columnsCount; column++)
		for (int i = column*actionsPerColumn; i < (column+1)*actionsPerColumn; i++)
		{
      if(i < KickSelectionTable::numberOfActions)
      {
			  KickSelectionTable::ActionID action = (KickSelectionTable::ActionID)i;
			  paintDot(dc, 1, xOffset+(column*columnWidth), (yOffset + (i % actionsPerColumn) * ySpacing), paintStyles[action]);
			  dc.TextOut(xOffset+(column*columnWidth) +20, (yOffset + (i % actionsPerColumn) * ySpacing) - fontSize / 2, KickSelectionTable::getActionName(action));
      }
		}
	dc.SelectObject(oldFont);
 

}

void CKickEditorDlgBar::changeKickPerLegend(CPoint point) {
   
	for (int column = 0; column < columnsCount; column++)
		for (int i = column*actionsPerColumn; i < (column+1)*actionsPerColumn; i++)
		{
			KickSelectionTable::ActionID action = (KickSelectionTable::ActionID)i;
	     	if ((point.x > xOffset+(column*columnWidth)) && (point.x < xOffset+((column+1)*columnWidth)))
				if ((point.y > yOffset + ((i % actionsPerColumn)+1) * ySpacing) && (point.y < yOffset + (i % actionsPerColumn) * ySpacing + 2*ySpacing)) {
					  m_actionCombo.SetCurSel(i);
					  OnSelchangeActionCombo();
			 
			}

		}
}


void CKickEditorDlgBar::OnSave() 
{
  CString defaultPath = File::getGTDir();
  defaultPath += "/Config/";
  defaultPath.Replace('/','\\');
  CString pathName = 
    AfxGetApp()->GetProfileString("KickEditorDlg", "savePath", defaultPath);
  pathName += "*.kst";

  CFileDialog fileDialog(false, ".kst",pathName,
    OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
    , "kick selection tables (*.kst)|*.kst", this);

  if (fileDialog.DoModal()==IDOK)
  {
    CString fileName = fileDialog.GetFileName();
    CString pathAndFileName = fileDialog.GetPathName();
    CString pathName = 
      pathAndFileName.Left(
      pathAndFileName.GetLength() -
      fileName.GetLength()
      );
    AfxGetApp()->WriteProfileString("KickEditorDlg", "savePath", pathName);
  
    OutTextFile fout(pathAndFileName);
    if (fout.exists())
    {
      fout << kickSelectionTable;
    }
    else
    {
      CString message;
      message.Format( "File %s not found.", pathAndFileName);
      AfxMessageBox( message, MB_OK);
    }
  }
}

void CKickEditorDlgBar::OnSelchangeActionCombo() 
{
  currentAction = 
    (KickSelectionTable::ActionID)m_actionCombo.GetCurSel();
  InvalidateRect(NULL, false);
}

void CKickEditorDlgBar::OnSelchangeTableCombo() 
{
  currentTable = 
    (KickSelectionTable::KickSelectionTableID)m_tableCombo.GetCurSel();
  InvalidateRect(NULL, false);
}

void CKickEditorDlgBar::OnOpen() 
{
  CString defaultPath = File::getGTDir();
  defaultPath += "/Config/";
  defaultPath.Replace('/','\\');
  CString pathName = 
    AfxGetApp()->GetProfileString("KickEditorDlg", "openPath", defaultPath);
  pathName += "*.kst";

  CFileDialog fileDialog(true, ".kst",pathName,
    OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
    , "kick selection tables (*.kst)|*.kst", this);

  if (fileDialog.DoModal()==IDOK)
  {
    CString fileName = fileDialog.GetFileName();
    CString pathAndFileName = fileDialog.GetPathName();
    pathName = 
      pathAndFileName.Left(
      pathAndFileName.GetLength() -
      fileName.GetLength()
      );
    AfxGetApp()->WriteProfileString("KickEditorDlg", "openPath", pathName);
  
    InTextFile fin(pathAndFileName);
    if (fin.exists())
    {
      fin >> kickSelectionTable;
    }
    else
    {
      CString message;
      message.Format( "File %s not found.", pathAndFileName);
      AfxMessageBox( message, MB_OK);
    }
  }
}

void CKickEditorDlgBar::copy()
{
  selectedSectorWhenCopied = selectedSector;
  for (int x = 0; x < kickSelectionTable.xRange; x++)
  {
    for (int y = 0; y < kickSelectionTable.yRange; y++)
    {
      for (int s = 0; s < kickSelectionTable.numberOfSectors; s++)
      {
        kickSelectionTableForCopies.action[x][y][s][0] = 
          kickSelectionTable.action[x][y][s][currentTable];
      }
    }
  }
}

void CKickEditorDlgBar::pasteForCurrentSector()
{
  for (int x = 0; x < kickSelectionTable.xRange; x++)
  {
    for (int y = 0; y < kickSelectionTable.yRange; y++)
    {
      kickSelectionTable.action[x][y][selectedSector][currentTable] = 
        kickSelectionTableForCopies.action[x][y][selectedSectorWhenCopied][0];
    }
  }
}

void CKickEditorDlgBar::pasteForAllSectors()
{
  for (int x = 0; x < kickSelectionTable.xRange; x++)
  {
    for (int y = 0; y < kickSelectionTable.yRange; y++)
    {
      for (int s = 0; s < kickSelectionTable.numberOfSectors; s++)
      {
        kickSelectionTable.action[x][y][s][currentTable] = 
          kickSelectionTableForCopies.action[x][y][s][0];
      }
    }
  }
}

void CKickEditorDlgBar::mirrorForCurrentSector()
{
  int x,y;
  for (x = 0; x < kickSelectionTable.xRange; x++)
  {
    for (y = 0; y < kickSelectionTable.yRange; y++)
    {
      kickSelectionTableForCopies.action[x][kickSelectionTable.yRange - 1 - y][selectedSector][1] = 
        kickSelectionTable.action[x][y][selectedSector][currentTable];
    }
  }
  for (x = 0; x < kickSelectionTable.xRange; x++)
  {
    for (y = 0; y < kickSelectionTable.yRange; y++)
    {
      kickSelectionTable.action[x][y][selectedSector][currentTable] = 
        kickSelectionTableForCopies.action[x][y][selectedSector][1];
    }
  }
}

void CKickEditorDlgBar::mirrorForAllSectors()
{
  int x,y,s;
  for (x = 0; x < kickSelectionTable.xRange; x++)
  {
    for (y = 0; y < kickSelectionTable.yRange; y++)
    {
      for (s = 0; s < kickSelectionTable.numberOfSectors; s++)
      {
        kickSelectionTableForCopies.action[x][kickSelectionTable.yRange - 1 - y][s][1] = 
          kickSelectionTable.action[x][y][s][currentTable];
      }
    }
  }
  for (x = 0; x < kickSelectionTable.xRange; x++)
  {
    for (y = 0; y < kickSelectionTable.yRange; y++)
    {
      for (s = 0; s < kickSelectionTable.numberOfSectors; s++)
      {
        kickSelectionTable.action[x][y][s][currentTable] = 
          kickSelectionTableForCopies.action[x][y][s][1];
      }
    }
  }
}

void CKickEditorDlgBar::paintCaseBase(CDC& dc, double scale, KickSelectionTable::ActionID action)
{
  for(int i = 0; i < caseBase[action].GetSize(); i++)
  {
      if(action != KickSelectionTable::nothing)
		  paintKickCase(dc, scale, caseBase[action][i]);
  }
}

void CKickEditorDlgBar::paintKickCase(CDC& dc, double scale, KickCase& kickCase)
{
  int ballRadius = 5;
  
  double angle = atan2(kickCase.ballXA, kickCase.ballYA);
  int distance = (int)sqrt(sqr((double)highlightedY) + sqr((double)highlightedX));
  
  int sector = (int)((normalize(-angle - pi / 2 + pi2 / KickSelectionTable::numberOfSectors / 2) + pi) / pi2 * KickSelectionTable::numberOfSectors);

  int red = 200;
  int green = 200;
  int blue = 200;
  if(sector == selectedSector)
  {
    red = 0; 
    green = 0;
    blue = 255;
  }

  CBrush brush;
  brush.CreateSolidBrush(RGB(red,green,blue));
  CBrush* oldBrush;
  oldBrush = dc.SelectObject(&brush);

  CPen pen;
  pen.CreatePen(PS_SOLID, 2, RGB(red,green,blue));
  CPen* oldPen;
  oldPen = dc.SelectObject(&pen);

  dc.Ellipse(
    (int)(scale*(kickCase.ballXR - ballRadius)), (int)(scale*(kickCase.ballYR - ballRadius)),
    (int)(scale*(kickCase.ballXR + ballRadius)), (int)(scale*(kickCase.ballYR + ballRadius))
    );

  dc.MoveTo((int)(scale*kickCase.ballXR), (int)(scale*kickCase.ballYR));
  dc.LineTo((int)(scale*kickCase.ballXA), (int)(scale*kickCase.ballYA));

  dc.Ellipse(
    (int)(scale*(kickCase.ballXA - ballRadius)), (int)(scale*(kickCase.ballYA - ballRadius)),
    (int)(scale*(kickCase.ballXA + ballRadius)), (int)(scale*(kickCase.ballYA + ballRadius))
    );
  dc.SelectObject(oldBrush);
  dc.SelectObject(oldPen);
  brush.DeleteObject();
  pen.DeleteObject();
}


void CKickEditorDlgBar::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
  m_LeftDownPos = point;

  highlightedX = - point.y + verticalOffset + currentHeight / 4 * 3;
  highlightedY = - point.x + horizontalOffset + currentWidth / 2;
  // is the mouse over the grid for the ball positions?
  int x,y;
  x = (int)(1 + highlightedX / 10.0 / scale) - 1;
  y = (int)(1 + highlightedY / 10.0 / scale + KickSelectionTable::yRange / 2) -1;
  if(x >= 0 && x < KickSelectionTable::xRange &&
     y >= 0 && y < KickSelectionTable::yRange )
  {
    if(oldAction != KickSelectionTable::nothing)
    {
      kickSelectionTable.action[x][y][selectedSector][currentTable] = oldAction;
      if (kickSelectionTable.action[x][y][selectedSector][currentTable] != currentAction)
      {
        floodFill( x, y, oldAction, currentAction);
      /* floodFill created by Matthias
      floodFill(x, y, oldAction,  currentAction, 1);
      */
      }
    }
  }
  
  RedrawWindow(NULL, NULL, RDW_INVALIDATE);

  CDynamicBarDlg::OnLButtonDblClk(nFlags, point);
}


/*
void CKickEditorDlgBar::floodFill(int x, int y, KickSelectionTable::ActionID oldAction, KickSelectionTable::ActionID newAction, int depth)
{
  kickSelectionTable.action[x][y][selectedSector][currentTable] = newAction;
  if (depth > 70) return;
  if(x + 1 < KickSelectionTable::xRange)
  {
    if(kickSelectionTable.action[x + 1][y][selectedSector][currentTable] == oldAction)
      floodFill(x + 1, y, oldAction, newAction, depth + 1);
  }
  if(y + 1 < KickSelectionTable::yRange)
  {
    if(kickSelectionTable.action[x][y + 1][selectedSector][currentTable] == oldAction)
      floodFill(x, y + 1, oldAction, newAction, depth + 1);
  }
  if(x - 1 >= 0)
  {
    if(kickSelectionTable.action[x - 1][y][selectedSector][currentTable] == oldAction)
      floodFill(x - 1, y, oldAction, newAction, depth + 1);
  }
  if(y - 1 >= 0)
  {
    if(kickSelectionTable.action[x][y - 1][selectedSector][currentTable] == oldAction)
      floodFill(x, y - 1, oldAction, newAction, depth + 1);
  }
}

*/

void CKickEditorDlgBar::floodFill(int x, int y, KickSelectionTable::ActionID oldAction, KickSelectionTable::ActionID newAction)
{

    /***************************
    use this function to color all oldActions on 
    the field - not only the connected points 

    for (int i=0; i<20; i++)
    {
      for (int k=0; k<40; k++)
      {
        if(kickSelectionTable.action[i][k][selectedSector][currentTable] == oldAction)
          setFloodFillColor(i, k, newAction);
      }
    }
    ***************************/

   if(kickSelectionTable.action[x][y][selectedSector][currentTable] == oldAction) 
   {
    		setFloodFillColor(x, y, newAction);
        if( x + 1 < KickSelectionTable::xRange)
        {
    		  floodFill(x+1, y, oldAction, newAction);
        }
    		if( x - 1 >= 0)
        {
          floodFill(x-1, y, oldAction, newAction);
        }
    		if( y + 1 < KickSelectionTable::yRange)
        {
          floodFill(x, y+1, oldAction, newAction);
        }
        if( y - 1 >= 0)
        {
          floodFill(x, y-1, oldAction, newAction);
        }
     		if( (x - 1 >= 0) && (y + 1 < KickSelectionTable::yRange))
        {
          floodFill(x-1, y+1, oldAction, newAction);
        }
        if( (x - 1 >= 0) && ( y - 1 >= 0))
        {
          floodFill(x-1, y-1, oldAction, newAction);
        }
        if( (x + 1 < KickSelectionTable::xRange) && ( y + 1 < KickSelectionTable::yRange))
        {
          floodFill(x+1, y+1, oldAction, newAction);
        }
        if( (x + 1 < KickSelectionTable::xRange) && ( y - 1 >= 0))
        {
          floodFill( x+1, y-1, oldAction, newAction);
        }

   }
}

void CKickEditorDlgBar::setFloodFillColor(int x, int y, KickSelectionTable::ActionID newAction)
{
    kickSelectionTable.action[x][y][selectedSector][currentTable] = newAction;
}


void CKickEditorDlgBar::OnSend() 
{
  getQueues().toPhysical.selectedRobot.out.text << kickSelectionTable;
  getQueues().toPhysical.selectedRobot.out.finishMessage(idKickSelectionTable);

  getQueues().toSimulated.selectedRobot.out.text << kickSelectionTable;
  getQueues().toSimulated.selectedRobot.out.finishMessage(idKickSelectionTable);
}

void CKickEditorDlgBar::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
  scale = (m_sizeSlider.GetPos() + 2) / 20.0 ;

  setSliders();

  InvalidateRect(NULL,FALSE);
	
	CDynamicBarDlg::OnHScroll(nSBCode, nPos, pScrollBar);
}

void CKickEditorDlgBar::setSliders()
{
  m_sizeSlider.SetPos((int)(scale * 20.0) - 2);
}


/*
 * Change log :
 * 
 * $Log: KickEditorDlgBar.cpp,v $
 * Revision 1.12  2004/05/07 17:25:39  goetzke
 * improved floodfill
 *
 * Revision 1.11  2004/05/04 18:10:25  juengel
 * Added msh kicks
 *
 * Revision 1.10  2004/04/21 05:50:34  juengel
 * improved
 *
 * Revision 1.9  2004/04/05 17:56:50  loetzsch
 * merged the local German Open CVS of the aibo team humboldt with the tamara CVS
 *
 * Revision 1.2  2004/04/03 02:30:36  jumped
 * added more kicks
 *
 * Revision 1.1.1.1  2004/03/31 11:16:56  loetzsch
 * created ATH repository for german open 2004
 *
 * Revision 1.8  2004/03/19 23:01:51  altmeyer
 * added changeKickPerLegend
 *
 * Revision 1.7  2004/03/19 21:19:05  altmeyer
 * SideHeadKick and SideHeadKick2 changed and renamed to SideHeadKick L and SideHeadKickR
 * a few ATH ERS7-kicks removed
 * bug fix: kick cases shown in action "nothing"
 *
 * Revision 1.6  2004/03/19 11:26:22  juengel
 * recursion depth of floodFill bounded
 *
 * Revision 1.5  2004/03/17 19:55:26  juengel
 * Added OnSend() and floodFill()
 *
 * Revision 1.4  2004/03/16 16:29:45  altmeyer
 * added change log
 *
 * Revision 1.3  2004/03/16 16:29:00  altmeyer
 * added change log
 *
 * Revision 1.2  2004/03/16 16:24:53  altmeyer
 * improved visualization using columns and different colors/symbol for each special action
 *
 * Revision 1.1  2004/03/16 14:00:22  juengel
 * Integrated Improvments from "Gnne"
 * -ATH2004ERS7Behavior
 * -ATHHeadControl
 * -KickSelectionTable
 * -KickEditor
 *
 * Revision 1.14  2004/03/11 17:28:02  juengel
 * Added visualization of KickCases.
 *
 * Revision 1.13  2004/03/11 11:41:19  loetzsch
 * implemented loading of case base
 *
 * Revision 1.12  2004/03/10 22:18:18  juengel
 * Fixed bug in mirror functions.
 *
 * Revision 1.11  2004/03/10 22:02:03  juengel
 * Added copy and mirror methods.
 *
 * Revision 1.10  2004/03/10 21:17:00  juengel
 * Added Buttons to KickEditor.
 *
 * Revision 1.9  2004/03/10 20:56:52  juengel
 * Moved origin of KickEditor.
 *
 * Revision 1.8  2004/03/10 16:37:58  juengel
 * Added combo boxes to KickEditor.
 *
 * Revision 1.7  2004/03/10 14:52:11  juengel
 * Removed scaling bug.
 *
 * Revision 1.6  2004/03/10 14:05:36  loetzsch
 * added saving
 *
 * Revision 1.5  2004/03/10 13:47:49  juengel
 * Added save button to KickEditorDlg.
 *
 * Revision 1.4  2004/03/09 18:44:11  juengel
 * Some improvements.
 *
 * Revision 1.3  2004/03/09 13:41:54  juengel
 * Moved KickSelectionTable to Tools.
 *
 * Revision 1.2  2004/03/09 01:14:44  juengel
 * Added class KickDirections.
 *
 * Revision 1.1  2004/03/08 00:17:04  juengel
 * Added the KickEditorDlgBar.
 *
 */
