/**
* @file LogPlayerToolBar.cpp
* 
* Implementation of the log player toolbar.
*
* @author Matthias Jngel
*/

#define USE_GETMESSAGEIDNAME
#include "StdAfx.h"
#include "resource.h"
#include "LogPlayerToolBar.h"
#include "RobotControlQueues.h"

IMPLEMENT_DYNAMIC( CLogPlayerToolBar, CToolBarEx )


void CLogPlayerToolBar::Init()
{
  static TBBUTTONEX tbButtons[] =
  {
    { {  0, IDC_LOGPLAYER_NEW,     TBSTATE_ENABLED, TBSTYLE_BUTTON , 0, 0 }, true },
    { {  1, IDC_LOGPLAYER_OPEN,     TBSTATE_ENABLED, TBSTYLE_BUTTON , 0, 0 }, true },
    { {  0, 0, 0, TBSTYLE_SEP, 0,0 }, true },
    { {  9, IDC_LOGPLAYER_PLAY_SPEED,     TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0 }, true },
    { {  10, IDC_LOGPLAYER_SMOOTH,     TBSTATE_ENABLED, TBSTYLE_CHECK, 0, 0 }, true },
    { {  0, 0, 0, TBSTYLE_SEP, 0,0 }, true },
    { {  2, IDC_LOGPLAYER_PLAY,     TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0 }, true },
    { {  3, IDC_LOGPLAYER_STOP,     TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0 }, true },
    { {  4, IDC_LOGPLAYER_PAUSE,     TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0 }, true },
    { {  5, IDC_LOGPLAYER_STEP_BACKWARD,     TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0 }, true },
    { {  6, IDC_LOGPLAYER_STEP_FORWARD,     TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0 }, true },
    { {  11, IDC_LOGPLAYER_STEP_REPEAT,     TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0 }, true },
    { {  9, IDC_LOGPLAYER_JUMP_FRAME,     TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0 }, true },
    { {  0, 0, 0, TBSTYLE_SEP, 0,0 }, true },
    { {  7, IDC_LOGPLAYER_RECORD,     TBSTATE_ENABLED, TBSTYLE_CHECK, 0, 0 }, true },
    { {  8, IDC_LOGPLAYER_SAVE,     TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0 }, true },
    
  };
  
  SetBitmaps( IDB_LOGPLAYER, IDB_LOGPLAYER_COLD);
  SetButtons( sizeof( tbButtons ) / sizeof( tbButtons[ 0 ] ), tbButtons);
  
  AddCombo(&speedCombo,IDC_LOGPLAYER_PLAY_SPEED, 60);
  speedCombo.AddString("800%");  
  speedCombo.AddString("400%");
  speedCombo.AddString("200%");
  speedCombo.AddString("100%");
  speedCombo.AddString("50%");
  speedCombo.AddString("25%");
  speedCombo.AddString("10%");
  speedCombo.AddString("5%");
  
  speedCombo.SetCurSel(3);

  AddEdit(&m_wndLogPlayerJumpFrameEdit,IDC_LOGPLAYER_JUMP_FRAME,35,&m_wndLogPlayerJumpFrameEditDummyCombo);
  m_wndLogPlayerJumpFrameEdit.SetWindowText("0");
}

CLogPlayerToolBar::CLogPlayerToolBar()
:logPlayer(getQueues().logPlayer),
fileName("new log file")
{
}

bool CLogPlayerToolBar::handleSelChange(UINT nID)
{
  if (nID == IDC_LOGPLAYER_PLAY_SPEED)
  {
    CString s;
    speedCombo.GetWindowText(s);
    int speed;
    sscanf(s,"%i%%",&speed);
    logPlayer.setPlaySpeed(speed/100.0);
    return true;
  }
  return false;
}

bool CLogPlayerToolBar::handleEditChange(UINT nID)
{
  if (nID==IDC_LOGPLAYER_JUMP_FRAME)
  {
    if (logPlayer.getNumberOfMessages() > 0)
    {
      CString n;
      m_wndLogPlayerJumpFrameEdit.GetWindowText(n);
      
      logPlayer.jumpFrame(atoi(n.GetBuffer(n.GetLength())));
    }
    return true;
  }
  return false;
}

void CLogPlayerToolBar::updateUI(CCmdUI* pCmdUI)
{
  switch(pCmdUI->m_nID )
  {
  case IDC_STATUS_BAR_LOGPLAYER_FILENAME:
    pCmdUI->Enable(TRUE);
    pCmdUI->SetText(fileName);
    return;
  case IDC_STATUS_BAR_LOGPLAYER_MESSAGE:
    {
      CString text;
      text.Format("%i/%i",logPlayer.getCurrentMessageNumber()+1,logPlayer.getNumberOfMessages());
      pCmdUI->Enable(TRUE);
      pCmdUI->SetText(text);
      return;
    }
  case IDC_STATUS_BAR_LOGPLAYER_MESSAGE_ID:
    {
      pCmdUI->Enable(TRUE);
      pCmdUI->SetText(getMessageIDName(logPlayer.getCurrentMessageID()));
      return;
    }
  case IDC_LOGPLAYER_NEW:
  case IDC_LOGPLAYER_OPEN:
  case IDC_LOGPLAYER_PLAY:
  case IDC_LOGPLAYER_STOP:
  case IDC_LOGPLAYER_PAUSE:
  case IDC_LOGPLAYER_STEP_FORWARD:
  case IDC_LOGPLAYER_STEP_BACKWARD:
  case IDC_LOGPLAYER_STEP_REPEAT:
  case IDC_LOGPLAYER_JUMP_FRAME:
  case IDC_LOGPLAYER_RECORD:
  case IDC_LOGPLAYER_SAVE:
  case IDC_LOGPLAYER_SMOOTH:
    {
      switch (logPlayer.getState())
      {
      case LogPlayer::initial:
        {
          switch(pCmdUI->m_nID)
          {
          case IDC_LOGPLAYER_OPEN:
            pCmdUI->Enable(true);
            return;
          case IDC_LOGPLAYER_RECORD:
            pCmdUI->Enable(true);
            pCmdUI->SetCheck(false);
            return;
          default:
            pCmdUI->Enable(false);
            return;
          }
        }
      case LogPlayer::recording:
        {
          switch(pCmdUI->m_nID)
          {
          case IDC_LOGPLAYER_RECORD:
            pCmdUI->Enable(true);
            pCmdUI->SetCheck(true);
            return;
          case IDC_LOGPLAYER_SMOOTH:
            pCmdUI->Enable(false);
            pCmdUI->SetCheck(false);
            return;
          default : 
            pCmdUI->Enable(false);
            return;
          }
        }
      case LogPlayer::paused:
        {
          switch(pCmdUI->m_nID)
          {
          case IDC_LOGPLAYER_NEW:
          case IDC_LOGPLAYER_OPEN:
          case IDC_LOGPLAYER_SAVE:
          case IDC_LOGPLAYER_PLAY:
          case IDC_LOGPLAYER_SMOOTH:
            pCmdUI->Enable(true);
            return;
          case IDC_LOGPLAYER_RECORD:
            pCmdUI->Enable(true);
            pCmdUI->SetCheck(false);
            return;
          case IDC_LOGPLAYER_STEP_FORWARD:
            pCmdUI->Enable(logPlayer.getCurrentMessageNumber() < logPlayer.getNumberOfMessages()-1);
            return;
          case IDC_LOGPLAYER_STOP:
          case IDC_LOGPLAYER_STEP_BACKWARD:
            pCmdUI->Enable(logPlayer.getCurrentMessageNumber() > 0);
            return;
          case IDC_LOGPLAYER_STEP_REPEAT:
            pCmdUI->Enable(logPlayer.getCurrentMessageNumber() >= 0);
            return;
          case IDC_LOGPLAYER_JUMP_FRAME:
            pCmdUI->Enable(logPlayer.getNumberOfMessages() > 0);
            return;
          default:
            pCmdUI->Enable(false);
            return;
          }
        }
      case LogPlayer::playing:
        {
          switch(pCmdUI->m_nID)
          {
          case IDC_LOGPLAYER_STOP:
          case IDC_LOGPLAYER_PAUSE:
          case IDC_LOGPLAYER_SMOOTH:
            pCmdUI->Enable(true);
            return;
          case IDC_LOGPLAYER_RECORD:
            pCmdUI->Enable(false);
            pCmdUI->SetCheck(false);
            return;
          default:
            pCmdUI->Enable(false);
            return;
          }
        }
      default:
        pCmdUI->Enable(false);
        return;
      }
    }
  default: 
    return;
  }
}

bool CLogPlayerToolBar::handleCommand(UINT command)
{
  switch(command)
  {
  case IDC_LOGPLAYER_NEW:
    logPlayer._new();
    fileName="new log file";
    return true;
  case IDC_LOGPLAYER_PLAY:
    logPlayer.play();
    return true;
  case IDC_LOGPLAYER_STOP:
    logPlayer.stop();
    return true;
  case IDC_LOGPLAYER_PAUSE:
    logPlayer.pause();
    updateFrameNumber();
    return true;
  case IDC_LOGPLAYER_STEP_FORWARD:
    logPlayer.stepForward();
    updateFrameNumber();
    return true;
  case IDC_LOGPLAYER_STEP_BACKWARD:
    logPlayer.stepBackward();
    updateFrameNumber();
    return true;
  case IDC_LOGPLAYER_STEP_REPEAT:
    logPlayer.stepRepeat();
    return true;
  case IDC_LOGPLAYER_RECORD:
    logPlayer.record();
    return true;
  case IDC_LOGPLAYER_SMOOTH:
    logPlayer.smooth();
    return true;
  case IDC_LOGPLAYER_SAVE:
    {
      CString defaultPath = File::getGTDir();
      defaultPath += "/Config/";
      defaultPath.Replace('/','\\');
      CString pathName = AfxGetApp()->GetProfileString("LogPlayer", "path", defaultPath);
      pathName += "*.log";
      
      CFileDialog fileDialog(false, ".log",pathName,
        OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
        , "Log file (*.log)|*.log|Enumerated Bitmaps (*.bmp, *.jpg)|*.bmp|Comma Separated Values (*.csv)|*.csv|AIBOVision movie (*.amv)|*.amv||", this);
      if (fileDialog.DoModal()==IDOK)
      {
        pathName = fileDialog.GetPathName().Left(
          fileDialog.GetPathName().GetLength() - fileDialog.GetFileName().GetLength());
        AfxGetApp()->WriteProfileString("LogPlayer", "path", pathName);
        if (strcmp(fileDialog.GetFileExt(),"log")==0)
        {
          if (!logPlayer.save(fileDialog.GetPathName()))
            AfxMessageBox("Could not write " + fileDialog.GetPathName());
          else
            fileName = fileDialog.GetFileName();
        }
        else if (strcmp(fileDialog.GetFileExt(),"amv")==0)
        {
          if (!logPlayer.saveAMV(fileDialog.GetPathName()))
            AfxMessageBox("Could not write " + fileDialog.GetPathName());
        }
        else if (strcmp(fileDialog.GetFileExt(),"csv")==0)
        {
          if (!logPlayer.saveCSV(fileDialog.GetPathName()))
            AfxMessageBox("Could not write " + fileDialog.GetPathName());
        }
        else
        {
          if (!logPlayer.saveImages(fileDialog.GetPathName()))
          {
            AfxMessageBox("Could not write one of the images starting with " + fileDialog.GetPathName());
          }
        }
      }
      return true;
    }
  case IDC_LOGPLAYER_OPEN:
    {
      CString defaultPath = File::getGTDir();
      defaultPath += "/Config/";
      defaultPath.Replace('/','\\');
      
      CString pathName = AfxGetApp()->GetProfileString("LogPlayer", "path", defaultPath);
      pathName += "*.log";
      
      CFileDialog fileDialog(true, ".log",pathName,
        OFN_HIDEREADONLY | OFN_EXPLORER | OFN_ENABLESIZING | OFN_NOCHANGEDIR | OFN_NONETWORKBUTTON
        , "Log file (*.log)|*.log||", this);
      
      if (fileDialog.DoModal()==IDOK)
      {
        pathName = fileDialog.GetPathName().Left(
          fileDialog.GetPathName().GetLength() - fileDialog.GetFileName().GetLength());
        AfxGetApp()->WriteProfileString("LogPlayer", "path", pathName);
        
        if (!logPlayer.open(fileDialog.GetPathName()))
          AfxMessageBox("Could not load " + fileDialog.GetPathName());
        else
          fileName = fileDialog.GetFileName();
        
      }
      return true;
    }
  default:
    return false;
  }
}

bool CLogPlayerToolBar::HasButtonText( UINT nID )
{
  if (nID == IDC_LOGPLAYER_PLAY_SPEED) return true;
  return false;
}

void CLogPlayerToolBar::updateFrameNumber()
{
  char msg[10];
  sprintf(msg, "%d", logPlayer.getCurrentMessageNumber()+1);
  m_wndLogPlayerJumpFrameEdit.SetWindowText(msg);
}

/*
* Change log :
* 
* $Log: LogPlayerToolBar.cpp,v $
* Revision 1.11  2004/05/19 07:58:44  dueffert
* saving to CSV implemented
*
* Revision 1.10  2004/03/27 13:27:23  risler
* enlarged log jump edit
*
* Revision 1.9  2004/03/26 16:33:57  thomas
* added field in logplayer to jump directly to a given frame-number
*
* Revision 1.8  2004/03/24 12:55:43  risler
* added logplayer repeat button
*
* Revision 1.7  2004/03/03 10:12:29  dueffert
* logic bug concerning saving images fixed
*
* Revision 1.6  2004/02/16 12:26:40  nistico
* Added noise reduction functionality for jpeg images in log file player
*
* Revision 1.5  2004/01/20 12:40:08  nistico
* - Added support for ColorTable32K (65K elements in packed format)
* - RobotControl can now convert GT *.log files into AIBOVision (external ColorTable32K calibration tool) *.amv file format
*
* Revision 1.4  2003/12/19 17:37:28  loetzsch
* removed access from the worldstateanalysisdlgbar to the logPlayerToolbar
*
* Revision 1.3  2003/12/19 11:09:28  hamerla
* Logplayer extension  Seek
*
* Revision 1.2  2003/12/06 06:31:20  loetzsch
* no message
*
* Revision 1.1  2003/10/07 10:09:38  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 11:40:12  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.1.1.1  2003/07/02 09:40:25  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.7  2003/05/11 23:46:34  dueffert
* Depend now works with RobotControl too
*
* Revision 1.6  2003/03/24 14:05:42  loetzsch
* moved the logPlayer instance from CRobotControlApp to CRobotControlQueues
*
* Revision 1.5  2003/03/06 18:19:34  dueffert
* reduced amount of warnings drastically by ifdefing global functions
*
* Revision 1.4  2002/12/15 23:34:13  dueffert
* saving images from logfiles added
*
* Revision 1.3  2002/10/04 10:27:33  loetzsch
* Added functionality to adjust the speed of playing log files.
*
* Revision 1.2  2002/10/02 15:52:53  juengel
* The status bar shows the type of the current message in the LogPlayer tool bar.
*
* Revision 1.1  2002/09/10 15:49:05  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.5  2002/08/13 12:55:10  loetzsch
* - Added base class CRobotControlToolBar for tool bars
* - Moved command handling from CRobotControlMainFrame to the tool bars
*
* Revision 1.4  2002/08/08 16:40:30  loetzsch
* added class LogPlayer and redesigned RobotControl's Logplayer GUI
*
* Revision 1.3  2002/07/23 13:43:36  loetzsch
* - new streaming classes
* - removed many #include statements
* - 5 instead of 3 debug queues in RobotControl
* - exchanged StaticQueue with MessageQueue
* - new debug message handling
* - empty constructors in bars / dialogs
* - access to debugkeytables and queues via RobotControlQueues.h and RobotControlDebugKeyTables.h
* - general clean up
*
* Revision 1.2  2002/06/08 09:34:33  Thomas Rfer
* Also save percept collections into log files
*
* Revision 1.1.1.1  2002/05/10 12:40:21  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.11  2002/03/24 18:15:01  loetzsch
* continued change to blocking sensor data receivers
*
* Revision 1.10  2002/02/24 18:28:12  juengel
* Record button improved.
*
* Revision 1.9  2002/02/18 14:08:52  juengel
* Basic record features added.
*
* Revision 1.8  2002/02/07 16:25:49  loetzsch
* made the bar smaller
*
* Revision 1.7  2002/01/25 16:53:21  loetzsch
* OnTimer replaced by OnIdle()
*
* Revision 1.6  2001/12/22 08:25:30  roefer
* Enhanced constructor of class File, removed second one
*
* Revision 1.5  2001/12/20 17:14:54  loetzsch
* Debugging mechanisms continued
*
* Revision 1.4  2001/12/14 16:07:03  juengel
* Laden von Dateien verbessert.
*
* Revision 1.3  2001/12/14 12:26:12  loetzsch
* CameraParameters Toolbar weitergemacht
*
* Revision 1.2  2001/12/12 15:21:09  juengel
* CDebugKeyToolBar in eigenes .cpp und .h File verlagert.
*
* Revision 1.1  2001/12/11 15:14:24  juengel
* CLogPlayerToolBar in eigenes .cpp und .h File verlagert. Abspielfunktionalitt des LogPlayers implementiert.
*
*
*/
